# -*- coding: utf-8 -*-
import os
import ssl
import uuid
import sqlite3
import logging
import traceback
from datetime import datetime
import re
import html
import random
import requests
import pandas as pd
from lxml import etree
from copy import deepcopy
from signxml import XMLSigner, methods
from cryptography.hazmat.backends import default_backend
from cryptography.hazmat.primitives.serialization import pkcs12, Encoding
from OpenSSL.crypto import load_certificate, FILETYPE_PEM
from requests.adapters import HTTPAdapter
from urllib3.poolmanager import PoolManager
from requests_pkcs12 import Pkcs12Adapter
from decimal import Decimal, ROUND_HALF_UP
from lxml.etree import DocumentInvalid
import time
from nfse_db import upsert_status_nf
from pathlib import Path
from functools import wraps
from gerar_pdf_nfse import gerar_pdf_nfse
from nfse_db import init_nfse_schema,upsert_lote,insert_rps_snapshot,log_evento,atualizar_situacao_lote,salvar_notas_da_consulta_lote,alocar_numero_rps

NS_ABRASF = "http://www.abrasf.org.br/nfse.xsd"


def _formatar_data_br(s: str) -> str:
    """
    Converte strings ISO como '2025-08-20T23:30:14', '2025-08-20T23:30:14.123-03:00'
    ou '2025-08-20T23:30:14Z' para '20/08/2025 às 23:30:14'.
    Se não conseguir parsear, retorna o valor original.
    """
    s = (s or "").strip()
    if not s:
        return ""
    # trata 'Z' (UTC) para o fromisoformat
    s = s.replace("Z", "+00:00")
    try:
        dt = datetime.fromisoformat(s)  # aceita com/sem fuso e milissegundos
    except Exception:
        try:
            dt = datetime.strptime(s[:19], "%Y-%m-%dT%H:%M:%S")  # fallback simples
        except Exception:
            return s
    return dt.strftime("%d/%m/%Y às %H:%M:%S")



def _gerar_pdfs_para_protocolo(protocolo: str, conn=None, pasta_saida=None):
    """Gera os PDFs de NFS-e de um protocolo"""
    try:
        fechar_conn = False
        if conn is None:
            conn = sqlite3.connect(CAMINHO_BANCO)
            fechar_conn = True

        conn.row_factory = sqlite3.Row
        cur = conn.cursor()

        cur.execute("""
            SELECT 
                -- Dados da nota
                n.numero_nfse, n.codigo_verificacao, n.data_emissao_nfse, n.competencia,
                n.cnpj_prestador, n.im_prestador, n.valor_servicos, n.valor_iss, n.aliquota,
                n.outras_informacoes,

                -- Dados do RPS
                r.numero_rps, r.serie_rps, r.tipo_rps, r.data_emissao_rps, r.discriminacao,
                r.valor_servicos, r.valor_liquido,

                -- Dados do prestador (sempre sua empresa)
                m.razao_social   AS razao_social_prestador,
                m.nome           AS nome_fantasia_prestador,
                m.logradouro     AS logradouro_prestador,
                m.numero         AS numero_prestador,
                m.bairro         AS bairro_prestador,
                m.cep            AS cep_prestador,
                m.cidade         AS cidade_prestador,
                m.uf             AS uf_prestador,
                m.telefone       AS telefone_prestador,
                m.email          AS email_prestador,
                m.inscricao_municipal AS inscricao_municipal_prestador,

                -- Dados do tomador (convenio)
                c.razao_social   AS razao_social_tomador,
                c.cnpj           AS cnpj_tomador,
                c.inscricao_municipal AS im_tomador,
                c.logradouro     AS logradouro_tomador,
                c.numero         AS numero_tomador,
                c.bairro         AS bairro_tomador,
                c.cep            AS cep_tomador,
                c.cidade         AS cidade_tomador,
                c.uf             AS uf_tomador,
                c.telefone       AS telefone_tomador,
                c.email          AS email_tomador

            FROM nfse_notas n
            JOIN nfse_lotes l ON l.protocolo = n.protocolo
            LEFT JOIN nfse_rps r ON r.protocolo = n.protocolo
            LEFT JOIN medical_laudos m
                ON REPLACE(REPLACE(REPLACE(m.cnpj, '.', ''), '-', ''), '/', '') =
                    REPLACE(REPLACE(REPLACE(n.cnpj_prestador, '.', ''), '-', ''), '/', '')

            LEFT JOIN convenios c ON c.id = l.convenio_id
            WHERE n.protocolo = ?
        """, (protocolo,))
        rows = cur.fetchall()

        if not rows:
            logging.warning(f"Nenhuma nota encontrada para protocolo {protocolo}")
            return

        from gerar_pdf_nfse import gerar_pdf_nfse

        for row in rows:
            dados_nfse = dict(row)
            
            # Garantir compatibilidade            
            dados_nfse["valor_liquido"] = row["valor_liquido"]
            
            # Formata datas para o PDF
            dados_nfse["data_emissao_nfse"] = _formatar_data_br(dados_nfse.get("data_emissao_nfse") or "")
            # Se quiser padronizar a data do RPS no PDF:
            dados_nfse["data_rps"] = _formatar_data_br(dados_nfse.get("data_emissao_rps") or "")

            # 🔹 Quebrando em dicionários separados
            dados_prestador = {
                "razao_social_prestador": row["razao_social_prestador"],
                "nome_fantasia": row["nome_fantasia_prestador"],
                "cnpj": row["cnpj_prestador"],
                "inscricao_municipal": row["inscricao_municipal_prestador"],
                "logradouro": row["logradouro_prestador"],
                "numero": row["numero_prestador"],
                "bairro": row["bairro_prestador"],
                "cep": row["cep_prestador"],
                "cidade": row["cidade_prestador"],
                "uf": row["uf_prestador"],
                "telefone": row["telefone_prestador"],
                "email": row["email_prestador"],
            }




            dados_tomador = {
                "razao_social": dados_nfse["razao_social_tomador"],
                "cnpj": dados_nfse["cnpj_tomador"],
                "inscricao_municipal": dados_nfse["im_tomador"],
                "logradouro": dados_nfse["logradouro_tomador"],
                "numero": dados_nfse["numero_tomador"],
                "bairro": dados_nfse["bairro_tomador"],
                "cep": dados_nfse["cep_tomador"],
                "cidade": dados_nfse["cidade_tomador"],
                "uf": dados_nfse["uf_tomador"],
                "telefone": dados_nfse["telefone_tomador"],
                "email": dados_nfse["email_tomador"],
            }

            # 📝 Pasta de saída
            if pasta_saida is None:
                pasta_saida = os.path.join("notas_emitidas", f"protocolo_{protocolo}")
            os.makedirs(pasta_saida, exist_ok=True)

            pdf_path = os.path.join(pasta_saida, f"NFSe_{dados_nfse['numero_nfse']}.pdf")

            # 🔹 Chamada correta
            gerar_pdf_nfse(
                dados_nfse["numero_nfse"],        # numero_nfse
                dados_nfse["competencia"],        # competencia
                dados_prestador,                  # prestador
                dados_tomador,                    # tomador
                dados_nfse,                       # dados da nota
                pdf_path                          # destino
            )

            logging.info(f"✅ PDF gerado: {pdf_path}")

    except Exception as e:
        logging.error(f"Erro ao gerar PDFs do protocolo {protocolo}: {e}", exc_info=True)
    finally:
        if conn and fechar_conn:
            conn.close()




def _atualizar_outras_info_em_nfse(inner_xml: str, conn: sqlite3.Connection):
    """
    Percorre o outputXML de ConsultarLoteRps e grava <OutrasInformacoes>
    na tabela nfse_notas (coluna já criada) por número da NFSe.
    """
    if not inner_xml:
        return
    try:
        root = etree.fromstring(inner_xml.encode("utf-8"))
    except Exception:
        return

    ns = {"n": NS_ABRASF}

    # ABRASF 1.00 PBH: ListaNfse/CompNfse/Nfse/InfNfse/Numero + OutrasInformacoes
    comps = root.findall(".//n:ListaNfse/n:CompNfse", namespaces=ns)
    if not comps:
        return

    cur = conn.cursor()
    for comp in comps:
        # Número da NFS-e
        numero = comp.findtext(".//n:Nfse/n:InfNfse/n:Numero", namespaces=ns)
        if not numero:
            numero = comp.findtext(".//n:InfNfse/n:Numero", namespaces=ns)
        numero = (numero or "").strip()
        if not numero:
            continue

        # OutrasInformacoes (pode aparecer dentro de InfNfse)
        node = comp.find(".//n:Nfse/n:InfNfse/n:OutrasInformacoes", namespaces=ns)
        if node is None:
            node = comp.find(".//n:InfNfse/n:OutrasInformacoes", namespaces=ns)
        if node is None:
            continue

        texto = "".join(node.itertext()).strip()
        texto = re.sub(r"\s+", " ", texto)  # compacta espaços/linhas
        if not texto:
            continue

        # garante só dígitos no número (caso tenha lixo)
        num_dig = only_digits(numero)
        cur.execute(
            "UPDATE nfse_notas SET outras_informacoes = ? WHERE numero_nfse = ?",
            (texto, num_dig or numero),
        )

    conn.commit()



CAMINHO_BANCO = os.path.join("db", "sistema_financeiro.db")


class NfseError(Exception):
    """Erro geral de NFS-e"""

class SchemaValidationError(NfseError):
    pass

class DigitalSignatureError(NfseError):
    pass

class PBHRetornoError(NfseError):
    def __init__(self, codigo:str, mensagem:str, correcao:str="", dica:str=""):
        self.codigo = codigo
        self.correcao = correcao
        self.dica = dica
        super().__init__(f"PBH [{codigo}]: {mensagem}" + (f" | Correção: {correcao}" if correcao else ""))

PBH_DICAS = {
    "E174": "Erro na assinatura. Garanta ds:Signature como irmã de cada InfRps e irmã de LoteRps; Transforms na ordem: enveloped, depois exclusive c14n; Reference URI apontando para o Id correto.",
    "E181": "Namespace da assinatura incorreto. A tag <Signature> e descendentes devem estar em 'http://www.w3.org/2000/09/xmldsig#'. Use seu forcar_namespace_xmlsig antes de extrair/mover a assinatura.",
}

def extrair_mensagens_pbh(inner_xml:str):
    try:
        root = etree.fromstring(inner_xml.encode("utf-8"))
    except Exception:
        return []
    ns = {"n": NS_DADOS}
    msgs = []
    for m in root.findall(".//n:ListaMensagemRetorno/n:MensagemRetorno", namespaces=ns):
        cod = (m.findtext("n:Codigo", default="", namespaces=ns) or "").strip()
        msg = re.sub(r"\s+", " ", (m.findtext("n:Mensagem", default="", namespaces=ns) or "")).strip()
        cor = re.sub(r"\s+", " ", (m.findtext("n:Correcao", default="", namespaces=ns) or "")).strip()
        dicas = PBH_DICAS.get(cod, "")
        msgs.append({"codigo": cod, "mensagem": msg, "correcao": cor, "dica": dicas})
    return msgs

def raise_if_pbh_error(inner_xml:str):
    msgs = extrair_mensagens_pbh(inner_xml)
    if msgs:
        m = msgs[0]
        raise PBHRetornoError(m["codigo"], m["mensagem"], m["correcao"], m["dica"])

# ========================= CONFIG =========================
FATOR_BASE = Decimal("1.00")   # 100% do total; use Decimal("0.05") p/ 5%
ALIQUOTA = None                # ex.: Decimal("0.02") se quiser calcular ISS

# Namespace ABRASF 1.00 unificado (PBH)
NS_DADOS = "http://www.abrasf.org.br/nfse.xsd"
NS_DS    = "http://www.w3.org/2000/09/xmldsig#"
etree.register_namespace("ds", NS_DS)

# Use o XSD principal (unificado) do pacote nfse_schemas10.zip
BASE_DIR = os.path.dirname(os.path.abspath(__file__))
XSD_PATH = os.path.join(BASE_DIR, "schemas", "pbh", "nfse.xsd")
if not os.path.exists(XSD_PATH):
    raise FileNotFoundError(f"XSD não encontrado em {XSD_PATH}")

# Recomendação PBH: começar com 1 RPS por lote para validar
MAX_RPS_POR_LOTE = 1

HOMOLOG_URL = "https://bhisshomologaws.pbh.gov.br/bhiss-ws/nfse"
PROD_URL    = "https://bhissws.pbh.gov.br/bhiss-ws/nfse"

# carrega o XSD 1x só (se existir no disco)
_SCHEMA = None
if os.path.exists(XSD_PATH):
    _SCHEMA = etree.XMLSchema(etree.parse(XSD_PATH))

# ====================== LOGS ========================
logging.basicConfig(level=logging.INFO, format="%(asctime)s - %(levelname)s - %(message)s")
log_ass = logging.getLogger("assinatura_xml")
log_env = logging.getLogger("enviar_lote_rps")

# ====================== HELPERS =====================

# --- UF a partir do prefixo do código IBGE (2 primeiros dígitos) ---
UF_BY_IBGE_PREFIX = {
    '11':'RO','12':'AC','13':'AM','14':'RR','15':'PA','16':'AP','17':'TO',
    '21':'MA','22':'PI','23':'CE','24':'RN','25':'PB','26':'PE','27':'AL','28':'SE','29':'BA',
    '31':'MG','32':'ES','33':'RJ','35':'SP','41':'PR','42':'SC','43':'RS',
    '50':'MS','51':'MT','52':'GO','53':'DF'
}

def aguardar_status_final_rapido(*, protocolo:str, cnpj_prest:str, im_prest:str,
                                 cert_path:str, cert_pass:str, base_arquivos:str,
                                 url_ws:str, conn:sqlite3.Connection,
                                 status_map:dict, max_wait: int = 25) -> int | None:
    """
    Faz pequenas reconsultas (1s, 2s, 4s, 8s...) até 4 (sucesso) ou 5 (erro),
    por no máximo `max_wait` segundos. Atualiza o banco a cada mudança.
    Retorna a última situação (int) ou None.
    """
    delays = [1, 2, 4, 8, 10]  # total ~25s
    ultimo_sit = None
    for d in delays:
        time.sleep(d)
        sit, inner_sit = consultar_situacao_lote(
            protocolo=protocolo,
            cnpj_prest=cnpj_prest,
            im_prest=im_prest,
            caminho_certificado_pfx=cert_path,
            senha_certificado=cert_pass,
            base_arquivos=base_arquivos,
            url=url_ws,
        )
        msg_db = status_map.get(sit, "")
        out_path_sit = base_arquivos + "_consultar_situacao_outputXML.xml"

        atualizar_situacao_lote(
            conn,
            protocolo=protocolo,
            situacao=sit,
            mensagem=msg_db,
            outputxml_path=out_path_sit,
        )
        log_evento(conn, protocolo, "consulta_situacao",
                   situacao=sit, mensagem=msg_db, payload_path=out_path_sit)

        ultimo_sit = sit
        if sit in (4, 5):
            break
    return ultimo_sit

def uf_from_ibge(cod_ibge: str) -> str:
    cod = only_digits(cod_ibge or "")
    return UF_BY_IBGE_PREFIX.get(cod[:2], "") if len(cod) >= 2 else ""

def _strip_descendant_ids(elem):
    # Na CÓPIA do elemento, remove 'Id' dos descendentes (evita que o signxml aponte para eles)
    for e in elem.iterdescendants():
        if "Id" in e.attrib:
            e.set("IdBackup", e.get("Id"))  # só para não perder a info na cópia (não será usada)
            del e.attrib["Id"]
    return elem

def assert_xsd(elem, fase=""):
    """Valida contra o XSD da PBH e levanta erro detalhado."""
    if _SCHEMA is None:
        return True
    tree = elem if isinstance(elem, etree._ElementTree) else etree.ElementTree(elem)
    try:
        _SCHEMA.assertValid(tree)
    except DocumentInvalid as e:
        erros = "\n".join(f"[linha {err.line}] {err.message}" for err in _SCHEMA.error_log)
        raise SchemaValidationError(f"Falha XSD {('('+fase+')' if fase else '')}:\n{erros}") from e
    return True

def only_digits(s):
    return re.sub(r"\D", "", str(s or ""))

def fmt_item_lista(valor):
    """
    Normaliza ItemListaServico.
    Aceita '17.02' ou '1702'. Se vier bagunçado, tenta reduzir a dígitos.
    """
    s = str(valor or "").strip()
    if re.fullmatch(r"\d{2}\.\d{2}", s) or re.fullmatch(r"\d{4}", s):
        return s
    s = only_digits(s)
    if len(s) == 4:
        return s
    return s

def gerar_numero_lote_15():
    # 14 dígitos de timestamp + 1 dígito aleatório = 15 dígitos no total
    base14 = datetime.now().strftime("%Y%m%d%H%M%S")
    return base14 + str(random.randrange(10))

def validar_estrutura_xml(xml_path:str, xsd_path:str, log_path:str) -> bool:
    try:
        schema = etree.XMLSchema(etree.parse(xsd_path))
        schema.assertValid(etree.parse(xml_path))
        with open(log_path, "w", encoding="utf-8") as f:
            f.write("✅ Validação OK: XML é válido conforme o XSD\n")
        return True
    except Exception as e:
        with open(log_path, "w", encoding="utf-8") as f:
            f.write(f"❌ XML inválido conforme XSD.\n• {str(e)}\n")
        return False

def montar_cabecalho_nfse(versao:str = "1.00") -> str:
    cab = etree.Element(f"{{{NS_DADOS}}}cabecalho", versao=versao, nsmap={None: NS_DADOS})
    etree.SubElement(cab, f"{{{NS_DADOS}}}versaoDados").text = versao
    return etree.tostring(cab, encoding="utf-8", method="xml").decode("utf-8")

def dividir_em_lotes(df, tam:int):
    return [df.iloc[i:i + tam] for i in range(0, len(df), tam)]

# Marcação de duplicados (regras do domínio do usuário)
def tratar_duplicados(df: pd.DataFrame) -> pd.DataFrame:
    df = df.copy()
    df["Duplicado"] = False
    df["NaoDuplicadoOIT"] = False
    if df.empty or "Nome" not in df.columns or "Procedimento" not in df.columns:
        return df
    oit_df = df[df["Procedimento"].str.upper().str.contains("RX - TÓRAX OIT - 2 ASSINATURAS", na=False)].copy()
    restantes_df = df[~df.index.isin(oit_df.index)].copy()
    grp = oit_df.groupby(["Nome", "Procedimento"])
    for (_, _), g in grp:
        medicos = g["Médico"].nunique() if "Médico" in g.columns else g["Medico"].nunique()
        idxs = g.index.tolist()
        if len(idxs) == 2 and medicos == 2:
            oit_df.loc[idxs, "NaoDuplicadoOIT"] = True
        else:
            oit_df.loc[idxs[2:], "Duplicado"] = True
    chaves = ["Nome", "Empresa", "Tipo de Exame", "Procedimento", "Data Exame"]
    if all(c in restantes_df.columns for c in chaves):
        dup = restantes_df.duplicated(subset=chaves, keep="first")
        restantes_df.loc[dup, "Duplicado"] = True
    return pd.concat([oit_df, restantes_df]).sort_index()

# Adiciona elemento somente se houver valor (TAGs opcionais devem ser omitidas se vazias)
def sub_if(parent, tag_local, value):
    if value is None or str(value).strip() == "":
        return None
    el = etree.SubElement(parent, f"{{{NS_DADOS}}}{tag_local}")
    el.text = str(value)
    return el

def retry(exceptions, tries=6, delay=0.15, backoff=1.7):
    def deco(fn):
        @wraps(fn)
        def wrapper(*args, **kwargs):
            _tries, _delay = tries, delay
            while _tries > 1:
                try:
                    return fn(*args, **kwargs)
                except exceptions as e:
                    logging.warning(f"{fn.__name__} falhou: {e} | novo tente em {_delay:.2f}s")
                    time.sleep(_delay)
                    _tries -= 1
                    _delay *= backoff
            return fn(*args, **kwargs)
        return wrapper
    return deco

# ---------- Força namespace e corrige estrutura da assinatura (XMLDSIG) ----------
_SIG_LOCALS = {
    "Signature", "SignedInfo", "CanonicalizationMethod", "SignatureMethod",
    "Reference", "Transforms", "Transform", "DigestMethod", "DigestValue",
    "SignatureValue", "KeyInfo", "X509Data", "X509Certificate"
}

_DEF_ENV = "http://www.w3.org/2000/09/xmldsig#enveloped-signature"
_DEF_C14N = "http://www.w3.org/2001/10/xml-exc-c14n#"

def _fix_signature(sig_el: etree._Element) -> None:
    # CanonicalizationMethod = exclusive c14n (xml-exc-c14n)
    si = sig_el.find(f".//{{{NS_DS}}}SignedInfo")
    if si is None:
        return
    cm = si.find(f".//{{{NS_DS}}}CanonicalizationMethod")
    if cm is not None:
        cm.set("Algorithm", _DEF_C14N)

    # Transforms na ordem: enveloped-signature, depois c14n 2001
    ref = si.find(f".//{{{NS_DS}}}Reference")
    if ref is None:
        return
    tr_parent = ref.find(f".//{{{NS_DS}}}Transforms")
    if tr_parent is None:
        return

    # remove todos e recria na ordem desejada
    for t in list(tr_parent):
        tr_parent.remove(t)
    t1 = etree.SubElement(tr_parent, f"{{{NS_DS}}}Transform")
    t1.set("Algorithm", _DEF_ENV)
    t2 = etree.SubElement(tr_parent, f"{{{NS_DS}}}Transform")
    t2.set("Algorithm", _DEF_C14N)

def forcar_namespace_xmlsig(root):
    for el in root.iter():
        try:
            qn = etree.QName(el)
            local = qn.localname
            uri = qn.namespace
        except Exception:
            continue
        if local in _SIG_LOCALS and uri != NS_DS:
            el.tag = f"{{{NS_DS}}}{local}"
    return root

# ====================== ASSINATURA ===================
def assinar_lote_no_xml(root: etree._Element, cert_path: str, senha: str) -> etree._Element:
    """
    Assina:
      - cada <InfRps> (gera <ds:Signature> irmã de <InfRps> dentro de <Rps>)
      - o <LoteRps> (gera <ds:Signature> como IRMÃ de <LoteRps> no root, conforme modelo PBH)
    """
    # --- certificado/-chave ---
    with open(cert_path, "rb") as f:
        pfx = f.read()
    private_key, certificate, _ = pkcs12.load_key_and_certificates(
        pfx, senha.encode(), backend=default_backend()
    )
    if not private_key or not certificate:
        raise Exception("Falha ao carregar chave/certificado do PFX.")
    pem = certificate.public_bytes(Encoding.PEM)
    openssl_cert = load_certificate(FILETYPE_PEM, pem)

    ns = {"ns": NS_DADOS, "ds": NS_DS}

    # ===================== ASSINA CADA InfRps =====================
    signer_inf = XMLSigner(
        method=methods.enveloped,
        signature_algorithm="rsa-sha1",
        digest_algorithm="sha1",
        c14n_algorithm=_DEF_C14N,  # exclusive c14n
    )

    rps_list = root.xpath(".//ns:Rps", namespaces=ns)
    if not rps_list:
        raise Exception("Não encontrei <Rps> para assinar.")

    for rps in rps_list:
        inf_rps = rps.find(".//ns:InfRps", namespaces=ns)
        if inf_rps is None:
            raise Exception("Elemento <InfRps> não encontrado dentro de <Rps>.")
        inf_id = inf_rps.get("Id")
        if not inf_id:
            raise Exception("<InfRps> sem atributo 'Id'.")

        # limpa assinaturas anteriores
        for old in rps.xpath("./ds:Signature", namespaces=ns):
            old.getparent().remove(old)

        # assina uma cópia do InfRps e pega só a <Signature>
        inf_copy = etree.fromstring(etree.tostring(inf_rps))
        signed_inf = signer_inf.sign(
            inf_copy,
            key=private_key,
            cert=[openssl_cert],
            reference_uri=f"#{inf_id}",
        )

        # compat: às vezes o signxml retorna a própria <Signature>
        if isinstance(signed_inf.tag, str) and etree.QName(signed_inf).localname == "Signature" and etree.QName(signed_inf).namespace == NS_DS:
            sig_rps = signed_inf
        else:
            sig_rps = signed_inf.find(f".//{{{NS_DS}}}Signature")

        if sig_rps is None:
            raise Exception(f"Falha ao gerar <Signature> do InfRps '{inf_id}'.")

        ref = sig_rps.find(f".//{{{NS_DS}}}Reference")
        if ref is None or ref.get("URI") != f"#{inf_id}":
            raise Exception("Assinatura do RPS com URI incorreta.")

        # injeta a assinatura como irmã logo após o InfRps
        rps.insert(rps.index(inf_rps) + 1, sig_rps)

    # ===================== ASSINA O LOTE =====================
    lote_rps = root.find(".//ns:LoteRps", namespaces=ns)
    if lote_rps is None:
        raise Exception("<LoteRps> não encontrado.")
    lote_id = lote_rps.get("Id") or f"lote{uuid.uuid4().hex[:8]}"
    lote_rps.set("Id", lote_id)

    # remove assinaturas antigas
    for old in root.xpath("./ds:Signature", namespaces=ns):
        old.getparent().remove(old)
    for old in lote_rps.xpath("./ds:Signature", namespaces=ns):
        lote_rps.remove(old)

    signer_lote = XMLSigner(
        method=methods.enveloped,  # <<< enveloped
        signature_algorithm="rsa-sha1",
        digest_algorithm="sha1",
        c14n_algorithm=_DEF_C14N,  # exclusive c14n
    )

    # assina uma CÓPIA do LoteRps, substitui o original pela versão assinada
    parent = root
    idx = parent.index(lote_rps)
    lote_signed = signer_lote.sign(
        etree.fromstring(etree.tostring(lote_rps)),
        key=private_key,
        cert=[openssl_cert],
        reference_uri=f"#{lote_id}",
    )
    # normalize ds no elemento assinado antes de extrair/mover a assinatura
    forcar_namespace_xmlsig(lote_signed)

    parent.remove(lote_rps)
    parent.insert(idx, lote_signed)

    # extrai a assinatura criada dentro do Lote
    lote_sig = lote_signed.find(f"./{{{NS_DS}}}Signature")
    if lote_sig is None:
        if isinstance(lote_signed.tag, str) and etree.QName(lote_signed).localname == "Signature" and etree.QName(lote_signed).namespace == NS_DS:
            lote_sig = lote_signed
        else:
            lote_sig = lote_signed.find(f".//{{{NS_DS}}}Signature")

    if lote_sig is None:
        raise Exception("Falha ao gerar <Signature> do <LoteRps>.")

    # move para IRMÃ imediatamente após o <LoteRps> assinado
    lote_signed.remove(lote_sig)
    parent.insert(idx + 1, lote_sig)

    # normaliza namespace ds no documento inteiro e ajusta transforms/c14n
    forcar_namespace_xmlsig(root)
    for sig in root.xpath("//ds:Signature", namespaces=ns):
        _fix_signature(sig)

    return root

# ====================== ENVIO ========================
class SSLAdapter(HTTPAdapter):
    def init_poolmanager(self, *args, **kwargs):
        ctx = ssl.create_default_context()
        ctx.check_hostname = False
        ctx.verify_mode = ssl.CERT_NONE
        self.poolmanager = PoolManager(*args, ssl_context=ctx)

def enviar_lote_rps_com_requests(xml_assinado_path:str, caminho_certificado_pfx:str, senha_certificado:str, url:str=HOMOLOG_URL):
    log_env.info(f"Iniciando envio do lote {xml_assinado_path}")

    with open(xml_assinado_path, "r", encoding="utf-8") as f:
        raw = f.read()
    # remove declaração XML para embed em CDATA
    dados_xml = re.sub(r"<\?xml.*?\?>", "", raw).lstrip()
    cabecalho_xml = montar_cabecalho_nfse("1.00")

    envelope = f"""<?xml version="1.0" encoding="UTF-8"?>
<soapenv:Envelope xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/"
                  xmlns:ws="http://ws.bhiss.pbh.gov.br">
  <soapenv:Header/>
  <soapenv:Body>
    <ws:RecepcionarLoteRpsRequest>
      <nfseCabecMsg><![CDATA[{cabecalho_xml}]]></nfseCabecMsg>
      <nfseDadosMsg><![CDATA[{dados_xml}]]></nfseDadosMsg>
    </ws:RecepcionarLoteRpsRequest>
  </soapenv:Body>
</soapenv:Envelope>"""

    # salva o envelope enviado (auditoria)
    base = os.path.splitext(xml_assinado_path)[0]
    soap_env_path = base + "_soap_request.xml"
    with open(soap_env_path, "w", encoding="utf-8") as f:
        f.write(envelope)

    size = len(envelope.encode("utf-8"))
    if size > 800_000:
        raise Exception(f"Tamanho do envelope {size} bytes acima do limite seguro. Reduza o tamanho do lote.")

    # --- Sessão com PFX e TLS VERIFICADO ---
    s = requests.Session()
    s.mount("https://", Pkcs12Adapter(pkcs12_filename=caminho_certificado_pfx, pkcs12_password=senha_certificado))

    resp = s.post(
        url,
        data=envelope.encode("utf-8"),
        headers={
            "Content-Type": "text/xml; charset=UTF-8",
            "SOAPAction": "http://ws.bhiss.pbh.gov.br/RecepcionarLoteRps",
        },
        verify=True,      # <<< pare de ignorar TLS
        timeout=(15, 90), # conexao, leitura
    )

    # salva SOAP de resposta
    soap_resp_path = base + "_soap_response.xml"
    with open(soap_resp_path, "wb") as f:
        f.write(resp.content)

    if resp.status_code != 200:
        raise NfseError(f"Erro no envio HTTP {resp.status_code}: {resp.text[:800]}")

    # extrai <outputXML>
    soap = etree.fromstring(resp.content)
    fault = soap.find(".//{http://schemas.xmlsoap.org/soap/envelope/}Fault")
    if fault is not None:
        fs = (fault.findtext("faultstring") or "").strip()
        fd = (fault.findtext("detail") or "").strip()
        raise NfseError(f"SOAP Fault: {fs or 'Falha desconhecida'} | {fd}")

    out = soap.find(".//{*}outputXML")
    if out is None or (out.text or "").strip() == "":
        raise Exception("Não encontrei <outputXML> na resposta SOAP.")

    inner = html.unescape(out.text or "")

    # salva o outputXML “limpo”
    output_xml_path = base + "_outputXML.xml"
    with open(output_xml_path, "w", encoding="utf-8") as f:
        f.write(inner)

    # detecta erros da PBH
    raise_if_pbh_error(inner)

    # tenta extrair protocolo, se houver
    m = re.search(r"<Protocolo>(.*?)</Protocolo>", inner)
    if m:
        protocolo = m.group(1).strip()
        log_env.info(f"✅ Protocolo recebido: {protocolo}")
    else:
        log_env.info("⚠️ Sem <Protocolo> no outputXML (pode ser lote rejeitado ou retorno assíncrono).")

    return etree.fromstring(inner.encode("utf-8"))

def consultar_e_atualizar_lote_por_convenio(id_convenio: int, competencia: str, url_envio: str = HOMOLOG_URL):
    """
    Consulta a SITUAÇÃO do lote do convênio/competência. 
    Se estiver '4 - Processado', consulta o LOTE (retorna NFS-e),
    grava nfse_notas e atualiza as tabelas envolvidas.
    Retorna dict com situacao, protocolo e numeros_nf (se houver).
    """
    status_map = {
        1: "1 - Não recebido",
        2: "2 - Não processado",
        3: "3 - Processando",
        4: "Emitida",                 # texto que a sua UI entende
        5: "5 - Processado com erro",
    }

    with sqlite3.connect(CAMINHO_BANCO) as conn:
        cur = conn.cursor()

        # -- 0) Configs do prestador (certificado, fallback de CNPJ/IM)
        cur.execute("""
            SELECT certificado_path, senha_certificado, cnpj, inscricao_municipal
            FROM medical_laudos
            LIMIT 1
        """)
        row_prest = cur.fetchone()
        if not row_prest:
            raise ValueError("Configuração do prestador não encontrada (tabela medical_laudos).")
        certificado_path, senha_certificado, cnpj_fallback, im_fallback = row_prest

        # -- 1) Último lote do convênio/competência
        cur.execute("""
            SELECT *
            FROM nfse_lotes
            WHERE convenio_id = ? AND competencia = ?
            ORDER BY id DESC LIMIT 1
        """, (id_convenio, competencia))
        row = cur.fetchone()
        if not row:
            raise ValueError("Nenhum lote encontrado para este convênio/competência.")

        cols = [c[0] for c in cur.description]
        lote = dict(zip(cols, row))

        protocolo   = (lote.get("protocolo") or "").strip()
        if not protocolo:
            raise ValueError("Lote encontrado, porém sem protocolo.")

        convenio_nome = (lote.get("convenio_nome") or "").strip()  # ← vamos usar para gravar status

        # Pode vir do lote ou cair no fallback (medical_laudos)
        cnpj_prest  = (lote.get("cnpj_prestador") or cnpj_fallback or "").strip()
        im_prest    = (lote.get("im_prestador")  or im_fallback   or "").strip()

        xml_lote    = (lote.get("xml_lote_path") or "").strip()
        base_arquiv = os.path.splitext(xml_lote)[0] if xml_lote else os.path.join("notas_emitidas", f"lote_{protocolo}")
        os.makedirs(os.path.dirname(base_arquiv), exist_ok=True)
        
        # Prefixo dos arquivos de retorno (já vem do xml_lote_path salvo no envio)
        base_retornos = base_arquiv

        # Sobe da pasta "arquivos de retorno" para a pasta do convênio/competência e cria "Nota Fiscal"
        nota_fiscal_dir = Path(base_retornos).parent.parent / "Nota Fiscal"
        nota_fiscal_dir.mkdir(parents=True, exist_ok=True)

        # Prefixo para salvar as NFS-e finais (mesmo nome-base do retorno, porém na pasta "Nota Fiscal")
        base_nf = str(nota_fiscal_dir / Path(base_retornos).name)


        # -- 2) Consultar SITUAÇÃO do lote
        situacao, inner_sit = consultar_situacao_lote(
            protocolo=protocolo,
            cnpj_prest=cnpj_prest,
            im_prest=im_prest,
            caminho_certificado_pfx=certificado_path,
            senha_certificado=senha_certificado,
            base_arquivos=base_retornos,
            url=url_envio,
        )

        # Salva (se veio xml) e atualiza situação no banco
        out_path_sit = base_retornos + "_consultar_situacao_outputXML.xml"
        try:
            if inner_sit:
                with open(out_path_sit, "wb") as f:
                    f.write(inner_sit if isinstance(inner_sit, (bytes, bytearray)) else str(inner_sit).encode("utf-8"))
        except Exception:
            out_path_sit = None

        atualizar_situacao_lote(
            conn,
            protocolo=protocolo,
            situacao=int(situacao or 0),
            mensagem=status_map.get(int(situacao or 0), ""),
            outputxml_path=out_path_sit or "",
        )
        log_evento(
            conn, protocolo, "consulta_situacao",
            situacao=int(situacao or 0), mensagem=status_map.get(int(situacao or 0), ""),
            payload_path=out_path_sit or ""
        )

        numeros_nf = []

        # -- 3) Se processado (4), consulta o LOTE (NFSe) e grava
        if int(situacao or 0) == 4:
            xml_inner = consultar_lote_rps(
                protocolo=protocolo,
                cnpj_prest=cnpj_prest,
                im_prest=im_prest,
                caminho_certificado_pfx=certificado_path,
                senha_certificado=senha_certificado,
                base_arquivos=base_retornos,
                url=url_envio,
            )

            salvar_notas_da_consulta_lote(
                conn=conn,
                protocolo=protocolo,
                numero_lote=lote.get("numero_lote") or "",
                cnpj_prest=cnpj_prest,
                im_prest=im_prest,
                competencia=competencia,      # competência do serviço
                base_files=base_nf,
                xml_inner=xml_inner
            )
            
            # <<< NOVO: grava <OutrasInformacoes> nas notas emitidas deste lote
            try:
                _atualizar_outras_info_em_nfse(xml_inner, conn)
            except Exception as _e_:
                logging.warning(f"Falha ao atualizar OutrasInformacoes (consultar_e_atualizar_lote_por_convenio): {_e_}")
                
                
            _gerar_pdfs_para_protocolo(
                conn=conn,
                protocolo=protocolo,
                pasta_saida=str(nota_fiscal_dir)
            )



            
            # Recupera os números gravados para exibir/gravar no status
            cur.execute(
                "SELECT numero_nfse FROM nfse_notas WHERE protocolo = ? ORDER BY id",
                (protocolo,)
            )
            numeros_nf = [r[0] for r in cur.fetchall()]

        # -- 4) Persistir status em notas_emitidas_status (sempre)
        numero_nf = (numeros_nf[0] if numeros_nf else None)

        competencia_nfse = None
        if int(situacao or 0) == 4 and numero_nf:
            try:
                d = cur.execute("""
                    SELECT data_emissao_nfse FROM nfse_notas
                    WHERE numero_nfse = ? ORDER BY id DESC LIMIT 1
                """, (str(numero_nf),)).fetchone()
                if d and d[0]:
                    s = str(d[0])  # YYYY-MM-DD...
                    if len(s) >= 7 and "-" in s:
                        competencia_nfse = f"{s[5:7]}/{s[0:4]}"
            except Exception:
                pass

        try:
            upsert_status_nf(
                convenio=convenio_nome,
                competencia_servico=competencia,
                status=status_map.get(int(situacao or 0), "⏳ Aguardando"),
                numero_nfse=numero_nf,
                competencia_nfse=competencia_nfse
            )
        except Exception:
            pass

        return {
            "situacao": int(situacao or 0),
            "situacao_txt": status_map.get(int(situacao or 0), "—"),
            "protocolo": protocolo,
            "numeros_nf": numeros_nf,
        }


def consultar_situacao_lote(protocolo:str, cnpj_prest:str, im_prest:str, caminho_certificado_pfx:str, senha_certificado:str, base_arquivos:str, url:str=HOMOLOG_URL):
    cabecalho_xml = montar_cabecalho_nfse("1.00")
    dados = f"""<ConsultarSituacaoLoteRpsEnvio xmlns="{NS_DADOS}">
  <Prestador>
    <Cnpj>{only_digits(cnpj_prest)}</Cnpj>
    <InscricaoMunicipal>{only_digits(im_prest)}</InscricaoMunicipal>
  </Prestador>
  <Protocolo>{protocolo}</Protocolo>
</ConsultarSituacaoLoteRpsEnvio>"""

    envelope = f"""<?xml version="1.0" encoding="UTF-8"?>
<soapenv:Envelope xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/"
                  xmlns:ws="http://ws.bhiss.pbh.gov.br">
  <soapenv:Header/>
  <soapenv:Body>
    <ws:ConsultarSituacaoLoteRpsRequest>
      <nfseCabecMsg><![CDATA[{cabecalho_xml}]]></nfseCabecMsg>
      <nfseDadosMsg><![CDATA[{dados}]]></nfseDadosMsg>
    </ws:ConsultarSituacaoLoteRpsRequest>
  </soapenv:Body>
</soapenv:Envelope>"""

    req_path = f"{base_arquivos}_consultar_situacao_request.xml"
    resp_path = f"{base_arquivos}_consultar_situacao_response.xml"
    out_path  = f"{base_arquivos}_consultar_situacao_outputXML.xml"
    with open(req_path, "w", encoding="utf-8") as f: f.write(envelope)

    s = requests.Session()
    s.mount("https://", Pkcs12Adapter(pkcs12_filename=caminho_certificado_pfx, pkcs12_password=senha_certificado))
    r = s.post(url, data=envelope.encode("utf-8"),
               headers={"Content-Type":"text/xml; charset=UTF-8","SOAPAction":"http://ws.bhiss.pbh.gov.br/ConsultarSituacaoLoteRps"},
               verify=True, timeout=(15,90))
    if r.status_code != 200:
        raise NfseError(f"Erro HTTP {r.status_code} na consulta: {r.text[:800]}")

    with open(resp_path, "wb") as f: f.write(r.content)
    soap = etree.fromstring(r.content)
    out = soap.find(".//{*}outputXML")
    if out is None or not (out.text or "").strip():
        raise Exception("Sem <outputXML> em ConsultarSituacaoLoteRps.")
    inner = html.unescape(out.text or "")
    with open(out_path, "w", encoding="utf-8") as f:
        f.write(inner)

    # Se vier situação numérica, use-a; se vier ListaMensagemRetorno, trate como erro (5)
    m = re.search(r"<Situacao>(\d+)</Situacao>", inner)
    if m:
        return int(m.group(1)), inner

    msgs = extrair_mensagens_pbh(inner)
    if msgs:
        return 5, inner

    return None, inner

def consultar_lote_rps(protocolo:str, cnpj_prest:str, im_prest:str, caminho_certificado_pfx:str, senha_certificado:str, base_arquivos:str, url:str=HOMOLOG_URL):
    cabecalho_xml = montar_cabecalho_nfse("1.00")
    dados = f"""<ConsultarLoteRpsEnvio xmlns="{NS_DADOS}">
  <Prestador>
    <Cnpj>{only_digits(cnpj_prest)}</Cnpj>
    <InscricaoMunicipal>{only_digits(im_prest)}</InscricaoMunicipal>
  </Prestador>
  <Protocolo>{protocolo}</Protocolo>
</ConsultarLoteRpsEnvio>"""

    envelope = f"""<?xml version="1.0" encoding="UTF-8"?>
<soapenv:Envelope xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/"
                  xmlns:ws="http://ws.bhiss.pbh.gov.br">
  <soapenv:Header/>
  <soapenv:Body>
    <ws:ConsultarLoteRpsRequest>
      <nfseCabecMsg><![CDATA[{cabecalho_xml}]]></nfseCabecMsg>
      <nfseDadosMsg><![CDATA[{dados}]]></nfseDadosMsg>
    </ws:ConsultarLoteRpsRequest>
  </soapenv:Body>
</soapenv:Envelope>"""

    req_path = f"{base_arquivos}_consultar_lote_request.xml"
    resp_path = f"{base_arquivos}_consultar_lote_response.xml"
    out_path  = f"{base_arquivos}_consultar_lote_outputXML.xml"
    with open(req_path, "w", encoding="utf-8") as f: f.write(envelope)

    s = requests.Session()
    s.mount("https://", Pkcs12Adapter(pkcs12_filename=caminho_certificado_pfx, pkcs12_password=senha_certificado))
    r = s.post(url, data=envelope.encode("utf-8"),
               headers={"Content-Type":"text/xml; charset=UTF-8","SOAPAction":"http://ws.bhiss.pbh.gov.br/ConsultarLoteRps"},
               verify=True, timeout=(15,90))
    if r.status_code != 200:
        raise NfseError(f"Erro HTTP {r.status_code} na consulta: {r.text[:800]}")

    with open(resp_path, "wb") as f: f.write(r.content)
    soap = etree.fromstring(r.content)
    out = soap.find(".//{*}outputXML")
    if out is None or not (out.text or "").strip():
        raise Exception("Sem <outputXML> em ConsultarLoteRps.")
    inner = html.unescape(out.text or "")
    with open(out_path, "w", encoding="utf-8") as f: f.write(inner)
    raise_if_pbh_error(inner)

    # Retorna XML interno já decodificado (deve conter <ListaNfse><CompNfse>…)
    return inner

@retry((sqlite3.OperationalError,), tries=8, delay=0.1, backoff=1.6)
def _alocar_numero_rps_com_retry(conn, cnpj, serie):
    return alocar_numero_rps(conn, cnpj, serie)

# ====================== PRINCIPAL ====================
def gerar_xml_nfse_por_convenio(id_convenio:int, competencia:str, caminho_saida:str="nota.xml", enviar:bool=True, url_envio:str=HOMOLOG_URL):
    try:
        # 1) Carrega dados
        with sqlite3.connect(CAMINHO_BANCO) as conn:
            cursor = conn.cursor()

            cursor.execute("SELECT * FROM medical_laudos LIMIT 1")
            r = cursor.fetchone()
            if not r:
                raise ValueError("Dados da prestadora não encontrados")
            cols = [c[0] for c in cursor.description]
            prestador = dict(zip(cols, r))

            cursor.execute("SELECT * FROM convenios WHERE id = ?", (id_convenio,))
            r2 = cursor.fetchone()
            if not r2:
                raise ValueError("Convênio não encontrado")
            cols2 = [c[0] for c in cursor.description]
            tomador = dict(zip(cols2, r2))
            nome_convenio = tomador.get("nome")
            
            # --- pastas por convênio/competência ---
            safe_conv = re.sub(r"[^0-9A-Za-z_-]+", "_", (nome_convenio or "").strip())
            safe_comp = (competencia or "").replace("/", "-")

            lote_dir = Path("notas_emitidas") / f"{safe_conv}_{safe_comp}"
            nota_fiscal_dir = lote_dir / "Nota Fiscal"
            retornos_dir = lote_dir / "arquivos de retorno"

            nota_fiscal_dir.mkdir(parents=True, exist_ok=True)
            retornos_dir.mkdir(parents=True, exist_ok=True)


            df = pd.read_sql_query(
                "SELECT * FROM registros_financeiros WHERE Convenio = ? AND Competência = ?",
                conn, params=(nome_convenio, competencia)
            )

        if df.empty:
            raise ValueError("Nenhum registro financeiro encontrado")

        # 2) Normaliza e filtra
        df.columns = (df.columns
                      .str.normalize("NFKD").str.encode("ascii", errors="ignore")
                      .str.decode("utf-8").str.strip())
        df.rename(columns={
            "Medico": "Médico",
            "Valor Convenio": "Valor_Convenio",
            "Data Exame": "Data_Exame",
            "Tipo de Exame": "Tipo_Exame",
        }, inplace=True)
        df = tratar_duplicados(df)
        df = df[~df["Duplicado"]].reset_index(drop=True)

        df["Valor_Convenio"] = pd.to_numeric(df["Valor_Convenio"], errors="coerce").fillna(0)
        qtd_registros = int(len(df))
        total_convenio = float(df["Valor_Convenio"].sum())

        df_consolidado = pd.DataFrame([{
            "Valor_Convenio": total_convenio,
            "QtdRegistros": qtd_registros,
        }])

        lotes = [df_consolidado]  # UM lote com 1 RPS
        resultados = []

        for _, df_lote in enumerate(lotes, start=1):
            # 3) Monta XML (ABRASF 1.00, namespace único)
            NSMAP = {None: NS_DADOS, "ds": NS_DS}
            root = etree.Element(f"{{{NS_DADOS}}}EnviarLoteRpsEnvio", nsmap=NSMAP)

            id_lote = f"lote{uuid.uuid4().hex[:8]}"
            lote_rps = etree.SubElement(root, f"{{{NS_DADOS}}}LoteRps", Id=id_lote, versao="1.00")

            # NumeroLote único (15 dígitos)
            numero_lote = gerar_numero_lote_15()
            etree.SubElement(lote_rps, f"{{{NS_DADOS}}}NumeroLote").text = numero_lote

            cnpj_prest = only_digits(prestador.get("cnpj"))
            im_prest   = only_digits(prestador.get("inscricao_municipal"))

            etree.SubElement(lote_rps, f"{{{NS_DADOS}}}Cnpj").text = cnpj_prest
            sub_if(lote_rps, "InscricaoMunicipal", im_prest if im_prest else None)

            etree.SubElement(lote_rps, f"{{{NS_DADOS}}}QuantidadeRps").text = str(len(df_lote))
            lista_rps = etree.SubElement(lote_rps, f"{{{NS_DADOS}}}ListaRps")

            _ultimo_base_dec = None
            _ultima_discriminacao = ""
            infrps_id = None

            # === Campos fixos do tomador ===
            cod_ibge_tom = only_digits(tomador.get("codigo_municipio_ibge"))
            uf_tom = (tomador.get("uf") or "").strip().upper() or uf_from_ibge(cod_ibge_tom)
            cep_tom = only_digits(tomador.get("cep"))
            im_tom = only_digits(tomador.get("inscricao_municipal")) if cod_ibge_tom == "3106200" else ""

            _ultimo_numero_rps_seq = None

            for _, row in df_lote.iterrows():
                rps = etree.SubElement(lista_rps, f"{{{NS_DADOS}}}Rps")
                infrps_id = f"R{uuid.uuid4().hex[:8]}"
                inf_rps = etree.SubElement(rps, f"{{{NS_DADOS}}}InfRps", Id=infrps_id)

                ident = etree.SubElement(inf_rps, f"{{{NS_DADOS}}}IdentificacaoRps")
                # próximo número sequencial do RPS
                with sqlite3.connect(CAMINHO_BANCO, timeout=10) as conn_seq:
                    init_nfse_schema(conn_seq)
                    numero_rps_seq = _alocar_numero_rps_com_retry(conn_seq, cnpj_prest, "SN")
                    _ultimo_numero_rps_seq = numero_rps_seq

                etree.SubElement(ident, f"{{{NS_DADOS}}}Numero").text = str(numero_rps_seq)
                etree.SubElement(ident, f"{{{NS_DADOS}}}Serie").text  = "SN"
                etree.SubElement(ident, f"{{{NS_DADOS}}}Tipo").text   = "1"

                etree.SubElement(inf_rps, f"{{{NS_DADOS}}}DataEmissao").text = datetime.now().strftime("%Y-%m-%dT%H:%M:%S")
                etree.SubElement(inf_rps, f"{{{NS_DADOS}}}NaturezaOperacao").text = "1"
                etree.SubElement(inf_rps, f"{{{NS_DADOS}}}RegimeEspecialTributacao").text = "6"
                etree.SubElement(inf_rps, f"{{{NS_DADOS}}}OptanteSimplesNacional").text = "1"
                etree.SubElement(inf_rps, f"{{{NS_DADOS}}}IncentivadorCultural").text = "2"
                etree.SubElement(inf_rps, f"{{{NS_DADOS}}}Status").text = "1"

                # -------- SERVIÇO --------
                servico = etree.SubElement(inf_rps, f"{{{NS_DADOS}}}Servico")
                valores = etree.SubElement(servico, f"{{{NS_DADOS}}}Valores")

                total_convenio_dec = Decimal(str(row["Valor_Convenio"]))
                base_dec = (total_convenio_dec * FATOR_BASE).quantize(Decimal("0.01"), ROUND_HALF_UP)
                etree.SubElement(valores, f"{{{NS_DADOS}}}ValorServicos").text    = format(base_dec, ".2f")
                etree.SubElement(valores, f"{{{NS_DADOS}}}IssRetido").text        = "2"
                etree.SubElement(valores, f"{{{NS_DADOS}}}BaseCalculo").text      = format(base_dec, ".2f")
                etree.SubElement(valores, f"{{{NS_DADOS}}}ValorLiquidoNfse").text = format(base_dec, ".2f")

                if ALIQUOTA is not None:
                    valor_iss = (base_dec * ALIQUOTA).quantize(Decimal("0.01"), ROUND_HALF_UP)
                    etree.SubElement(valores, f"{{{NS_DADOS}}}Aliquota").text = format(ALIQUOTA, ".4f")
                    etree.SubElement(valores, f"{{{NS_DADOS}}}ValorIss").text = format(valor_iss, ".2f")

                etree.SubElement(servico, f"{{{NS_DADOS}}}ItemListaServico").text = fmt_item_lista("17.02")
                etree.SubElement(servico, f"{{{NS_DADOS}}}CodigoTributacaoMunicipio").text = "170200688"
                qtd = int(row.get("QtdRegistros", 1))
                discr = f"Elaboração de laudos de exames complementares – competência {competencia} – {qtd} registro(s) consolidados"
                etree.SubElement(servico, f"{{{NS_DADOS}}}Discriminacao").text = discr
                cod_mun_prest = only_digits(prestador.get("codigo_municipio_ibge")) or "3106200"
                etree.SubElement(servico, f"{{{NS_DADOS}}}CodigoMunicipio").text = cod_mun_prest

                _ultimo_base_dec = base_dec
                _ultima_discriminacao = discr

                # -------- PRESTADOR --------
                prest = etree.SubElement(inf_rps, f"{{{NS_DADOS}}}Prestador")
                etree.SubElement(prest, f"{{{NS_DADOS}}}Cnpj").text = cnpj_prest
                sub_if(prest, "InscricaoMunicipal", im_prest if im_prest else None)

                # -------- TOMADOR --------
                tom = etree.SubElement(inf_rps, f"{{{NS_DADOS}}}Tomador")
                ident_t = etree.SubElement(tom, f"{{{NS_DADOS}}}IdentificacaoTomador")
                cpfcnpj = etree.SubElement(ident_t, f"{{{NS_DADOS}}}CpfCnpj")

                cnpj_tom = only_digits(tomador.get("cnpj"))
                sub_if(cpfcnpj, "Cnpj", cnpj_tom if cnpj_tom else None)

                if cod_ibge_tom == "3106200" and im_tom:
                    sub_if(ident_t, "InscricaoMunicipal", im_tom)

                sub_if(tom, "RazaoSocial", tomador.get("nome", ""))

                end = etree.SubElement(tom, f"{{{NS_DADOS}}}Endereco")
                sub_if(end, "Endereco", tomador.get("logradouro", ""))
                sub_if(end, "Numero", tomador.get("numero", ""))
                sub_if(end, "Complemento", tomador.get("complemento", ""))
                sub_if(end, "Bairro", tomador.get("bairro", ""))

                if len(cod_ibge_tom) == 7:
                    sub_if(end, "CodigoMunicipio", cod_ibge_tom)

                sub_if(end, "Uf", uf_tom if uf_tom else None)
                sub_if(end, "Cep", cep_tom if len(cep_tom) == 8 else None)

            # 4) Assina
            root_ass = assinar_lote_no_xml(root, prestador.get("certificado_path"), prestador.get("senha_certificado"))

            ns = {"ns": NS_DADOS, "ds": NS_DS}
            xml_ok = etree.ElementTree(root_ass)
            sig_rps  = xml_ok.xpath("//ds:Signature[ancestor::ns:Rps]", namespaces=ns)
            sig_lote = xml_ok.xpath("/ns:EnviarLoteRpsEnvio/ds:Signature", namespaces=ns)

            if len(sig_rps) != len(df_lote):
                raise Exception(f"Esperava {len(df_lote)} assinaturas de RPS, achei {len(sig_rps)}.")
            if len(sig_lote) != 1:
                raise Exception("Assinatura do Lote ausente ou duplicada.")

            # Referências
            alvo_lote_ok = False
            for ref in xml_ok.xpath("//ds:Reference", namespaces=ns):
                uri = ref.get("URI", "")
                if not uri or not uri.startswith("#"):
                    raise Exception("Reference sem URI #id.")
                target_id = uri[1:]
                hits = xml_ok.xpath(f"//*[@Id='{target_id}']")
                if len(hits) != 1:
                    raise Exception(f"URI {uri} não resolve para 1 elemento.")
                if target_id == id_lote:
                    alvo_lote_ok = True
            if not alvo_lote_ok:
                raise Exception("A assinatura do lote não referencia o Id do LoteRps.")

            # 5) Salva e valida
            base_name = f"NF_{safe_conv}_{safe_comp}_lote_{numero_lote}"

            xml_path_lote = retornos_dir / f"{base_name}.xml"
            with open(xml_path_lote, "wb") as f:
                etree.ElementTree(root_ass).write(f, encoding="utf-8", xml_declaration=True, pretty_print=False)

            # prefixos (onde ficarão os “retornos”)
            base_prefix = str(retornos_dir / base_name)
            # prefixo da pasta onde a(s) NFSe(s) final(is) deve(m) ser salva(s)
            base_nf_prefix = str(nota_fiscal_dir / base_name)

            log_path = f"{base_prefix}_validacao.log"


            if os.path.exists(XSD_PATH):
                ok = validar_estrutura_xml(str(xml_path_lote), XSD_PATH, log_path)
                if not ok:
                    raise Exception(f"XML inválido perante XSD ({xml_path_lote}). Veja o log: {log_path}")
            else:
                with open(log_path, "w", encoding="utf-8") as f:
                    f.write("⚠️ Validação NÃO executada.\n")
                    f.write(f"Schema XSD não encontrado em:\n{os.path.abspath(XSD_PATH)}\n")

            # 6) Metadados comuns para persistência
            ambiente = "HML" if url_envio == HOMOLOG_URL else "PRD"
            xml_lote_path  = str(xml_path_lote)
            soap_req_path  = f"{base_prefix}_soap_request.xml"  if enviar else ""
            soap_resp_path = f"{base_prefix}_soap_response.xml" if enviar else ""
            outputxml_path = f"{base_prefix}_outputXML.xml"     if enviar else ""


            cnpj_tom = only_digits(tomador.get("cnpj"))
            cod_ibge_tom = only_digits(tomador.get("codigo_municipio_ibge"))
            im_tom = only_digits(tomador.get("inscricao_municipal")) if cod_ibge_tom == "3106200" else ""

            valor_servicos = format(_ultimo_base_dec, ".2f")
            base_calculo   = format(_ultimo_base_dec, ".2f")
            valor_liquido  = format(_ultimo_base_dec, ".2f")
            aliquota_txt   = format(ALIQUOTA, ".4f") if ALIQUOTA is not None else ""
            valor_iss_txt  = format((_ultimo_base_dec * ALIQUOTA).quantize(Decimal("0.01"), ROUND_HALF_UP), ".2f") if ALIQUOTA is not None else ""

            # 7) Envio (opcional)
            protocolo = ""
            if enviar:
                resposta = enviar_lote_rps_com_requests(
                    xml_assinado_path=str(xml_path_lote),  # <--- aqui
                    caminho_certificado_pfx=prestador.get("certificado_path"),
                    senha_certificado=prestador.get("senha_certificado"),
                    url=url_envio,
                )
                ns = {"n": NS_DADOS}
                protocolo = resposta.findtext(".//n:Protocolo", namespaces=ns) or ""
                resultados.append(protocolo or numero_lote)
            else:
                resultados.append(numero_lote)

            # 8) Persistência (sempre)
            with sqlite3.connect(CAMINHO_BANCO) as conn:
                init_nfse_schema(conn)

                upsert_lote(conn, {
                    "protocolo": protocolo,
                    "numero_lote": numero_lote,
                    "id_lote_xml": id_lote,
                    "cnpj_prestador": cnpj_prest,
                    "im_prestador": im_prest,
                    "competencia": competencia,
                    "convenio_id": tomador.get("id"),
                    "convenio_nome": tomador.get("nome"),
                    "qtd_rps": 1,
                    "data_envio": datetime.now().strftime("%Y-%m-%d %H:%M:%S"),
                    "ambiente": ambiente,
                    "url_ws": url_envio,
                    "xml_lote_path": xml_lote_path,
                    "soap_req_path": soap_req_path,
                    "soap_resp_path": soap_resp_path,
                    "outputxml_path": outputxml_path,
                    "situacao": None,
                    "data_ultima_consulta": None,
                    "mensagem_retorno": None,
                    "boleto_pago": 0,
                })

                conn.execute("""
                    UPDATE nfse_lotes SET boleto_pago = 0
                    WHERE protocolo = ? AND competencia = ? AND convenio_id = ?
                """, (protocolo, competencia, tomador.get("id")))

                # status inicial (só se enviou)
                if enviar:
                    try:
                        upsert_status_nf(
                            convenio=nome_convenio,
                            competencia_servico=competencia,
                            status="2 - Não processado"
                        )
                    except Exception:
                        pass

                # snapshot do RPS (sempre)
                insert_rps_snapshot(conn, protocolo, {
                    "numero_rps": _ultimo_numero_rps_seq, "serie_rps": "SN", "tipo_rps": "1",
                    "id_infrps": infrps_id,
                    "data_emissao_rps": datetime.now().strftime("%Y-%m-%d %H:%M:%S"),
                    "item_lista_servico": "17.02",
                    "codigo_trib_municipio": "170200688",
                    "discriminacao": _ultima_discriminacao,
                    "codigo_municipio_prestacao": "3106200",
                    "cnpj_tomador": cnpj_tom, "im_tomador": im_tom, "razao_tomador": tomador.get("nome"),
                    "endereco": tomador.get("logradouro"), "numero": tomador.get("numero"),
                    "complemento": tomador.get("complemento"), "bairro": tomador.get("bairro"),
                    "codigo_municipio_ibge": cod_ibge_tom, "uf": uf_tom,
                    "cep": only_digits(tomador.get("cep")),
                    "valor_servicos": valor_servicos, "base_calculo": base_calculo, "valor_liquido": valor_liquido,
                    "iss_retido": "2", "aliquota": aliquota_txt, "valor_iss": valor_iss_txt,
                    "qtd_registros_consolidados": qtd_registros,
                    "valor_total_convenio": f"{total_convenio:.2f}",
                    "fator_base_utilizado": str(FATOR_BASE),
                })

                # log
                if enviar and protocolo:
                    log_evento(conn, protocolo, "envio_lote",
                               mensagem=f"Lote {numero_lote} enviado. Protocolo {protocolo}.",
                               payload_path=outputxml_path)
                else:
                    log_evento(conn, protocolo or numero_lote, "gerar_lote",
                               mensagem=f"Lote {numero_lote} gerado (sem envio).",
                               payload_path=xml_lote_path)

                # 9) Consultas só se houve envio e protocolo
                if enviar and protocolo:
                    status_map = {
                        1: "1 - Não recebido",
                        2: "2 - Não processado",
                        3: "3 - Processando",
                        4: "Emitida",
                        5: "5 - Processado com erro",
                    }
                    base_retornos = base_prefix + "_consultar_situacao_outputXML.xml"


                    # consulta situação
                    sit, inner_sit = consultar_situacao_lote(
                        protocolo=protocolo,
                        cnpj_prest=cnpj_prest,
                        im_prest=im_prest,
                        caminho_certificado_pfx=prestador.get("certificado_path"),
                        senha_certificado=prestador.get("senha_certificado"),
                        base_arquivos=base_retornos,   # <--- aqui
                        url=url_envio,
                    )


                    msg_db = status_map.get(sit, "")
                    if sit is None and "<ListaMensagemRetorno" in inner_sit:
                        code = re.search(r"<Codigo>(.*?)</Codigo>", inner_sit)
                        msg  = re.search(r"<Mensagem>(.*?)</Mensagem>", inner_sit, re.DOTALL)
                        code_txt = code.group(1).strip() if code else "?"
                        msg_txt  = re.sub(r"\s+", " ", msg.group(1)).strip() if msg else "Erro no retorno"
                        msg_db   = f"PBH [{code_txt}]: {msg_txt}"
                        sit      = 5

                    out_path_sit = base_retornos + "_consultar_situacao_outputXML.xml"
                    atualizar_situacao_lote(conn, protocolo=protocolo, situacao=sit, mensagem=msg_db, outputxml_path=out_path_sit)
                    log_evento(conn, protocolo, "consulta_situacao", situacao=sit, mensagem=msg_db, payload_path=out_path_sit)

                    try:
                        upsert_status_nf(
                            convenio=nome_convenio,
                            competencia_servico=competencia,
                            status=status_map.get(int(sit or 0), "⏳ Aguardando")
                        )
                    except Exception:
                        pass

                    # polling se necessário
                    if sit in (1, 2, 3, None):
                        sit_polled = aguardar_status_final_rapido(
                            protocolo=protocolo,
                            cnpj_prest=cnpj_prest,
                            im_prest=im_prest,
                            cert_path=prestador.get("certificado_path"),
                            cert_pass=prestador.get("senha_certificado"),
                            base_arquivos=base_retornos,
                            url_ws=url_envio,
                            conn=conn,
                            status_map=status_map,
                            max_wait=25,
                        )
                        if sit_polled is not None:
                            sit = sit_polled

                    # se processado, consulta lote e grava notas
                    if sit == 4:
                        try:
                            inner_lote = consultar_lote_rps(
                                protocolo=protocolo,
                                cnpj_prest=cnpj_prest,
                                im_prest=im_prest,
                                caminho_certificado_pfx=prestador.get("certificado_path"),
                                senha_certificado=prestador.get("senha_certificado"),
                                base_arquivos=base_retornos,   # <--- mantém em 'arquivos de retorno'
                                url=url_envio,
                            )
                            out_path_cons_lote = base_retornos + "_consultar_lote_outputXML.xml"
                            with sqlite3.connect(CAMINHO_BANCO) as conn2:
                                salvar_notas_da_consulta_lote(
                                    conn=conn2,
                                    protocolo=protocolo,
                                    numero_lote=numero_lote,
                                    cnpj_prest=cnpj_prest,
                                    im_prest=im_prest,
                                    competencia=competencia,
                                    base_files=base_nf_prefix,     # <--- agora salva NFSe em 'Nota Fiscal'
                                    xml_inner=inner_lote
                                )
                                log_evento(conn2, protocolo, "consulta_lote_rps",
                                            mensagem="Consulta de lote processado com sucesso e notas gravadas.",
                                            payload_path=out_path_cons_lote)

                                # <<< NOVO: grava <OutrasInformacoes>
                                try:
                                    _atualizar_outras_info_em_nfse(inner_lote, conn2)
                                except Exception as _e_:
                                    logging.warning(f"Falha ao atualizar OutrasInformacoes: {_e_}")

                                # <<< NOVO: gera os PDFs das notas emitidas
                                _gerar_pdfs_para_protocolo(
                                    conn=conn2,
                                    protocolo=protocolo,
                                    pasta_saida=str(nota_fiscal_dir)
                                )

                        except Exception as e:
                            logging.warning(
                                f"Falha ao consultar lote processado (protocolo {protocolo}): {e}"
                            )



        return {"status": "Gerado", "lotes": resultados, "enviado": enviar}

    except PBHRetornoError as e:
        logging.error(f"{e} | Dica: {e.dica or '-'}")
        try:
            upsert_status_nf(
                convenio=(nome_convenio if 'nome_convenio' in locals() else str(id_convenio)),
                competencia_servico=competencia,
                status="Erro PBH",
                numero_nfse=None,
                competencia_nfse=None
            )
        except Exception:
            pass
        return {"status": "Erro", "codigo": e.codigo, "mensagem": str(e), "dica": e.dica or ""}

    except SchemaValidationError as e:
        logging.exception("XSD inválido")
        return {"status": "Erro", "mensagem": f"XSD inválido: {e}"}

    except DigitalSignatureError as e:
        logging.exception("Falha na assinatura")
        return {"status": "Erro", "mensagem": f"Assinatura: {e}"}

    except requests.exceptions.SSLError:
        logging.exception("Falha SSL/TLS")
        return {"status": "Erro", "mensagem": "Falha SSL/TLS ao comunicar com a PBH. Verifique certificado PFX e cadeia."}

    except requests.exceptions.Timeout:
        logging.exception("Timeout")
        return {"status": "Erro", "mensagem": "Timeout no envio/consulta ao webservice PBH."}

    except Exception as e:
        logging.exception("Erro ao processar NFS-e")
        try:
            upsert_status_nf(
                convenio=(nome_convenio if 'nome_convenio' in locals() else str(id_convenio)),
                competencia_servico=competencia,
                status="Erro",
                numero_nfse=None,
                competencia_nfse=None
            )
        except Exception:
            pass
        return {"status": "Erro", "mensagem": str(e)}

# ====================== CLI =========================
if __name__ == "__main__":
    print(gerar_xml_nfse_por_convenio(
        id_convenio=12,
        competencia="06/2025",
        caminho_saida="notas_emitidas/nota.xml",
        enviar=True,
        url_envio=HOMOLOG_URL
    ))
