# -*- coding: utf-8 -*-
import os
import pdfkit
import base64
import shutil
from html import escape as _esc
import re

import re

def _nao_informado(v: str | None) -> str:
    s = (str(v or "")).strip()
    # se vazio ou "0"/"000..." trata como não informado
    if not s or (s.isdigit() and int(s) == 0):
        return "Não Informado"
    return s


def _only_digits(s):
    return re.sub(r"\D", "", str(s or ""))

def _fmt_cpf_cnpj(v: str) -> str:
    d = _only_digits(v)
    if len(d) == 14:  # CNPJ
        return f"{d[:2]}.{d[2:5]}.{d[5:8]}/{d[8:12]}-{d[12:]}"
    if len(d) == 11:  # CPF
        return f"{d[:3]}.{d[3:6]}.{d[6:9]}-{d[9:]}"
    return v or ""

def _fmt_cep(v: str) -> str:
    d = _only_digits(v)
    return f"{d[:5]}-{d[5:]}" if len(d) == 8 else (v or "")



def _guess_chave_acesso_from_outras(outras: str) -> str:
    """Tenta achar uma 'chave de acesso pública' dentro de OutrasInformacoes."""
    if not outras:
        return ""
    t = " ".join(str(outras).split())

    # Padrões comuns: "Chave de Acesso", "Código de Verificação", query ?chave=...
    pats = [
        r"chave(?:\s*de\s*acesso|\s*p[úu]blica)?[:\s-]*([A-Za-z0-9]{8,64})",
        r"c[oó]digo\s*de\s*verifica[cç][aã]o[:\s-]*([A-Za-z0-9]{6,64})",
        r"[?&](?:chave|cod|codigo|token)=([A-Za-z0-9]{6,64})",
    ]
    for pat in pats:
        m = re.search(pat, t, flags=re.IGNORECASE)
        if m:
            return m.group(1).strip()

    # Fallback: pega a maior sequência alfanumérica com 16+ caracteres
    cands = re.findall(r"[A-Za-z0-9]{16,64}", t)
    return max(cands, key=len) if cands else ""


def substituir_variaveis(template: str, variaveis: dict) -> str:
    """Substitui {{chave}} no HTML por valores de 'variaveis' com escape básico."""
    for chave, valor in variaveis.items():
        if valor is None:
            valor = ""
        # Para números, não precisa escapar; para strings/base64, escapamos
        if isinstance(valor, (int, float)):
            s = str(valor)
        else:
            s = _esc(str(valor))
        template = template.replace(f"{{{{{chave}}}}}", s)
    return template


def carregar_logo_base64(caminho_imagem: str) -> str:
    """Carrega imagem e retorna data URL base64 PNG; retorna '' se não encontrar."""
    if not os.path.exists(caminho_imagem):
        print(f"⚠️ Logo não encontrado: {caminho_imagem}")
        return ""
    with open(caminho_imagem, "rb") as img_file:
        encoded = base64.b64encode(img_file.read()).decode("utf-8")
        return f"data:image/png;base64,{encoded}"


def _format_brl(valor) -> str:
    """Formata número 1234.56 -> '1.234,56' (pt-BR)."""
    try:
        num = float(valor or 0)
    except Exception:
        num = 0.0
    return f"{num:,.2f}".replace(",", "X").replace(".", ",").replace("X", ".")


def _pdfkit_config_windows_first():
    """Usa o caminho padrão do Windows; se não achar, tenta o wkhtmltopdf do PATH."""
    win_path = r"C:\Program Files\wkhtmltopdf\bin\wkhtmltopdf.exe"
    if os.path.exists(win_path):
        return pdfkit.configuration(wkhtmltopdf=win_path)
    found = shutil.which("wkhtmltopdf")
    if found:
        return pdfkit.configuration(wkhtmltopdf=found)
    # Último recurso: None (pdfkit tentará localizar)
    return None


def _sanitize(d: dict | None) -> dict:
    """Converte None -> '' em todas as chaves de um dicionário (defensivo)."""
    d = d or {}
    return {k: ("" if v is None else v) for k, v in d.items()}


def gerar_pdf_nfse(
    numero_nfse,
    competencia,
    dados_prestador,
    dados_tomador,
    dados_nfse,
    caminho_pdf_saida="nfse.pdf",
):
    """
    Gera o PDF da NFS-e a partir do template 'modelo_nfse.html'.

    Parâmetros:
      - numero_nfse (str)
      - competencia (str, ex.: '08/2025')
      - dados_prestador (dict): precisa conter 'razao_social', 'nome' (fantasia), 'cnpj',
                                'inscricao_municipal', 'logradouro', 'numero', 'bairro',
                                'cep', 'cidade', 'uf', 'telefone', 'email'
      - dados_tomador (dict):   idem (cpf/cnpj pode ser 'cnpj' ou 'cpf')
      - dados_nfse (dict):      campos tributários, datas, códigos, etc.
      - caminho_pdf_saida (str): caminho final do PDF
    """
    try:
        # Sanitiza dicionários para evitar 'None' vazando para o HTML
        dados_prestador = _sanitize(dados_prestador)
        dados_tomador = _sanitize(dados_tomador)
        dados_nfse = _sanitize(dados_nfse)

        # Config do wkhtmltopdf (Windows -> PATH -> default)
        config = _pdfkit_config_windows_first()

        # Template HTML
        base_dir = os.path.dirname(__file__)
        caminho_html_base = os.path.join(base_dir, "modelo_nfse.html")
        if not os.path.exists(caminho_html_base):
            raise FileNotFoundError(f"Template HTML não encontrado: {caminho_html_base}")
        print("📄 Template HTML encontrado:", caminho_html_base)

        with open(caminho_html_base, "r", encoding="utf-8") as f:
            template = f.read()

        # Logo em base64
        caminho_logo = os.path.join(base_dir, "imagens", "medical_laudos_logo.png")
        logo_base64 = carregar_logo_base64(caminho_logo)
        if logo_base64:
            template = template.replace(
                'src="imagens/medical_laudos_logo.png"',
                f'src="{logo_base64}" height="60"',
            )

        # Valores formatados
        valor_servico_formatado = _format_brl(dados_nfse.get("valor_servico", 0))

        # OutrasInformacoes -> tentativa de extrair a "chave de acesso"
        outras_info = dados_nfse.get("outras_informacoes", "")
        chave_guess = _guess_chave_acesso_from_outras(outras_info)
        chave_final = (
            dados_nfse.get("chave_acesso")
            or chave_guess
            or dados_nfse.get("codigo_verificacao")
            or "N/A"
        )

        # Variáveis do template
        variaveis = {
            "numero_nfse": numero_nfse,
            "data_emissao_nfse": dados_nfse.get("data_emissao_nfse", ""),
            "competencia": competencia,
            "codigo_verificacao": dados_nfse.get("codigo_verificacao", ""),

            # Prestador
            "razao_social_prestador": dados_prestador.get("razao_social_prestador")
                          or dados_prestador.get("razao_social") or "",
            "nome_fantasia_prestador": dados_prestador.get("nome_fantasia", ""),
            "cnpj_prestador": _fmt_cpf_cnpj(dados_prestador.get("cnpj", "")),
            "inscricao_municipal_prestador": dados_prestador.get("inscricao_municipal")
                                 or dados_nfse.get("im_prestador", ""),
            "endereco_prestador": (
                f"{dados_prestador.get('logradouro') or ''}, "
                f"{dados_prestador.get('numero') or ''}, "
                f"{dados_prestador.get('bairro') or ''}, "
                f"{_fmt_cep(dados_prestador.get('cep')) or ''}, "
                f"{dados_prestador.get('cidade') or ''} - "
                f"{dados_prestador.get('uf') or ''}"
            ),
            "telefone_prestador": dados_prestador.get("telefone", ""),
            "email_prestador": dados_prestador.get("email", ""),

            # Tomador
            "razao_social_tomador": dados_tomador.get("razao_social", ""),
            "cpf_cnpj_tomador": _fmt_cpf_cnpj(dados_tomador.get("cnpj", "") or dados_tomador.get("cpf", "")),
            "inscricao_municipal_tomador": _nao_informado(dados_tomador.get("inscricao_municipal")),
            "endereco_tomador": (
                f"{dados_tomador.get('logradouro') or ''}, "
                f"{dados_tomador.get('numero') or ''}, "
                f"{dados_tomador.get('bairro') or ''}, "
                f"{_fmt_cep(dados_tomador.get('cep')) or ''}, "
                f"{dados_tomador.get('cidade') or ''} - "
                f"{dados_tomador.get('uf') or ''}"
            ),
            "telefone_tomador": dados_tomador.get("telefone", ""),
            "email_tomador": dados_tomador.get("email", ""),

            # Serviço / tributação
            "descricao_servico": dados_nfse.get("discriminacao", ""),
            "subitem_lista": dados_nfse.get(
                "subitem_lista",
                "17.02 / Datilografia, digitacao, estenografia, expediente, "
                "secretaria em geral, resposta audivel, redacao, edicao, "
                "interpretacao, revisao, traducao, apoio e infra-estrutura "
                "administrativa e congeneres.",
            ),
            "codigo_tributacao": dados_nfse.get(
                "codigo_tributacao",
                "1702-0/06-88 / Servicos combinados de escritorio e apoio administrativo",
            ),
            "municipio_prestacao": dados_nfse.get("municipio_prestacao", "Belo Horizonte/MG"),

            # 💰 Valores (formatados aqui)
            "valor_servico": valor_servico_formatado,
            "desconto": _format_brl(dados_nfse.get("desconto", 0)),
            "retencao_federal": _format_brl(dados_nfse.get("retencao_federal", 0)),
            "valor_iss": _format_brl(dados_nfse.get("valor_iss", 0)),
            "valor_liquido": _format_brl(dados_nfse.get("valor_liquido", dados_nfse.get("valor_servico", 0))),
            "deducoes": _format_brl(dados_nfse.get("deducoes", 0)),
            "desconto_incondicionado": _format_brl(dados_nfse.get("desconto_incondicionado", 0)),
            "desconto_condicionado": _format_brl(dados_nfse.get("desconto_condicionado", 0)),
            "base_calculo": _format_brl(dados_nfse.get("base_calculo", dados_nfse.get("valor_servico", 0))),
            "aliquota_iss": str(dados_nfse.get("aliquota_iss", 0)).replace(".", ","),  # exibe 3,00
            "outras_retencoes": _format_brl(dados_nfse.get("outras_retencoes", 0)),

            # 📋 Complementares
            "municipio_incidencia": dados_nfse.get("municipio_incidencia", "3106200 / Belo Horizonte"),
            "regime_tributario": dados_nfse.get("regime_tributario", "ME ou EPP do Simples Nacional"),
            "natureza_operacao": dados_nfse.get("natureza_operacao", "Tributação no município"),
            "incentivo_fiscal": (
                "Sim"
                if str(dados_nfse.get("incentivo_fiscal", "")).strip().lower() in ("sim", "1", "true")
                else "Não"
            ),

            # 📄 RPS
            "numero_rps": dados_nfse.get("numero_rps", ""),
            "serie_rps": dados_nfse.get("serie_rps", "BH"),
            "tipo_rps": dados_nfse.get("tipo_rps", "1"),
            "data_rps": dados_nfse.get("data_rps", dados_nfse.get("data_emissao", "")),

            # 🔑 Autenticidade / consulta (definir UMA vez)
            "outras_informacoes": outras_info,
            "url_consulta": dados_nfse.get("url_consulta", ""),
            "chave_acesso_nfse": chave_final,
        }

        html_final = substituir_variaveis(template, variaveis)

        # Salva HTML de debug ao lado do PDF
        # caminho_pdf_saida = os.path.abspath(caminho_pdf_saida)
        # out_dir = os.path.dirname(caminho_pdf_saida) or "."
        # os.makedirs(out_dir, exist_ok=True)
        # caminho_html_temp = caminho_pdf_saida.replace(".pdf", ".debug.html")
        # with open(caminho_html_temp, "w", encoding="utf-8") as f:
        #     f.write(html_final)
        # print(f"🧪 HTML de debug salvo em: {caminho_html_temp}")

        # Geração do PDF
        print("📄 Gerando PDF em:", caminho_pdf_saida)
        options = {
            "enable-local-file-access": "",
            "quiet": "",
            "encoding": "UTF-8",
            "page-size": "A4",
            "dpi": 300,
            "margin-top": "10mm",
            "margin-right": "10mm",
            "margin-bottom": "12mm",
            "margin-left": "10mm",
        }
        pdfkit.from_string(html_final, caminho_pdf_saida, configuration=config, options=options)

        print(f"✅ PDF gerado com sucesso: {caminho_pdf_saida}")
        return caminho_pdf_saida

    except Exception as e:
        print(f"❌ Erro ao gerar PDF com pdfkit: {e}")
        return None
