# gui/dialog_editar_medico.py

from PyQt5.QtWidgets import (
    QDialog, QVBoxLayout, QLabel, QLineEdit,
    QPushButton, QHBoxLayout, QMessageBox
)
import sqlite3
import os

CAMINHO_BANCO = os.path.join("db", "sistema_financeiro.db")

class DialogEditarMedico(QDialog):
    def __init__(self, nome_medico, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Editar Médico")
        self.setMinimumWidth(400)
        self.nome_antigo = nome_medico

        self.layout = QVBoxLayout(self)

        # Campos
        self.input_nome = QLineEdit()
        self.input_crm = QLineEdit()
        self.input_rqe = QLineEdit()
        self.input_especialidade = QLineEdit()

        self.layout.addWidget(QLabel("Nome:"))
        self.layout.addWidget(self.input_nome)
        self.layout.addWidget(QLabel("CRM:"))
        self.layout.addWidget(self.input_crm)
        self.layout.addWidget(QLabel("RQE:"))
        self.layout.addWidget(self.input_rqe)
        self.layout.addWidget(QLabel("Especialidade:"))
        self.layout.addWidget(self.input_especialidade)

        # Botões
        botoes = QHBoxLayout()
        btn_salvar = QPushButton("💾 Salvar")
        btn_cancelar = QPushButton("Cancelar")
        botoes.addWidget(btn_salvar)
        botoes.addWidget(btn_cancelar)
        self.layout.addLayout(botoes)

        btn_salvar.clicked.connect(self.salvar)
        btn_cancelar.clicked.connect(self.reject)

        self.carregar_dados()

    def carregar_dados(self):
        try:
            conn = sqlite3.connect(CAMINHO_BANCO)
            cursor = conn.cursor()
            cursor.execute("""
                SELECT nome, crm, rqe, especialidade
                FROM medicos WHERE nome = ?
            """, (self.nome_antigo,))
            resultado = cursor.fetchone()
            conn.close()

            if resultado:
                self.input_nome.setText(str(resultado[0]))
                self.input_crm.setText(str(resultado[1]))
                self.input_rqe.setText(str(resultado[2]))
                self.input_especialidade.setText(str(resultado[3]))
        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao carregar dados: {e}")

    def salvar(self):
        novo_nome = self.input_nome.text().strip()
        crm = self.input_crm.text().strip()
        rqe = self.input_rqe.text().strip()
        especialidade = self.input_especialidade.text().strip()

        if not novo_nome:
            QMessageBox.warning(self, "Atenção", "O campo 'Nome' é obrigatório.")
            return

        try:
            conn = sqlite3.connect(CAMINHO_BANCO)
            cursor = conn.cursor()

            # Verifica se já existe outro médico com o mesmo nome
            cursor.execute("""
                SELECT COUNT(*) FROM medicos WHERE nome = ? AND nome != ?
            """, (novo_nome, self.nome_antigo))
            if cursor.fetchone()[0] > 0:
                QMessageBox.warning(self, "Atenção", f"Já existe um médico com o nome '{novo_nome}'.")
                return

            # Confirma se deseja atualizar o nome em outras tabelas
            atualizar_relacionados = False
            if novo_nome != self.nome_antigo:
                resposta = QMessageBox.question(
                    self, "Confirmar",
                    f"Você alterou o nome do médico.\nDeseja atualizar esse nome também nas tabelas relacionadas (procedimentos e registros)?",
                    QMessageBox.Yes | QMessageBox.No
                )
                atualizar_relacionados = (resposta == QMessageBox.Yes)

            # Atualiza o médico
            cursor.execute("""
                UPDATE medicos
                SET nome = ?, crm = ?, rqe = ?, especialidade = ?
                WHERE nome = ?
            """, (novo_nome, crm, rqe, especialidade, self.nome_antigo))

            # Atualiza nome em tabelas relacionadas, se necessário
            if atualizar_relacionados:
                cursor.execute("""
                    UPDATE medico_procedimentos
                    SET nome_medico = ?
                    WHERE nome_medico = ?
                """, (novo_nome, self.nome_antigo))

                cursor.execute("""
                    UPDATE registros_financeiros
                    SET medico = ?
                    WHERE medico = ?
                """, (novo_nome, self.nome_antigo))

            conn.commit()
            conn.close()

            QMessageBox.information(self, "Sucesso", "Dados do médico atualizados com sucesso.")
            self.accept()
        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao salvar: {e}")
