from PyQt5.QtWidgets import (
    QDialog, QFormLayout, QLabel, QLineEdit, QPushButton,
    QHBoxLayout, QMessageBox, QFileDialog
)
from PyQt5.QtCore import Qt
import requests
import re
from PyQt5.QtGui import QRegularExpressionValidator
from PyQt5.QtCore import QRegularExpression


class DialogEmpresaConfiguracao(QDialog):
    def __init__(self, parent=None, dados_existentes=None):
        super().__init__(parent)
        self.setWindowTitle("Configurações da Empresa Prestadora de Serviço")
        self.setMinimumWidth(500)
        self.setModal(True)

        self.dados_existentes = dados_existentes or {}
        self.campos = {}

        layout = QFormLayout(self)

        campos_labels = [
            ("nome", "Nome Fantasia"),
            ("razao_social", "Razão Social"),
            ("cnpj", "CNPJ"),
            ("cep", "CEP"),
            ("logradouro", "Logradouro"),
            ("numero", "Número"),
            ("complemento", "Complemento"),
            ("bairro", "Bairro"),
            ("cidade", "Cidade"),
            ("uf", "UF"),
            ("codigo_municipio_ibge", "Código Município IBGE (Prestador)"),  # <-- NOVO
            ("email", "E-mail"),
            ("telefone", "Telefone"),
            ("inscricao_estadual", "Inscrição Estadual"),
            ("inscricao_municipal", "Inscrição Municipal"),
            ("discriminacao", "Discriminação do serviço")
        ]

        for chave, label in campos_labels:
            campo = QLineEdit()
            campo.setText(self.dados_existentes.get(chave, ""))
            layout.addRow(QLabel(label + ":"), campo)
            self.campos[chave] = campo
            
        # Config extra para o IBGE do PRESTADOR
        ibge_input = self.campos["codigo_municipio_ibge"]
        ibge_input.setPlaceholderText("Ex.: 3106200 (Belo Horizonte)")
        ibge_input.setMaxLength(7)
        ibge_input.setValidator(QRegularExpressionValidator(QRegularExpression(r"^\d{0,7}$"), self))


        # Certificado
        self.input_certificado = QLineEdit()
        self.input_certificado.setReadOnly(True)
        btn_certificado = QPushButton("Selecionar .pfx")
        btn_certificado.clicked.connect(self.selecionar_certificado)

        self.input_certificado.setText(self.dados_existentes.get("certificado_path", ""))
        layout.addRow(QLabel("Certificado Digital (.pfx):"), self.input_certificado)
        layout.addRow("", btn_certificado)

        # Senha do certificado
        self.input_senha = QLineEdit()
        self.input_senha.setEchoMode(QLineEdit.Password)
        self.input_senha.setText(self.dados_existentes.get("senha_certificado", ""))
        layout.addRow(QLabel("Senha do Certificado:"), self.input_senha)

        # Campo de logo
        self.input_logo = QLineEdit()
        self.input_logo.setReadOnly(True)
        btn_logo = QPushButton("Selecionar Logo")
        btn_logo.clicked.connect(self.selecionar_logo)

        self.input_logo.setText(self.dados_existentes.get("logo_path", ""))
        layout.addRow(QLabel("Logo da Empresa:"), self.input_logo)
        layout.addRow("", btn_logo)

        # Buscar CNPJ
        btn_cnpj = QPushButton("🔍 Buscar dados pelo CNPJ")
        btn_cnpj.clicked.connect(self.buscar_cnpj)
        layout.addRow(btn_cnpj)

        self.campos["cep"].textChanged.connect(self.buscar_cep)

        # Botões
        botoes_layout = QHBoxLayout()
        btn_salvar = QPushButton("Salvar")
        btn_cancelar = QPushButton("Cancelar")
        btn_salvar.clicked.connect(self.validar_e_aceitar)
        btn_cancelar.clicked.connect(self.reject)
        botoes_layout.addWidget(btn_salvar)
        botoes_layout.addWidget(btn_cancelar)
        layout.addRow(botoes_layout)

    def selecionar_certificado(self):
        caminho, _ = QFileDialog.getOpenFileName(self, "Selecionar Certificado Digital (.pfx)", "", "Certificados (*.pfx)")
        if caminho:
            self.input_certificado.setText(caminho)

    def selecionar_logo(self):
        caminho, _ = QFileDialog.getOpenFileName(self, "Selecionar Logo da Empresa", "", "Imagens (*.png *.jpg *.jpeg *.bmp)")
        if caminho:
            self.input_logo.setText(caminho)

    def buscar_cep(self):
        cep = re.sub(r'\D', '', self.campos["cep"].text())
        if len(cep) != 8:
            return

        try:
            resposta = requests.get(f"https://viacep.com.br/ws/{cep}/json/")
            if resposta.status_code == 200:
                dados = resposta.json()
                self.campos["logradouro"].setText(dados.get("logradouro", ""))
                self.campos["bairro"].setText(dados.get("bairro", ""))
                self.campos["cidade"].setText(dados.get("localidade", ""))
                self.campos["uf"].setText(dados.get("uf", ""))
                self.campos["codigo_municipio_ibge"].setText(dados.get("ibge", ""))
        except Exception as e:
            print(f"Erro ao buscar CEP: {e}")

    def buscar_cnpj(self):
        cnpj = re.sub(r'\D', '', self.campos["cnpj"].text())
        if len(cnpj) != 14:
            QMessageBox.warning(self, "CNPJ Inválido", "O CNPJ deve conter 14 dígitos.")
            return

        try:
            resposta = requests.get(f"https://receitaws.com.br/v1/cnpj/{cnpj}")
            if resposta.status_code == 200:
                dados = resposta.json()

                if dados.get("status") == "OK":
                    self.campos["razao_social"].setText(dados.get("nome", ""))
                    self.campos["nome"].setText(dados.get("fantasia", ""))
                    self.campos["cep"].setText(dados.get("cep", "").replace(".", "").replace("-", ""))
                    self.campos["logradouro"].setText(dados.get("logradouro", ""))
                    self.campos["bairro"].setText(dados.get("bairro", ""))
                    self.campos["cidade"].setText(dados.get("municipio", ""))
                    self.campos["uf"].setText(dados.get("uf", ""))
                    self.campos["email"].setText(dados.get("email", ""))
                    self.campos["telefone"].setText(dados.get("telefone", ""))
                    self.campos["inscricao_estadual"].setText(dados.get("ie", ""))
                else:
                    QMessageBox.warning(self, "Erro", "CNPJ não encontrado ou erro na consulta.")
            else:
                QMessageBox.warning(self, "Erro", f"Erro ao consultar CNPJ. Código: {resposta.status_code}")
        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao consultar CNPJ:\n{e}")

    def validar_e_aceitar(self):
        obrigatorios = ["nome", "razao_social", "cnpj", "cep", "logradouro", "numero", "bairro", "cidade", "uf"]
        for campo in obrigatorios:
            if not self.campos[campo].text().strip():
                QMessageBox.warning(self, "Campo obrigatório", f"Preencha o campo: {campo.replace('_', ' ').capitalize()}")
                return

        cnpj = re.sub(r'\D', '', self.campos["cnpj"].text())
        if len(cnpj) != 14:
            QMessageBox.warning(self, "CNPJ Inválido", "O CNPJ deve conter 14 dígitos.")
            return

        self.accept()

    def get_dados(self):
        dados = {chave: campo.text().strip() for chave, campo in self.campos.items()}
        dados["certificado_path"] = self.input_certificado.text().strip()
        dados["senha_certificado"] = self.input_senha.text().strip()
        dados["caminho_logo"] = self.input_logo.text().strip()

        # normaliza IBGE para até 7 dígitos
        dados["codigo_municipio_ibge"] = re.sub(r"\D", "", dados.get("codigo_municipio_ibge", ""))[:7]
        return dados

