from PyQt5.QtWidgets import (
    QDialog, QVBoxLayout, QLabel, QPushButton, QTableWidget,
    QTableWidgetItem, QHBoxLayout, QFileDialog, QMessageBox, QInputDialog
)
from PyQt5.QtCore import Qt
import sqlite3
import os
import pandas as pd

CAMINHO_BANCO = os.path.join("db", "sistema_financeiro.db")

class DialogExibirComissao(QDialog):
    def __init__(self, nome_vendedor, parent=None):
        super().__init__(parent)
        self.setWindowTitle(f"Comissão - {nome_vendedor}")
        self.setMinimumSize(800, 500)
        self.nome_vendedor = nome_vendedor

        # Buscar competências disponíveis ANTES de tudo
        try:
            conn = sqlite3.connect(CAMINHO_BANCO)
            cursor = conn.cursor()
            cursor.execute("SELECT DISTINCT Competência FROM registros_financeiros ORDER BY Competência DESC")
            competencias = [row[0] for row in cursor.fetchall() if row[0]]
            conn.close()

            if not competencias:
                QMessageBox.information(self, "Aviso", "Nenhuma competência encontrada.")
                self.reject()
                return

            competencia_escolhida, ok = QInputDialog.getItem(
                self, "Escolher Competência", "Selecione a competência para calcular comissão:",
                competencias, editable=False
            )
            if not ok or not competencia_escolhida:
                self.reject()
                return

            self.competencia_escolhida = competencia_escolhida

        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao buscar competências:\n{e}")
            self.reject()
            return


        layout = QVBoxLayout(self)

        self.label_info = QLabel("Calculando...")
        layout.addWidget(self.label_info)

        self.tabela = QTableWidget()
        layout.addWidget(self.tabela)

        botoes = QHBoxLayout()
        self.btn_exportar = QPushButton("⬇️ Exportar Excel")
        self.btn_fechar = QPushButton("Fechar")
        botoes.addStretch()
        botoes.addWidget(self.btn_exportar)
        botoes.addWidget(self.btn_fechar)
        layout.addLayout(botoes)

        self.btn_exportar.clicked.connect(self.exportar_excel)
        self.btn_fechar.clicked.connect(self.reject)

        self.carregar_dados()

    def carregar_dados(self):
        try:
            conn = sqlite3.connect(CAMINHO_BANCO)
            cursor = conn.cursor()

            # Obter ID e comissão do vendedor
            cursor.execute("SELECT id, comissao_padrao, cpf FROM vendedores WHERE nome = ?", (self.nome_vendedor,))
            row = cursor.fetchone()
            if not row:
                raise Exception("Vendedor não encontrado.")
            id_vendedor, comissao_padrao, cpf = row

            self.comissao_percentual = comissao_padrao
            self.vendedor_cpf = cpf

            # Buscar convênios vinculados
            cursor.execute("SELECT nome_convenio FROM vendedor_convenio WHERE id_vendedor = ?", (id_vendedor,))
            convenios = [r[0] for r in cursor.fetchall()]

            if not convenios:
                self.label_info.setText("Nenhum convênio vinculado a este vendedor.")
                self.tabela.setRowCount(0)
                return            

            # Buscar registros da competência e convênios vinculados
            marcadores = ','.join(['?'] * len(convenios))
            query = f"""
                SELECT Nome, Convenio, Procedimento, `Valor Convenio`, `Data Exame`
                FROM registros_financeiros
                WHERE Competência = ? AND Convenio IN ({marcadores})
                ORDER BY `Data Exame` DESC
            """
            cursor.execute(query, [self.competencia_escolhida] + convenios)
            registros = cursor.fetchall()
            conn.close()

            # Atualizar label e tabela
            total_registros = len(registros)
            total_valor = 0
            self.tabela.setColumnCount(6)
            self.tabela.setHorizontalHeaderLabels([
                "Nome", "Convênio", "Procedimento", "Valor Convênio", "Data Exame", "Comissão (R$)"
            ])
            self.tabela.setRowCount(total_registros)

            self.df_export = []

            for i, row in enumerate(registros):
                nome, convenio, procedimento, valor_conv_str, data_exame = row

                # Limpar e converter valor
                valor_float = 0.0
                try:
                    if isinstance(valor_conv_str, str):
                        valor_float = float(valor_conv_str.replace("R$", "").replace(".", "").replace(",", "."))
                    else:
                        valor_float = float(valor_conv_str)
                except:
                    pass

                comissao_valor = round(valor_float * comissao_padrao / 100, 2)
                total_valor += comissao_valor

                for j, val in enumerate([nome, convenio, procedimento, f"R$ {valor_float:.2f}", data_exame]):
                    item = QTableWidgetItem(str(val))
                    item.setFlags(Qt.ItemIsSelectable | Qt.ItemIsEnabled)
                    self.tabela.setItem(i, j, item)

                item_comissao = QTableWidgetItem(f"R$ {comissao_valor:.2f}")
                item_comissao.setTextAlignment(Qt.AlignCenter)
                self.tabela.setItem(i, 5, item_comissao)

                self.df_export.append({
                    "Nome": nome,
                    "Convênio": convenio,
                    "Procedimento": procedimento,
                    "Valor Convênio": valor_float,
                    "Data Exame": data_exame,
                    "Comissão (R$)": comissao_valor
                })

            self.label_info.setText(
                f"Vendedor: <b>{self.nome_vendedor}</b> | CPF: {cpf} | Comissão: <b>{comissao_padrao:.2f}%</b><br>"
                f"Competência: <b>{self.competencia_escolhida}</b> | Registros: <b>{total_registros}</b> | "
                f"Total a pagar: <b>R$ {total_valor:.2f}</b>"
            )

        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao carregar comissão:\n{e}")

    def exportar_excel(self):
        if not self.df_export:
            QMessageBox.warning(self, "Aviso", "Nada a exportar.")
            return
        

        df = pd.DataFrame(self.df_export)
        df["Valor Convênio"] = pd.to_numeric(df["Valor Convênio"], errors="coerce").fillna(0.0)


        nome_maiusculo = self.nome_vendedor.upper()
        competencia = self.competencia_escolhida

        caminho, _ = QFileDialog.getSaveFileName(
            self, "Salvar como", f"Comissao_{nome_maiusculo}_{competencia.replace('/', '-')}.xlsx", "Excel Files (*.xlsx)"
        )
        if not caminho:
            return

        try:
            from xlsxwriter import Workbook  # para garantir que o import existe
            writer = pd.ExcelWriter(caminho, engine='xlsxwriter')
            workbook = writer.book
            df.to_excel(writer, index=False, sheet_name='Comissão', startrow=2)
            worksheet = writer.sheets['Comissão']

            azul_cabecalho = '#B7D6F4'
            header_format = workbook.add_format({'bold': True, 'bg_color': '#D9D9D9', 'border': 1, 'align': 'center'})
            center_format = workbook.add_format({'align': 'center', 'valign': 'vcenter'})
            money_format = workbook.add_format({'num_format': '"R$" #,##0.00', 'align': 'center'})
            bold_blue = workbook.add_format({'bold': True, 'font_color': 'black', 'bg_color': azul_cabecalho, 'border': 1, 'align': 'center'})
            bold_right = workbook.add_format({'bold': True, 'align': 'right', 'bg_color': azul_cabecalho, 'border': 1})
            value_right = workbook.add_format({'align': 'right', 'bg_color': azul_cabecalho, 'border': 1})
            money_right = workbook.add_format({'num_format': '"R$" #,##0.00', 'align': 'right', 'bg_color': azul_cabecalho, 'border': 1})
            logo_background = workbook.add_format({'bg_color': azul_cabecalho, 'border': 1})

            total_registros = len(df)
            total_valor = df["Valor Convênio"].sum() if "Valor Convênio" in df.columns else 0
            total_comissao = df["Comissão (R$)"].sum() if "Comissão (R$)" in df.columns else 0

            # Cabeçalho com 4 linhas
            worksheet.merge_range("A1:A2", "", logo_background)
            worksheet.merge_range("B1:C1", f"COMISSÃO - {nome_maiusculo}", bold_blue)
            worksheet.merge_range("B2:C2", f"Competência: {competencia}", bold_blue)
            worksheet.write("D1", "TOTAL", bold_right)
            worksheet.write("E1", total_valor, money_right)
            worksheet.write("D2", "REGISTROS", bold_right)
            worksheet.write("E2", total_registros, value_right)
            worksheet.write("F1", "TOTAL COMISSÃO (R$)", bold_right)            
            worksheet.write("F2", total_comissao, money_right)

            # Inserir logo
            caminho_logo = os.path.join("icones", "medical_laudos_logo.png")
            if os.path.exists(caminho_logo):
                worksheet.insert_image("A1", caminho_logo, {'x_scale': 0.45, 'y_scale': 0.45, 'x_offset': 2, 'y_offset': 2})

            # Cabeçalhos das colunas
            for col_num, col_name in enumerate(df.columns):
                worksheet.write(2, col_num, col_name, header_format)
                largura_max = max([len(str(cell)) for cell in [col_name] + df[col_name].astype(str).tolist()]) + 2
                worksheet.set_column(col_num, col_num, largura_max)

                # Ajustar a largura da coluna F manualmente, pois o cabeçalho adicional é maior que os dados
                worksheet.set_column("F:F", 25)  # ✅ correto


            # Conteúdo da tabela
            for row_num, row_data in df.iterrows():
                for col_num, valor in enumerate(row_data):
                    formato = money_format if "Valor" in df.columns[col_num] else center_format
                    linha_excel = row_num + 3
                    if isinstance(valor, (int, float)):
                        worksheet.write_number(linha_excel, col_num, valor, formato)
                    else:
                        worksheet.write(linha_excel, col_num, str(valor), formato)

            worksheet.autofilter(2, 0, 2 + len(df), len(df.columns) - 1)
            writer.close()
            QMessageBox.information(self, "Sucesso", f"Arquivo exportado com sucesso para:\n{caminho}")

        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao exportar Excel:\n{e}")

