from PyQt5.QtWidgets import (
    QDialog, QVBoxLayout, QFormLayout, QLineEdit,
    QPushButton, QHBoxLayout, QMessageBox
)
from PyQt5.QtCore import Qt
import sqlite3
import os

CAMINHO_BANCO = os.path.join("db", "sistema_financeiro.db")

class DialogNovoVendedor(QDialog):
    def __init__(self, parent=None, dados_existentes=None):
        super().__init__(parent)
        self.setWindowTitle("Editar Vendedor" if dados_existentes else "Novo Vendedor")
        self.setMinimumWidth(400)
        self.dados_existentes = dados_existentes


        layout = QVBoxLayout(self)

        form_layout = QFormLayout()
        self.campo_nome = QLineEdit()
        self.campo_cpf = QLineEdit()
        self.campo_comissao = QLineEdit()

        if self.dados_existentes:
            self.campo_nome.setText(self.dados_existentes["nome"])
            self.campo_cpf.setText(self.dados_existentes["cpf"])
            self.campo_cpf.setDisabled(True)  # Evita editar o CPF (chave de identificação)
            self.campo_comissao.setText(str(self.dados_existentes["comissao"]))


        self.campo_nome.setPlaceholderText("Nome completo do vendedor")
        self.campo_cpf.setPlaceholderText("CPF (somente números)")
        self.campo_comissao.setPlaceholderText("Ex: 5 para 5%")

        form_layout.addRow("Nome:", self.campo_nome)
        form_layout.addRow("CPF:", self.campo_cpf)
        form_layout.addRow("Comissão (%):", self.campo_comissao)

        layout.addLayout(form_layout)

        botoes_layout = QHBoxLayout()
        self.botao_salvar = QPushButton("Salvar")
        self.botao_cancelar = QPushButton("Cancelar")

        self.botao_salvar.clicked.connect(self.salvar_vendedor)
        self.botao_cancelar.clicked.connect(self.reject)

        botoes_layout.addStretch()
        botoes_layout.addWidget(self.botao_salvar)
        botoes_layout.addWidget(self.botao_cancelar)

        layout.addLayout(botoes_layout)

    def salvar_vendedor(self):
        nome = self.campo_nome.text().strip()
        cpf = self.campo_cpf.text().strip()
        comissao_texto = self.campo_comissao.text().strip().replace(",", ".")

        if not nome or not cpf or not comissao_texto:
            QMessageBox.warning(self, "Campos obrigatórios", "Preencha todos os campos.")
            return

        try:
            comissao = float(comissao_texto)
            if comissao < 0 or comissao > 100:
                raise ValueError
        except ValueError:
            QMessageBox.warning(self, "Comissão inválida", "Insira um valor de comissão entre 0 e 100.")
            return

        try:
            conn = sqlite3.connect(CAMINHO_BANCO)
            cursor = conn.cursor()
            cursor.execute("""
                CREATE TABLE IF NOT EXISTS vendedores (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    nome TEXT NOT NULL,
                    cpf TEXT NOT NULL,
                    comissao_padrao REAL NOT NULL
                )
            """)
            if self.dados_existentes:
                cursor.execute("""
                    UPDATE vendedores
                    SET nome = ?, comissao_padrao = ?
                    WHERE cpf = ?
                """, (nome, comissao, cpf))
            else:
                cursor.execute("""
                    INSERT INTO vendedores (nome, cpf, comissao_padrao)
                    VALUES (?, ?, ?)
                """, (nome, cpf, comissao))

            conn.commit()
            conn.close()
            QMessageBox.information(self, "Sucesso", "Vendedor cadastrado com sucesso!")
            self.accept()
        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao salvar vendedor:\n{e}")
