from PyQt5.QtWidgets import (
    QWidget, QVBoxLayout, QPushButton, QHBoxLayout, QTableWidget, QTableWidgetItem,
    QMessageBox, QHeaderView, QLineEdit, QLabel, QDialog
)
from PyQt5.QtCore import Qt
import sqlite3
import os
from gui.dialog_empresa_configuracao import DialogEmpresaConfiguracao

CAMINHO_BANCO = os.path.join("db", "sistema_financeiro.db")

class ModuloConfiguracoesEmpresa(QWidget):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.parent_main = parent
        self.layout = QVBoxLayout(self)

        estilo_botao = """
            QPushButton {
                background-color: #3498db;
                color: white;
                padding: 6px 12px;
                font-weight: bold;
                border-radius: 4px;
            }
            QPushButton:hover {
                background-color: #2980b9;
            }
        """

        titulo = QLabel("Configurações da Empresa Prestadora de Serviço")
        titulo.setAlignment(Qt.AlignCenter)
        titulo.setStyleSheet("font-size: 18px; font-weight: bold; margin-bottom: 10px;")
        self.layout.addWidget(titulo)

        # Botões
        botoes_layout = QHBoxLayout()
        self.botao_cadastrar = QPushButton("➕ Cadastrar Empresa")
        self.botao_editar = QPushButton("✏️ Editar Empresa")
        self.botao_excluir = QPushButton("🗑️ Excluir Empresa")

        for botao in [self.botao_cadastrar, self.botao_editar, self.botao_excluir]:
            botao.setStyleSheet(estilo_botao)

        botoes_layout.addWidget(self.botao_cadastrar)
        botoes_layout.addWidget(self.botao_editar)
        botoes_layout.addWidget(self.botao_excluir)

        # Tabela
        self.tabela = QTableWidget()
        self.tabela.setColumnCount(3)
        self.tabela.setHorizontalHeaderLabels(["Nome Fantasia", "CNPJ", "E-mail"])
        self.tabela.verticalHeader().setVisible(False)
        self.tabela.setSelectionBehavior(QTableWidget.SelectRows)
        self.tabela.setSelectionMode(QTableWidget.SingleSelection)
        self.tabela.setEditTriggers(QTableWidget.NoEditTriggers)
        self.tabela.setAlternatingRowColors(True)
        self.tabela.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        self.tabela.horizontalHeader().setDefaultAlignment(Qt.AlignCenter)

        self.layout.addLayout(botoes_layout)
        self.layout.addWidget(self.tabela)

        self.botao_cadastrar.clicked.connect(self.abrir_modal_novo)
        self.botao_editar.clicked.connect(self.abrir_modal_editar)
        self.botao_excluir.clicked.connect(self.excluir_empresa)

        self.carregar_empresa()

    def carregar_empresa(self):
        self.tabela.setRowCount(0)
        try:
            conn = sqlite3.connect(CAMINHO_BANCO)
            cursor = conn.cursor()
            cursor.execute("SELECT id, nome, cnpj, email FROM medical_laudos")
            empresas = cursor.fetchall()
            conn.close()

            self.tabela.setRowCount(len(empresas))
            for i, (id_emp, nome, cnpj, email) in enumerate(empresas):
                for j, dado in enumerate([nome, cnpj, email]):
                    item = QTableWidgetItem(dado)
                    item.setTextAlignment(Qt.AlignCenter)
                    self.tabela.setItem(i, j, item)
        except Exception as e:
            print(f"[ERRO] ao carregar empresa: {e}")

    def abrir_modal_novo(self):
        dialog = DialogEmpresaConfiguracao(self)
        if dialog.exec_() == dialog.Accepted:
            dados = dialog.get_dados()
            caminho_logo = dados.pop("caminho_logo", None)

            try:
                conn = sqlite3.connect(CAMINHO_BANCO)
                cursor = conn.cursor()

                cursor.execute("SELECT COUNT(*) FROM medical_laudos")
                existe = cursor.fetchone()[0]

                if existe == 0:
                    # 🔁 Copiar imagem
                    if caminho_logo and os.path.exists(caminho_logo):
                        os.makedirs("imagens", exist_ok=True)
                        extensao = os.path.splitext(caminho_logo)[1]
                        destino = os.path.join("imagens", "logo_empresa" + extensao)
                        with open(caminho_logo, 'rb') as origem, open(destino, 'wb') as destino_arq:
                            destino_arq.write(origem.read())
                        dados["logo_path"] = destino

                    campos = ", ".join(dados.keys())
                    valores = ", ".join("?" for _ in dados)
                    cursor.execute(
                        f"INSERT INTO medical_laudos ({campos}) VALUES ({valores})",
                        tuple(dados.values())
                    )
                else:
                    QMessageBox.warning(self, "Aviso", "Já existe um registro da empresa. Use o botão de editar.")
                    conn.close()
                    return

                conn.commit()
                conn.close()
                self.carregar_empresa()
                QMessageBox.information(self, "Sucesso", "Empresa cadastrada com sucesso.")
            except Exception as e:
                QMessageBox.critical(self, "Erro", f"Erro ao salvar empresa:\n{e}")



    def abrir_modal_editar(self):
        try:
            conn = sqlite3.connect(CAMINHO_BANCO)
            cursor = conn.cursor()
            cursor.execute("SELECT * FROM medical_laudos LIMIT 1")
            row = cursor.fetchone()

            if not row:
                QMessageBox.information(self, "Aviso", "Nenhum registro de empresa encontrado.")
                conn.close()
                return

            colunas = [desc[0] for desc in cursor.description]
            dados_existentes = dict(zip(colunas, row))
            conn.close()

            dialog = DialogEmpresaConfiguracao(self, dados_existentes=dados_existentes)
            if dialog.exec_() == dialog.Accepted:
                novos_dados = dialog.get_dados()
                caminho_logo = novos_dados.pop("caminho_logo", None)

                # 🔁 Atualiza imagem, se fornecida
                if caminho_logo and os.path.exists(caminho_logo):
                    os.makedirs("imagens", exist_ok=True)
                    extensao = os.path.splitext(caminho_logo)[1]
                    destino = os.path.join("imagens", "logo_empresa" + extensao)
                    with open(caminho_logo, 'rb') as origem, open(destino, 'wb') as destino_arq:
                        destino_arq.write(origem.read())
                    novos_dados["logo_path"] = destino

                campos_update = ", ".join([f"{chave}=?" for chave in novos_dados.keys()])
                valores = list(novos_dados.values())

                conn = sqlite3.connect(CAMINHO_BANCO)
                cursor = conn.cursor()
                cursor.execute(f"UPDATE medical_laudos SET {campos_update} WHERE id = 1", valores)
                conn.commit()
                conn.close()
                self.carregar_empresa()
                QMessageBox.information(self, "Sucesso", "Empresa atualizada com sucesso.")

        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao editar empresa:\n{e}")



    def excluir_empresa(self):
        row = self.tabela.currentRow()
        if row < 0:
            QMessageBox.warning(self, "Aviso", "Selecione a empresa para excluir.")
            return

        item_nome = self.tabela.item(row, 0)
        if item_nome is None:
            QMessageBox.warning(self, "Aviso", "Linha inválida ou tabela vazia.")
            return

        nome = item_nome.text()

        confirmar = QMessageBox.question(
            self,
            "Confirmar exclusão",
            f"Deseja excluir a empresa '{nome}'?",
            QMessageBox.Yes | QMessageBox.No
        )
        if confirmar == QMessageBox.Yes:
            try:
                conn = sqlite3.connect(CAMINHO_BANCO)
                cursor = conn.cursor()
                cursor.execute("DELETE FROM medical_laudos WHERE id = 1")
                conn.commit()
                conn.close()
                self.carregar_empresa()
                QMessageBox.information(self, "Sucesso", f"Empresa '{nome}' excluída.")
            except Exception as e:
                QMessageBox.critical(self, "Erro", f"Erro ao excluir empresa:\n{e}")


