# gui/modulo13_analises.py
# -*- coding: utf-8 -*-
from __future__ import annotations
import os
import sqlite3
from typing import Optional, Tuple

import numpy as np
import pandas as pd

from PyQt5.QtWidgets import (
    QWidget, QVBoxLayout, QHBoxLayout, QLabel, QComboBox, QPushButton,
    QTabWidget, QFileDialog, QFrame, QMessageBox,
    QTableWidget, QTableWidgetItem, QHeaderView, QAbstractItemView,
    QSpinBox, QCheckBox, QGridLayout
)
from PyQt5.QtCore import Qt

# Matplotlib embutido no PyQt5 (compatível com várias versões)
try:
    # Matplotlib 3.5+ (Qt5/Qt6 unificado)
    from matplotlib.backends.backend_qtagg import FigureCanvasQTAgg as FigureCanvas
except Exception:
    # Fallback para versões mais antigas
    from matplotlib.backends.backend_qt5agg import FigureCanvasQTAgg as FigureCanvas
from matplotlib.figure import Figure
import matplotlib.dates as mdates
from matplotlib.ticker import FuncFormatter

import re
import unicodedata

CAMINHO_BANCO = os.path.join("db", "sistema_financeiro.db")

# Mapeamento UF -> Região (macro)
BR_UF_TO_REGIAO = {
    "AC":"Norte","AP":"Norte","AM":"Norte","PA":"Norte","RO":"Norte","RR":"Norte","TO":"Norte",
    "AL":"Nordeste","BA":"Nordeste","CE":"Nordeste","MA":"Nordeste","PB":"Nordeste","PE":"Nordeste","PI":"Nordeste","RN":"Nordeste","SE":"Nordeste",
    "DF":"Centro-Oeste","GO":"Centro-Oeste","MT":"Centro-Oeste","MS":"Centro-Oeste",
    "ES":"Sudeste","MG":"Sudeste","RJ":"Sudeste","SP":"Sudeste",
    "PR":"Sul","RS":"Sul","SC":"Sul",
}

# ----------------- helpers -----------------

def fmt_money(v: float) -> str:
    try:
        s = f"{float(v):,.2f}".replace(",", "X").replace(".", ",").replace("X", ".")
        return f"R$ {s}"
    except Exception:
        return "R$ 0,00"


def _competencia_key(mm_aaaa: str) -> Tuple[int, int]:
    """Transforma 'MM/AAAA' em (AAAA, MM) para ordenação e comparações."""
    if not isinstance(mm_aaaa, str) or "/" not in mm_aaaa:
        return (0, 0)
    mm, aa = mm_aaaa.split("/")
    try:
        return (int(aa), int(mm))
    except Exception:
        return (0, 0)


# ----------------- Canvas de gráfico reutilizável -----------------
class Chart(QWidget):
    def __init__(self, titulo: str = ""):
        super().__init__()
        self.fig = Figure(figsize=(6, 4), tight_layout=True)
        self.canvas = FigureCanvas(self.fig)
        self.ax = self.fig.add_subplot(111)
        lay = QVBoxLayout(self)
        lay.setContentsMargins(0, 0, 0, 0)
        lay.addWidget(self.canvas)
        if titulo:
            self.ax.set_title(titulo)

    def clear(self):
        self.ax.clear()

    def draw(self):
        self.canvas.draw_idle()


class Spark(QWidget):
    """Mini-gráfico sem eixos para tendência (sparklines)."""
    def __init__(self, width=2.2, height=0.6):
        super().__init__()
        self.fig = Figure(figsize=(width, height), tight_layout=True)
        self.canvas = FigureCanvas(self.fig)
        self.ax = self.fig.add_subplot(111)
        lay = QVBoxLayout(self)
        lay.setContentsMargins(0, 0, 0, 0)
        lay.addWidget(self.canvas)
        self._setup()

    def _setup(self):
        self.ax.set_axis_off()

    def clear(self):
        self.ax.cla()
        self._setup()

    def draw(self):
        self.canvas.draw_idle()


# ----------------- Módulo principal -----------------
class ModuloAnalises(QTabWidget):
    """
    Módulo 13 – Análises & Projeções
    - Visão Geral (KPIs + Receita/Volume por mês)
    - Convênios (ranking, participação)
    - Médicos (pagamentos/volume)
    - Procedimentos (mix)
    - Regiões (ranking por Cidade/UF/Região)
    - Previsões (tendência + projeção)
    """
    
    # ---------- Normalização de nome de empresa ----------
    def _normalize_empresa(self, s: str) -> str:
        """
        Reduz variações: remove acentos, pontuação, espaços extras e tokens pouco informativos.
        Ex.: "Hospital X LTDA", "HOSP. X ltda", "HOSPITAL X" -> "HOSPITAL X" -> com stopwords -> "X"
        """
        t = str(s or "").strip().upper()
        # remove acentos
        t = unicodedata.normalize('NFKD', t)
        t = ''.join(ch for ch in t if not unicodedata.combining(ch))
        # mantém apenas letras/números/espaço
        t = re.sub(r'[^A-Z0-9 ]+', ' ', t)
        t = re.sub(r'\s+', ' ', t).strip()

        STOP = {
            'LTDA','EIRELI','ME','EPP','SA','S A','HOLDING','GRUPO',
            'HOSPITAL','CLINICA','CLINICO','LABORATORIO','LAB','SERVICOS','SERVICO',
            'ASSOCIACAO','COOPERATIVA','FUNDACAO','INSTITUTO',
            'DE','DA','DO','DAS','DOS','SAUDE','SISTEMA'
        }
        tokens = [tok for tok in t.split() if tok not in STOP and len(tok) > 1]
        if not tokens:
            # volta ao original "limpo" se ficar vazio
            tokens = [w for w in t.split() if len(w) > 1] or [t]
        return ' '.join(tokens)

    def _canonical_empresa(self, serie_original):
        """Seleciona um nome 'canônico' para exibição: o mais frequente entre os originais."""
        vc = serie_original.value_counts()
        return str(vc.index[0]) if not vc.empty else ""

    # ---------- Empesas: UI ----------
    def _montar_empresas(self):
        self.tab_empresas = QWidget()
        self.addTab(self.tab_empresas, "Empresas")

        root = QVBoxLayout(self.tab_empresas)
        root.setContentsMargins(20, 20, 20, 20)
        root.setSpacing(18)

        # Filtros padrão do módulo (já incluem 'Base: Receita | Volume')
        filtros = self._make_filtros_bar(self.tab_empresas)
        root.addLayout(filtros)

        # Controles da aba
        top = QHBoxLayout()
        top.addWidget(QLabel("Top N:"))
        self.spin_emp_topn = QSpinBox(); self.spin_emp_topn.setRange(3, 50); self.spin_emp_topn.setValue(10)
        top.addWidget(self.spin_emp_topn)
        self.chk_emp_outros = QCheckBox("Agrupar restantes em 'Outros'"); self.chk_emp_outros.setChecked(True)
        top.addSpacing(12); top.addWidget(self.chk_emp_outros)
        top.addStretch(1)
        root.addLayout(top)
        self.spin_emp_topn.valueChanged.connect(self._update_empresas)
        self.chk_emp_outros.toggled.connect(self._update_empresas)

        # Gráfico principal: ranking por empresa
        self.chart_empresas = Chart("Empresas – Volume/Receita (Top N)")
        root.addWidget(self.chart_empresas)

        # Tabela com detalhamento
        self.tbl_emp = QTableWidget()
        self.tbl_emp.setColumnCount(5)
        self.tbl_emp.setHorizontalHeaderLabels(["Empresa (canônica)", "Volume", "% do Total", "# Convênios", "Top convênios"])
        self.tbl_emp.verticalHeader().setVisible(False)
        self.tbl_emp.setEditTriggers(QAbstractItemView.NoEditTriggers)
        self.tbl_emp.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.tbl_emp.setSelectionMode(QAbstractItemView.SingleSelection)
        self.tbl_emp.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        self.tbl_emp.itemSelectionChanged.connect(self._on_empresa_selected)
        root.addWidget(self.tbl_emp)

        # Gráfico de detalhamento por convênio (ao selecionar uma linha)
        self.chart_empresas_conv = Chart("Empresa selecionada – Convênios (Volume/Receita)")
        root.addWidget(self.chart_empresas_conv)

        # Comparativo (mesmo padrão das outras abas)
        self.comp_emp_box, self.comp_emp_refs = self._make_comp_box()
        root.addWidget(self.comp_emp_box)

        # Primeira renderização
        self._update_empresas()
        self._update_comparativos(self.comp_emp_refs)

    def _update_empresas(self):
        df = self._filtrar_df()
        self.chart_empresas.clear()
        self.tbl_emp.setRowCount(0)
        self.chart_empresas_conv.clear()

        # Descobre a coluna de empresa
        emp_col = next((c for c in ["empresa", "Empresa", "EMPRESA"] if c in df.columns), None)
        if df.empty or emp_col is None:
            self.chart_empresas.ax.text(0.5, 0.5, "Sem dados ou coluna 'empresa' não encontrada", ha='center', va='center')
            self.chart_empresas.draw()
            return

        base = self._base_kind()  # "receita" ou "volume"
        tmp = df[[emp_col]].copy()
        # Traz convênio se existir (para # convênios distintos e top convênios)
        if "Convenio" in df.columns:
            tmp["Convenio"] = df["Convenio"].astype(str)
        else:
            tmp["Convenio"] = ""

        # métrica
        if base == "receita" and "Valor Convenio" in df.columns:
            tmp["valor"] = pd.to_numeric(df["Valor Convenio"], errors="coerce").fillna(0.0)
        else:
            tmp["valor"] = 1.0  # volume

        # chave normalizada + nome canônico
        tmp["__emp_norm__"] = tmp[emp_col].map(self._normalize_empresa).fillna("")
        # agrega
        g = tmp.groupby("__emp_norm__")["valor"].sum().reset_index(name="Metrica")
        g = g.sort_values("Metrica", ascending=False)
        total = float(g["Metrica"].sum()) or 1.0

        # nome canônico e # convênios distintos
        can = tmp.groupby("__emp_norm__")[emp_col].apply(self._canonical_empresa).reset_index(name="Canonico")
        convs = tmp.groupby("__emp_norm__")["Convenio"].nunique().reset_index(name="Conv_Dist")
        g = g.merge(can, on="__emp_norm__", how="left").merge(convs, on="__emp_norm__", how="left")

        # Top convênios por empresa (pré-cálculo para preencher tabela)
        dist = tmp.groupby(["__emp_norm__", "Convenio"]).size().reset_index(name="qtd")
        dist_receita = None
        if base == "receita":
            dist_receita = tmp.groupby(["__emp_norm__", "Convenio"])["valor"].sum().reset_index(name="val")

        # Top N + Outros
        topn = self.spin_emp_topn.value() if hasattr(self, "spin_emp_topn") else 15
        if len(g) > topn and self.chk_emp_outros.isChecked():
            top = g.iloc[:topn].copy()
            outros_sum = g.iloc[topn:]["Metrica"].sum()
            if outros_sum > 0:
                top = pd.concat([top, pd.DataFrame({"__emp_norm__": ["__OUTROS__"], "Metrica": [outros_sum],
                                                    "Canonico": ["Outros"], "Conv_Dist": [0]})], ignore_index=True)
        else:
            top = g.iloc[:topn].copy()

        # --- Gráfico principal ---
        y = top["Canonico"].iloc[::-1]
        xvals = top["Metrica"].iloc[::-1]
        bars = self.chart_empresas.ax.barh(y, xvals)
        self.chart_empresas.ax.set_xlabel("Receita (R$)" if base == "receita" else "# Exames")
        self.chart_empresas.ax.yaxis.set_tick_params(labelsize=8)
        if base == "receita":
            self.chart_empresas.ax.xaxis.set_major_formatter(FuncFormatter(lambda x, p: (f"R$ {x:,.0f}".replace(",","X").replace(".",",").replace("X","."))))
        for rect in bars:
            w = rect.get_width()
            val_txt = (f"R$ {w:,.0f}".replace(",","X").replace(".",",").replace("X",".")) if base=="receita" else str(int(w))
            self.chart_empresas.ax.text(w*1.01, rect.get_y()+rect.get_height()/2, val_txt, va='center', fontsize=8)
        self.chart_empresas.ax.grid(True, axis='y', linestyle='--', alpha=0.3)
        self.chart_empresas.ax.set_title(f"Empresas – {'Receita' if base=='receita' else 'Volume'} (Top {topn})")
        self.chart_empresas.draw()

        # --- Tabela ---
        for _, row in g.iterrows():
            r = self.tbl_emp.rowCount(); self.tbl_emp.insertRow(r)
            item_emp = QTableWidgetItem(str(row["Canonico"] or row["__emp_norm__"]))
            item_emp.setData(Qt.UserRole, row["__emp_norm__"])  # chave para o detalhamento
            self.tbl_emp.setItem(r, 0, item_emp)
            self.tbl_emp.setItem(r, 1, QTableWidgetItem(
                (f"R$ {row['Metrica']:,.0f}".replace(",","X").replace(".",",").replace("X",".")) if base=="receita" else str(int(row["Metrica"]))
            ))
            self.tbl_emp.setItem(r, 2, QTableWidgetItem(f"{row['Metrica']/total*100:.1f}%"))
            self.tbl_emp.setItem(r, 3, QTableWidgetItem(str(int(row.get("Conv_Dist", 0)))))

            # Top convênios da empresa (string compacta)
            key = row["__emp_norm__"]
            d = dist[dist["__emp_norm__"] == key]
            if base == "receita" and dist_receita is not None:
                d = d.merge(dist_receita[dist_receita["__emp_norm__"] == key], on=["__emp_norm__","Convenio"], how="left")
                d = d.sort_values("val", ascending=False)
                topc = d.head(3)
                tot_val = float(d["val"].sum()) or 1.0
                txt = " | ".join(f"{c} ({v/tot_val*100:.0f}%)" for c, v in zip(topc["Convenio"], topc["val"]))
            else:
                d = d.sort_values("qtd", ascending=False)
                topc = d.head(3)
                tot_q = int(d["qtd"].sum()) or 1
                txt = " | ".join(f"{c} ({q/tot_q*100:.0f}%)" for c, q in zip(topc["Convenio"], topc["qtd"]))
            self.tbl_emp.setItem(r, 4, QTableWidgetItem(txt))

        self.tbl_emp.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)

        # limpa o gráfico de convênios até selecionar
        self.chart_empresas_conv.ax.text(0.5, 0.5, "Selecione uma empresa na tabela", ha='center', va='center')
        self.chart_empresas_conv.draw()

    def _on_empresa_selected(self):
        row = self.tbl_emp.currentRow()
        if row < 0:
            return
        key = self.tbl_emp.item(row, 0).data(Qt.UserRole)
        if not key:
            return
        self._plot_empresa_convenios(key)

    def _plot_empresa_convenios(self, emp_key: str):
        df = self._filtrar_df()
        self.chart_empresas_conv.clear()
        emp_col = next((c for c in ["empresa", "Empresa", "EMPRESA"] if c in df.columns), None)
        if df.empty or emp_col is None:
            self.chart_empresas_conv.ax.text(0.5, 0.5, "Sem dados", ha='center', va='center')
            self.chart_empresas_conv.draw()
            return
        base = self._base_kind()

        df = df.copy()
        df["__emp_norm__"] = df[emp_col].map(self._normalize_empresa)
        d = df[df["__emp_norm__"] == emp_key]
        if d.empty or "Convenio" not in d.columns:
            self.chart_empresas_conv.ax.text(0.5, 0.5, "Sem convênios para esta empresa", ha='center', va='center')
            self.chart_empresas_conv.draw()
            return

        if base == "receita" and "Valor Convenio" in d.columns:
            g = d.groupby("Convenio")["Valor Convenio"].sum().reset_index(name="Metrica")
        else:
            g = d.groupby("Convenio").size().reset_index(name="Metrica")
        g = g.sort_values("Metrica", ascending=False).head(12)

        bars = self.chart_empresas_conv.ax.barh(g["Convenio"][::-1], g["Metrica"][::-1])
        self.chart_empresas_conv.ax.set_xlabel("Receita (R$)" if base == "receita" else "# Exames")
        if base == "receita":
            self.chart_empresas_conv.ax.xaxis.set_major_formatter(FuncFormatter(lambda x, p: (f"R$ {x:,.0f}".replace(",","X").replace(".",",").replace("X","."))))
        for rect in bars:
            w = rect.get_width()
            val_txt = (f"R$ {w:,.0f}".replace(",","X").replace(".",",").replace("X",".")) if base=="receita" else str(int(w))
            self.chart_empresas_conv.ax.text(w*1.01, rect.get_y()+rect.get_height()/2, val_txt, va='center', fontsize=8)
        self.chart_empresas_conv.ax.grid(True, axis='y', linestyle='--', alpha=0.3)
        self.chart_empresas_conv.ax.set_title("Convênios por Empresa (Top 12)")
        self.chart_empresas_conv.draw()


    def __init__(self, parent=None):
        super().__init__(parent)
        self._controls: dict = {}  # combos por aba

        self._df = self._carregar_df()

        # === abas ===
        self.tab_overview = QWidget(); self.addTab(self.tab_overview, "Visão Geral")
        self.tab_convenio = QWidget(); self.addTab(self.tab_convenio, "Convênios")
        self.tab_medicos  = QWidget(); self.addTab(self.tab_medicos,  "Médicos")
        self.tab_proc     = QWidget(); self.addTab(self.tab_proc,     "Procedimentos")
        self.tab_reg      = QWidget(); self.addTab(self.tab_reg,      "Regiões")
        self.tab_prev     = QWidget(); self.addTab(self.tab_prev,     "Previsões")

        self._montar_overview()
        self._montar_convenios()
        self._montar_medicos()
        self._montar_procedimentos()
        self._montar_regioes()
        self._montar_empresas()
        self._montar_previsoes()

        self._aplicar_estilo()

    # -------------- base de dados --------------
    def _carregar_df(self) -> pd.DataFrame:
        if not os.path.exists(CAMINHO_BANCO):
            return pd.DataFrame()
        con = sqlite3.connect(CAMINHO_BANCO)
        try:
            df = pd.read_sql_query("SELECT * FROM registros_financeiros", con)
        except Exception:
            con.close(); return pd.DataFrame()
        con.close()

        # Normaliza nomes comuns
        def _ensure(_df: pd.DataFrame, target: str, candidates: list[str]) -> pd.DataFrame:
            if target in _df.columns:
                return _df
            for c in candidates:
                if c in _df.columns and c != target:
                    _df = _df.rename(columns={c: target}); break
            return _df

        df = _ensure(df, "competencia", ["Competência", "competência", "COMPETÊNCIA", "Competencia", "COMPETENCIA"])  # MM/AAAA
        df = _ensure(df, "Convenio", ["Convênio", "convenio", "CONVÊNIO", "CONVENIO"])  
        df = _ensure(df, "Medico", ["Médico", "medico", "MÉDICO", "MEDICO"])          
        df = _ensure(df, "Valor Convenio", ["Valor Convênio", "valor_convenio", "Valor_Convenio", "valor_convenio_float"])  
        df = _ensure(df, "Valor Medico", ["Valor Médico", "valor_medico", "Valor_Medico", "valor_medico_float"])             
        df = _ensure(df, "Procedimento", ["procedimento", "Procedimentos", "PROCEDIMENTO"])                                   
        df = _ensure(df, "Tipo Exame", ["TipoExame", "tipo_exame", "Tipo_Exame"])                                           

        for c in ("Valor Convenio", "Valor Medico"):
            if c in df.columns:
                df[c] = pd.to_numeric(df[c], errors="coerce").fillna(0.0)

        if "competencia" in df.columns:
            df["competencia"] = df["competencia"].astype(str)
        else:
            if "Data Exame" in df.columns:
                comp = pd.to_datetime(df["Data Exame"], dayfirst=True, errors="coerce").dt.strftime("%m/%Y")
                df["competencia"] = comp.fillna("")
            else:
                df["competencia"] = ""

        for c in ("Convenio", "Medico", "Procedimento", "Tipo Exame"):
            if c in df.columns:
                df[c] = df[c].astype(str)

        # Tenta enriquecer com Cidade/UF a partir da TABELA DE CONVÊNIOS
        df = self._attach_geo_from_convenios(df)

        df["_key"] = df["competencia"].map(_competencia_key)
        return df

    def _attach_geo_from_convenios(self, df: pd.DataFrame) -> pd.DataFrame:
        """Enriquece o DF com colunas Cidade/UF vindas da tabela "convenios" (ou similares)."""
        rf_key = "Convenio" if "Convenio" in df.columns else None
        if rf_key is None:
            return df

        # tenta ler uma das tabelas possíveis
        con = sqlite3.connect(CAMINHO_BANCO)
        conv_df = None
        for table in ("convenios", "cadastro_convenios", "cadastro_convenio", "tabela_convenios"):
            try:
                conv_df = pd.read_sql_query(f"SELECT * FROM {table}", con)
                if not conv_df.empty:
                    break
            except Exception:
                conv_df = None
        con.close()
        if conv_df is None or conv_df.empty:
            return df

        def _ensure(_df: pd.DataFrame, target: str, candidates: list[str]) -> pd.DataFrame:
            if target in _df.columns:
                return _df
            for c in candidates:
                if c in _df.columns and c != target:
                    _df = _df.rename(columns={c: target}); break
            return _df

        conv_df = _ensure(conv_df, "Convenio", ["Convenio", "Convênio", "Nome", "Nome Convenio", "Nome Convênio", "Descricao", "Descrição", "descricao", "nome"])
        conv_df = _ensure(conv_df, "Cidade",  ["Cidade", "cidade", "Municipio", "Município", "municipio"])  
        conv_df = _ensure(conv_df, "UF",      ["UF", "Estado", "estado", "Uf", "uf"])                    
        if "UF" in conv_df.columns:
            conv_df["UF"] = conv_df["UF"].astype(str).str.upper().str[:2]
        for col in ("Convenio", "Cidade"):
            if col in conv_df.columns:
                conv_df[col] = conv_df[col].astype(str)

        # escolhe melhor coluna de nome para join, caso "Convenio" não exista
        join_right = "Convenio" if "Convenio" in conv_df.columns else None
        if join_right is None:
            # escolhe a coluna texto com maior interseção com rf[Convenio]
            set_rf = set(df[rf_key].dropna().astype(str).unique())
            best, best_score = None, -1
            for col in conv_df.columns:
                if conv_df[col].dtype == object:
                    score = len(set(conv_df[col].dropna().astype(str).unique()) & set_rf)
                    if score > best_score:
                        best, best_score = col, score
            join_right = best

        if not join_right:
            return df

        cols_keep = [c for c in (join_right, "Cidade", "UF") if c in conv_df.columns]
        if len(cols_keep) <= 1:
            return df
        conv_min = conv_df[cols_keep].drop_duplicates(subset=[join_right])
        merged = df.merge(conv_min, left_on=rf_key, right_on=join_right, how="left", suffixes=("", "_conv"))
        # move para nomes finais
        if "Cidade" in merged.columns:
            merged["Cidade"] = merged["Cidade"].astype(str)
        if "UF" in merged.columns:
            merged["UF"] = merged["UF"].astype(str).str.upper().str[:2]
        return merged

    # -------------- filtros reutilizáveis --------------
    def _make_filtros_bar(self, host_tab: QWidget) -> QHBoxLayout:
        bar = QHBoxLayout(); bar.setSpacing(8)

        comps = sorted(set(self._df.get("competencia", pd.Series(dtype=str)).dropna().astype(str)), key=_competencia_key)
        if not comps:
            comps = ["01/2025"]
        combo_comp = QComboBox(); combo_comp.addItem("Todas as Competências")
        for c in comps: combo_comp.addItem(c)
        combo_comp.setCurrentIndex(len(comps))
        combo_comp.setObjectName("comp")
        bar.addWidget(QLabel("Competência:")); bar.addWidget(combo_comp)

        convs = sorted([c for c in self._df.get("Convenio", pd.Series(dtype=str)).dropna().unique()])
        combo_convenio = QComboBox(); combo_convenio.addItem("Todos")
        for c in convs: combo_convenio.addItem(str(c))
        combo_convenio.setObjectName("convenio")
        bar.addSpacing(12); bar.addWidget(QLabel("Convênio:")); bar.addWidget(combo_convenio)

        meds = sorted([m for m in self._df.get("Medico", pd.Series(dtype=str)).dropna().unique()])
        combo_medico = QComboBox(); combo_medico.addItem("Todos")
        for m in meds: combo_medico.addItem(str(m))
        combo_medico.setObjectName("medico")
        bar.addSpacing(12); bar.addWidget(QLabel("Médico:")); bar.addWidget(combo_medico)

        # Toggle de base
        combo_base = QComboBox(); combo_base.addItems(["Receita (R$)", "Volume (# exames)"])
        combo_base.setObjectName("base")
        bar.addSpacing(12); bar.addWidget(QLabel("Base:")); bar.addWidget(combo_base)

        btn_atualizar = QPushButton("Atualizar"); btn_atualizar.clicked.connect(self._refresh_all)
        bar.addSpacing(12); bar.addWidget(btn_atualizar)

        self._controls[host_tab] = {"comp": combo_comp, "convenio": combo_convenio, "medico": combo_medico, "base": combo_base}
        combo_comp.currentIndexChanged.connect(self._refresh_all)
        combo_convenio.currentIndexChanged.connect(self._refresh_all)
        combo_medico.currentIndexChanged.connect(self._refresh_all)
        combo_base.currentIndexChanged.connect(self._refresh_all)

        bar.addStretch(1)
        return bar

    def _current_ctrls(self) -> Tuple[str, str, str]:
        ctrls = self._controls.get(self.currentWidget(), {})
        comp = ctrls.get("comp").currentText() if ctrls.get("comp") else "Todas as Competências"
        conv = ctrls.get("convenio").currentText() if ctrls.get("convenio") else "Todos"
        med  = ctrls.get("medico").currentText() if ctrls.get("medico") else "Todos"
        return comp, conv, med

    def _base_kind(self, tab: Optional[Widget] = None) -> str:
        ctrls = self._controls.get(tab or self.currentWidget(), {})
        txt = ctrls.get("base").currentText() if ctrls.get("base") else "Receita (R$)"
        return "volume" if "Volume" in str(txt) else "receita"

    def _filtrar_df(self) -> pd.DataFrame:
        if self._df.empty:
            return self._df
        df = self._df.copy()
        comp_sel, conv_sel, med_sel = self._current_ctrls()
        if isinstance(comp_sel, str) and comp_sel.strip() and comp_sel != "Todas as Competências":
            df = df[df["competencia"] == comp_sel]
        if conv_sel != "Todos" and "Convenio" in df.columns:
            df = df[df["Convenio"] == conv_sel]
        if med_sel != "Todos" and "Medico" in df.columns:
            df = df[df["Medico"] == med_sel]
        return df

    # ---------- helpers de comparativo / formatação ----------
    def _df_no_comp(self, conv_sel: str, med_sel: str) -> pd.DataFrame:
        df = self._df.copy()
        if conv_sel != "Todos" and "Convenio" in df.columns:
            df = df[df["Convenio"] == conv_sel]
        if med_sel != "Todos" and "Medico" in df.columns:
            df = df[df["Medico"] == med_sel]
        return df

    def _last_months(self, comps: list[str], n: int = 6) -> list[str]:
        comps_sorted = sorted([c for c in comps if isinstance(c, str)], key=_competencia_key)
        return comps_sorted[-n:]

    def _mom_delta(self, series: pd.Series) -> Optional[float]:
        if len(series) < 2:
            return None
        prev = float(series.iloc[-2]); curr = float(series.iloc[-1])
        if prev == 0:
            return None
        return (curr - prev) / prev * 100.0

    def _axis_label(self, base: str) -> str:
        return "Receita (R$)" if base == "receita" else "# Exames"

    def _fmt_value_by_base(self, v: float, base: str) -> str:
        return fmt_money(v) if base == "receita" else str(int(v))

    # -------------- Comparativo rápido --------------
    def _make_comp_box(self) -> Tuple[QFrame, dict]:
        box = QFrame(); box.setObjectName("compBox")
        box.setStyleSheet("""
            QFrame#compBox { background:#ffffff; border:1px solid #e6eef5; border-radius:10px; }
            QLabel[role='title'] { color:#59636e; font-weight:600; }
            QLabel[role='value'] { font-size:12px; font-weight:600; }
        """)
        grid = QGridLayout(box); grid.setContentsMargins(12,10,12,10); grid.setHorizontalSpacing(16); grid.setVerticalSpacing(6)

        def row(r, titulo):
            lt = QLabel(titulo); lt.setProperty("role", "title"); grid.addWidget(lt, r, 0)
            lv = QLabel("–"); lv.setProperty("role", "value"); grid.addWidget(lv, r, 1)
            return lv

        refs = {
            "receita":  row(0, "Últ. meses – Receita"),
            "exames":   row(1, "Últ. meses – # Exames"),
            "top_proc": row(2, "Top Procedimento (mês alvo)"),
            "top_med":  row(3, "Top Médico (mês alvo)"),
        }
        # Sparklines
        spark_r = Spark(); grid.addWidget(spark_r, 0, 2)
        spark_q = Spark(); grid.addWidget(spark_q, 1, 2)
        refs["spark_receita"] = spark_r
        refs["spark_exames"]  = spark_q
        return box, refs

    def _fill_comp_box(self, refs: dict, comp_sel: str, conv_sel: str, med_sel: str):
        base = self._df_no_comp(conv_sel, med_sel)
        if base.empty:
            for k in ("receita","exames","top_proc","top_med"):
                refs[k].setText("–")
            refs["spark_receita"].clear(); refs["spark_receita"].draw()
            refs["spark_exames"].clear();  refs["spark_exames"].draw()
            return

        gm = base.groupby("competencia").agg(receita=("Valor Convenio","sum"), exames=("competencia","count")).reset_index()
        gm = gm.sort_values("competencia", key=lambda s: s.map(_competencia_key))
        last6 = self._last_months(list(gm["competencia"].astype(str)), 6)
        gm6 = gm[gm["competencia"].isin(last6)]
        last3 = self._last_months(list(gm["competencia"].astype(str)), 3)
        gm3 = gm[gm["competencia"].isin(last3)]

        def _fmt_last(df_col, df_use):
            pares = []
            for _, r in df_use.iterrows():
                pares.append(f"{r['competencia']}: {fmt_money(r[df_col])}" if df_col=="receita" else f"{r['competencia']}: {int(r[df_col])}")
            return "  |  ".join(pares)

        mom_r = self._mom_delta(gm3["receita"]) if len(gm3)>=2 else None
        mom_q = self._mom_delta(gm3["exames"]) if len(gm3)>=2 else None
        refs["receita"].setText(((_fmt_last("receita", gm3) + (f"  |  Δ M/M: {mom_r:+.1f}%" if mom_r is not None else "")) or "–"))
        refs["exames"].setText(((_fmt_last("exames", gm3) + (f"  |  Δ M/M: {mom_q:+.1f}%" if mom_q is not None else "")) or "–"))

        # Sparklines (últimos 6 meses)
        for key, col, spark in (("receita","receita",refs["spark_receita"]), ("exames","exames",refs["spark_exames"])):
            spark.clear()
            y = gm6[col].astype(float).values
            x = np.arange(len(y))
            if len(y) > 0:
                spark.ax.plot(x, y, marker='o')
            spark.draw()

        # mês alvo
        mes_alvo = comp_sel if comp_sel != "Todas as Competências" else (gm["competencia"].iloc[-1] if not gm.empty else None)
        if mes_alvo is None:
            refs["top_proc"].setText("–"); refs["top_med"].setText("–"); return
        df_mes = base[base["competencia"] == mes_alvo]

        if "Procedimento" in df_mes.columns and not df_mes.empty:
            rk_p = df_mes.groupby("Procedimento").size().reset_index(name="qtd").sort_values("qtd", ascending=False)
            proc = rk_p.iloc[0]["Procedimento"]; qtd = int(rk_p.iloc[0]["qtd"])
            aa, mm = _competencia_key(mes_alvo); prev = f"{(mm-1 if mm>1 else 12):02d}/{aa if mm>1 else aa-1:04d}"
            df_prev = base[base["competencia"] == prev]
            qtd_prev = int(df_prev[df_prev.get("Procedimento") == proc].shape[0]) if not df_prev.empty else 0
            delta = (qtd - qtd_prev) / qtd_prev * 100.0 if qtd_prev else None
            refs["top_proc"].setText(f"{proc}: {qtd}  |  Δ M/M: {delta:+.1f}%" if delta is not None else f"{proc}: {qtd}")
        else:
            refs["top_proc"].setText("–")

        if "Medico" in df_mes.columns and not df_mes.empty:
            rk_m = df_mes.groupby("Medico").size().reset_index(name="qtd").sort_values("qtd", ascending=False)
            med = rk_m.iloc[0]["Medico"]; qtdm = int(rk_m.iloc[0]["qtd"])
            aa, mm = _competencia_key(mes_alvo); prev = f"{(mm-1 if mm>1 else 12):02d}/{aa if mm>1 else aa-1:04d}"
            df_prev = base[base["competencia"] == prev]
            qtd_prev = int(df_prev[df_prev.get("Medico") == med].shape[0]) if not df_prev.empty else 0
            delta = (qtdm - qtd_prev) / qtd_prev * 100.0 if qtd_prev else None
            refs["top_med"].setText(f"{med}: {qtdm}  |  Δ M/M: {delta:+.1f}%" if delta is not None else f"{med}: {qtdm}")
        else:
            refs["top_med"].setText("–")

    def _update_comparativos(self, refs: dict):
        comp_sel, conv_sel, med_sel = self._current_ctrls()
        self._fill_comp_box(refs, comp_sel, conv_sel, med_sel)

    # -------------- Visão Geral --------------
    def _montar_overview(self):
        root = QVBoxLayout(self.tab_overview); root.setContentsMargins(20, 20, 20, 20); root.setSpacing(18)
        filtros = self._make_filtros_bar(self.tab_overview); root.addLayout(filtros)

        # Controles de visualização
        vis = QHBoxLayout()
        vis.addWidget(QLabel("Exibir:"))
        self.combo_chart_mode = QComboBox(); self.combo_chart_mode.addItems(["Barras (horizontal)", "Barras (vertical)", "Pizza"])
        vis.addWidget(self.combo_chart_mode)
        vis.addSpacing(12); vis.addWidget(QLabel("Top N:"))
        self.spin_topn = QSpinBox(); self.spin_topn.setRange(3, 50); self.spin_topn.setValue(12); vis.addWidget(self.spin_topn)
        vis.addSpacing(12)
        self.chk_outros = QCheckBox("Agrupar restantes em 'Outros'"); self.chk_outros.setChecked(True); vis.addWidget(self.chk_outros)
        vis.addStretch(1)
        self.combo_chart_mode.currentIndexChanged.connect(self._update_overview)
        self.spin_topn.valueChanged.connect(self._update_overview)
        self.chk_outros.toggled.connect(self._update_overview)
        root.addLayout(vis)

        # KPIs
        self.kpi_bar = self._kpi_bar(); root.addWidget(self.kpi_bar)

        # Gráfico
        self.chart_receita = Chart("Receita / Volume por Mês"); root.addWidget(self.chart_receita)

        # Comparativo rápido
        self.comp_overview_box, self.comp_overview_refs = self._make_comp_box(); root.addWidget(self.comp_overview_box)

        # Exportações
        btns = QHBoxLayout()
        self.btn_exportar_png = QPushButton("Exportar gráfico em PNG"); self.btn_exportar_png.clicked.connect(lambda: self._exportar_chart(self.chart_receita))
        self.btn_exportar_excel = QPushButton("Exportar dados em Excel"); self.btn_exportar_excel.clicked.connect(self._exportar_excel_overview)
        btns.addWidget(self.btn_exportar_png); btns.addWidget(self.btn_exportar_excel); btns.addStretch(1); root.addLayout(btns)

        self._update_overview(); self._update_comparativos(self.comp_overview_refs)

    def _kpi_bar(self) -> QFrame:
        frame = QFrame(); frame.setObjectName("kpiBar")
        frame.setStyleSheet("""
            QFrame#kpiBar { background:#ffffff; border:1px solid #e6eef5; border-radius:10px; }
            QLabel[role='title'] { color:#59636e; font-weight:600; }
            QLabel[role='value'] { font-size:18px; font-weight:700; }
        """)
        lay = QHBoxLayout(frame); lay.setContentsMargins(16, 12, 16, 12); lay.setSpacing(24)

        def _card(titulo: str) -> Tuple[QFrame, QLabel]:
            c = QFrame(); c.setStyleSheet("QFrame{background:transparent}")
            cl = QVBoxLayout(c); cl.setContentsMargins(8, 4, 8, 4)
            t = QLabel(titulo); t.setProperty("role", "title")
            v = QLabel("–"); v.setProperty("role", "value")
            cl.addWidget(t); cl.addWidget(v)
            return c, v

        c1, self.kpi_receita = _card("Receita Total")
        c2, self.kpi_exames  = _card("# Exames")
        c3, self.kpi_ticket  = _card("Ticket Médio")
        c4, self.kpi_vmedico = _card("Total Médicos")
        c5, self.kpi_crescm  = _card("Δ M/M")
        for c in (c1,c2,c3,c4,c5): lay.addWidget(c)
        lay.addStretch(1)
        return frame

    def _update_overview(self):
        df = self._filtrar_df()
        if df.empty:
            self.chart_receita.clear(); self.chart_receita.ax.text(0.5, 0.5, "Sem dados", ha='center', va='center'); self.chart_receita.draw()
            for w in (self.kpi_receita, self.kpi_exames, self.kpi_ticket, self.kpi_vmedico, self.kpi_crescm): w.setText("–")
            return

        comp_sel, conv_sel, _ = self._current_ctrls(); base = self._base_kind()

        receita_total = float(df.get("Valor Convenio", pd.Series(dtype=float)).sum())
        n_exames = int(len(df))
        v_medico = float(df.get("Valor Medico", pd.Series(dtype=float)).sum())
        ticket = receita_total / n_exames if n_exames else 0.0

        cresc = 0.0
        if comp_sel == "Todas as Competências":
            g_c = df.groupby("competencia")["Valor Convenio"].sum().reset_index().sort_values("competencia", key=lambda s: s.map(_competencia_key))
            if len(g_c) >= 2:
                atual = g_c["Valor Convenio"].iloc[-1]; prev  = g_c["Valor Convenio"].iloc[-2]
                cresc = ((atual - prev) / prev * 100.0) if prev else 0.0

        self.kpi_receita.setText(fmt_money(receita_total))
        self.kpi_exames.setText(str(n_exames))
        self.kpi_ticket.setText(fmt_money(ticket))
        self.kpi_vmedico.setText(fmt_money(v_medico))
        self.kpi_crescm.setText(f"{cresc:+.1f}%")

        # === GRÁFICO DINÂMICO ===
        self.chart_receita.clear(); self.chart_receita.ax.set_aspect('auto')
        if comp_sel == "Todas as Competências":
            if base == "receita":
                g = df.groupby("competencia")["Valor Convenio"].sum().reset_index()
            else:
                g = df.groupby("competencia").size().reset_index(name="Valor Convenio")
            g = g.sort_values("competencia", key=lambda s: s.map(_competencia_key))
            x = pd.to_datetime(g["competencia"].astype(str), format="%m/%Y", errors="coerce"); mask = ~x.isna()
            self.chart_receita.ax.plot(x[mask], g["Valor Convenio"][mask], marker='o')
            self.chart_receita.ax.set_title(("Receita" if base=="receita" else "Volume") + " por Competência" + (" – " + conv_sel if conv_sel != "Todos" else ""))
            self.chart_receita.ax.set_xlabel("Competência"); self.chart_receita.ax.set_ylabel(self._axis_label(base))
            self.chart_receita.ax.xaxis.set_major_locator(mdates.MonthLocator(interval=1))
            self.chart_receita.ax.xaxis.set_major_formatter(mdates.DateFormatter("%m/%Y"))
            if base == "receita":
                self.chart_receita.ax.yaxis.set_major_formatter(FuncFormatter(lambda x, p: (f"R$ {x:,.0f}".replace(",","X").replace(".",",").replace("X","."))))
            self.chart_receita.fig.autofmt_xdate()
        else:
            if conv_sel == "Todos" and "Convenio" in df.columns:
                if base == "receita":
                    g = df.groupby("Convenio")["Valor Convenio"].sum().reset_index().sort_values("Valor Convenio", ascending=False)
                else:
                    g = df.groupby("Convenio").size().reset_index(name="Valor Convenio").sort_values("Valor Convenio", ascending=False)
                topn = self.spin_topn.value() if hasattr(self, "spin_topn") else 12
                if len(g) > topn and getattr(self, "chk_outros", None) and self.chk_outros.isChecked():
                    top = g.iloc[:topn].copy(); outros_sum = g.iloc[topn:]["Valor Convenio"].sum()
                    if outros_sum > 0:
                        top = pd.concat([top, pd.DataFrame({"Convenio": ["Outros"], "Valor Convenio": [outros_sum]})], ignore_index=True)
                else:
                    top = g.iloc[:topn].copy()
                mode = self.combo_chart_mode.currentText() if hasattr(self, "combo_chart_mode") else "Barras (horizontal)"
                if "Pizza" in mode:
                    self.chart_receita.ax.pie(top["Valor Convenio"], labels=top["Convenio"], autopct=lambda p: f"{p:.1f}%")
                    self.chart_receita.ax.set_title(f"{('Receita' if base=='receita' else 'Volume')} por Convênio – {comp_sel}"); self.chart_receita.ax.set_aspect('equal', adjustable='box')
                elif "vertical" in mode:
                    bars = self.chart_receita.ax.bar(top["Convenio"], top["Valor Convenio"])
                    self.chart_receita.ax.set_title(f"{('Receita' if base=='receita' else 'Volume')} por Convênio – {comp_sel}")
                    self.chart_receita.ax.set_xlabel("Convênio"); self.chart_receita.ax.set_ylabel(self._axis_label(base))
                    self.chart_receita.ax.tick_params(axis='x', rotation=45)
                    for rect in bars:
                        h = rect.get_height(); self.chart_receita.ax.text(rect.get_x()+rect.get_width()/2, h, self._fmt_value_by_base(h, base), ha='center', va='bottom', fontsize=8, rotation=90)
                else:
                    y = top["Convenio"].iloc[::-1]; xvals = top["Valor Convenio"].iloc[::-1]
                    bars = self.chart_receita.ax.barh(y, xvals)
                    self.chart_receita.ax.set_title(f"{('Receita' if base=='receita' else 'Volume')} por Convênio – {comp_sel}")
                    self.chart_receita.ax.set_xlabel(self._axis_label(base)); self.chart_receita.ax.set_ylabel("")
                    self.chart_receita.ax.yaxis.set_tick_params(labelsize=8)
                    if base == "receita":
                        self.chart_receita.ax.xaxis.set_major_formatter(FuncFormatter(lambda x, p: (f"R$ {x:,.0f}".replace(",","X").replace(".",",").replace("X","."))))
                    for rect in bars:
                        w = rect.get_width(); self.chart_receita.ax.text(w * 1.01, rect.get_y() + rect.get_height()/2, self._fmt_value_by_base(w, base), va='center', fontsize=8)
                self.chart_receita.ax.grid(True, axis='y', linestyle='--', alpha=0.3)
            else:
                # convênio específico no mês
                if "Procedimento" in df.columns:
                    if base == "receita":
                        g = df.groupby("Procedimento")["Valor Convenio"].sum().reset_index().sort_values("Valor Convenio", ascending=False)
                    else:
                        g = df.groupby("Procedimento").size().reset_index(name="Valor Convenio").sort_values("Valor Convenio", ascending=False)
                    top = g.head(12)
                    bars = self.chart_receita.ax.bar(top["Procedimento"], top["Valor Convenio"])
                    self.chart_receita.ax.set_title(f"{conv_sel} – {('Receita' if base=='receita' else 'Volume')} por Procedimento – {comp_sel}")
                    self.chart_receita.ax.set_xlabel("Procedimento"); self.chart_receita.ax.set_ylabel(self._axis_label(base))
                    self.chart_receita.ax.tick_params(axis='x', rotation=45)
                    for rect in bars:
                        h = rect.get_height(); self.chart_receita.ax.text(rect.get_x()+rect.get_width()/2, h, self._fmt_value_by_base(h, base), ha='center', va='bottom', fontsize=8, rotation=90)
                else:
                    self.chart_receita.ax.text(0.5, 0.5, "Sem detalhamento de Procedimento", ha='center', va='center')
        self.chart_receita.fig.tight_layout(); self.chart_receita.draw()

    # -------------- Convênios --------------
    def _montar_convenios(self):
        root = QVBoxLayout(self.tab_convenio); root.setContentsMargins(20, 20, 20, 20); root.setSpacing(18)
        filtros = self._make_filtros_bar(self.tab_convenio); root.addLayout(filtros)

        self.chart_top_conv = Chart("Top por Convênio"); root.addWidget(self.chart_top_conv)
        self.comp_convenios_box, self.comp_convenios_refs = self._make_comp_box(); root.addWidget(self.comp_convenios_box)

        self.tbl_conv = QTableWidget(); self.tbl_conv.setColumnCount(4)
        self.tbl_conv.setHorizontalHeaderLabels(["Convênio", "Métrica", "% do Total", "# Exames"])
        self.tbl_conv.verticalHeader().setVisible(False)
        self.tbl_conv.setEditTriggers(QAbstractItemView.NoEditTriggers)
        self.tbl_conv.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.tbl_conv.setSelectionMode(QAbstractItemView.SingleSelection)
        self.tbl_conv.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        root.addWidget(self.tbl_conv)

        self._update_convenios(); self._update_comparativos(self.comp_convenios_refs)

    def _update_convenios(self):
        df = self._filtrar_df(); self.chart_top_conv.clear(); self.tbl_conv.setRowCount(0)
        ctrls = self._controls.get(self.currentWidget(), {})
        comp_sel = ctrls.get("comp").currentText() if ctrls.get("comp") else "Todas as Competências"
        conv_sel = ctrls.get("convenio").currentText() if ctrls.get("convenio") else "Todos"
        base = self._base_kind()

        if df.empty or "Convenio" not in df.columns:
            self.chart_top_conv.ax.text(0.5, 0.5, "Sem dados", ha='center', va='center'); self.chart_top_conv.draw()
            if hasattr(self, "comp_convenios_refs"): self._update_comparativos(self.comp_convenios_refs)
            return

        if comp_sel == "Todas as Competências" and conv_sel != "Todos":
            if base == "receita":
                g = df.groupby("competencia")["Valor Convenio"].sum().reset_index()
            else:
                g = df.groupby("competencia").size().reset_index(name="Valor Convenio")
            g = g.sort_values("competencia", key=lambda s: s.map(_competencia_key))
            x = pd.to_datetime(g["competencia"].astype(str), format="%m/%Y", errors="coerce"); mask = ~x.isna()
            self.chart_top_conv.ax.plot(x[mask], g["Valor Convenio"][mask], marker='o')
            self.chart_top_conv.ax.set_title(f"{conv_sel} – {('Receita' if base=='receita' else 'Volume')} por Competência")
            self.chart_top_conv.ax.set_ylabel(self._axis_label(base))
            self.chart_top_conv.ax.xaxis.set_major_locator(mdates.MonthLocator(interval=1))
            self.chart_top_conv.ax.xaxis.set_major_formatter(mdates.DateFormatter("%m/%Y"))
            self.chart_top_conv.ax.grid(True, linestyle='--', alpha=0.3); self.chart_top_conv.draw()

            self.tbl_conv.setColumnCount(3); self.tbl_conv.setHorizontalHeaderLabels(["Competência", ("Receita" if base=='receita' else "Qtde Exames"), "Qtde Exames"])
            if base == "receita":
                g2 = df.groupby("competencia").agg(Receita=("Valor Convenio","sum"), Qtd=("Convenio","count")).reset_index()
            else:
                g2 = df.groupby("competencia").agg(Qtde=("competencia","count")).reset_index().rename(columns={"Qtde":"Qtd"})
                g2["Receita"] = 0.0
            g2 = g2.sort_values("competencia", key=lambda s: s.map(_competencia_key))
            for _, row in g2.iterrows():
                r = self.tbl_conv.rowCount(); self.tbl_conv.insertRow(r)
                self.tbl_conv.setItem(r, 0, QTableWidgetItem(str(row["competencia"])));
                self.tbl_conv.setItem(r, 1, QTableWidgetItem(fmt_money(row["Receita"]) if base=='receita' else str(int(row["Qtd"]))))
                self.tbl_conv.setItem(r, 2, QTableWidgetItem(str(int(row.get("Qtd", 0)))))
            self.tbl_conv.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        else:
            if base == "receita":
                g = df.groupby("Convenio").agg(Metrica=("Valor Convenio", "sum"), Qtd=("Convenio", "count")).reset_index()
            else:
                g = df.groupby("Convenio").agg(Metrica=("Convenio", "count")).reset_index(); g["Qtd"] = g["Metrica"]
            g = g.sort_values("Metrica", ascending=False); total = float(g["Metrica"].sum()) or 1.0
            top = g.head(10)
            self.chart_top_conv.ax.bar(top["Convenio"], top["Metrica"])
            self.chart_top_conv.ax.set_ylabel(self._axis_label(base)); self.chart_top_conv.ax.tick_params(axis='x', rotation=45)
            self.chart_top_conv.ax.grid(True, axis='y', linestyle='--', alpha=0.3); self.chart_top_conv.draw()

            self.tbl_conv.setColumnCount(4); self.tbl_conv.setHorizontalHeaderLabels(["Convênio", ("Receita" if base=='receita' else "Qtde Exames"), "% do Total", "Qtde Exames"])
            for _, row in g.iterrows():
                r = self.tbl_conv.rowCount(); self.tbl_conv.insertRow(r)
                self.tbl_conv.setItem(r, 0, QTableWidgetItem(str(row["Convenio"])));
                self.tbl_conv.setItem(r, 1, QTableWidgetItem(fmt_money(row["Metrica"]) if base=='receita' else str(int(row["Metrica"]))));
                self.tbl_conv.setItem(r, 2, QTableWidgetItem(f"{row['Metrica']/total*100:.1f}%"));
                self.tbl_conv.setItem(r, 3, QTableWidgetItem(str(int(row.get("Qtd", row["Metrica"])))))
            self.tbl_conv.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)

        if hasattr(self, "comp_convenios_refs"): self._update_comparativos(self.comp_convenios_refs)

    # -------------- Médicos --------------
    def _montar_medicos(self):
        root = QVBoxLayout(self.tab_medicos); root.setContentsMargins(20, 20, 20, 20); root.setSpacing(18)
        filtros = self._make_filtros_bar(self.tab_medicos); root.addLayout(filtros)

        self.chart_medicos = Chart("Médicos – Receita / Volume"); root.addWidget(self.chart_medicos)
        self.comp_medicos_box, self.comp_medicos_refs = self._make_comp_box(); root.addWidget(self.comp_medicos_box)

        self.tbl_med = QTableWidget(); self.tbl_med.setColumnCount(3)
        self.tbl_med.setHorizontalHeaderLabels(["Médico", "Métrica", "# Exames"])
        self.tbl_med.verticalHeader().setVisible(False)
        self.tbl_med.setEditTriggers(QAbstractItemView.NoEditTriggers)
        self.tbl_med.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.tbl_med.setSelectionMode(QAbstractItemView.SingleSelection)
        self.tbl_med.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        root.addWidget(self.tbl_med)

        self._update_medicos(); self._update_comparativos(self.comp_medicos_refs)

    def _update_medicos(self):
        df = self._filtrar_df(); self.chart_medicos.clear(); self.tbl_med.setRowCount(0)
        comp_sel, _, _ = self._current_ctrls(); base = self._base_kind()
        if df.empty:
            self.chart_medicos.ax.text(0.5, 0.5, "Sem dados", ha='center', va='center'); self.chart_medicos.draw(); return

        if comp_sel == "Todas as Competências":
            if base == "receita":
                g = df.groupby("competencia")["Valor Medico"].sum().reset_index()
            else:
                g = df.groupby("competencia").size().reset_index(name="Valor Medico")
            g = g.sort_values("competencia", key=lambda s: s.map(_competencia_key))
            x = pd.to_datetime(g["competencia"].astype(str), format="%m/%Y", errors="coerce"); mask = ~x.isna()
            self.chart_medicos.ax.plot(x[mask], g["Valor Medico"][mask], marker='o')
            self.chart_medicos.ax.set_title(f"Médicos – {('Receita' if base=='receita' else 'Volume')} por Competência")
            self.chart_medicos.ax.set_ylabel(self._axis_label(base))
            self.chart_medicos.ax.xaxis.set_major_locator(mdates.MonthLocator(interval=1))
            self.chart_medicos.ax.xaxis.set_major_formatter(mdates.DateFormatter("%m/%Y"))
            self.chart_medicos.draw()
        else:
            if base == "receita":
                g = df.groupby("Medico").agg(Metrica=("Valor Medico","sum"), Qtd=("Medico","count")).reset_index().sort_values("Metrica", ascending=False)
            else:
                g = df.groupby("Medico").size().reset_index(name="Metrica").sort_values("Metrica", ascending=False); g["Qtd"] = g["Metrica"]
            top = g.head(12)
            bars = self.chart_medicos.ax.barh(top["Medico"][::-1], top["Metrica"][::-1])
            self.chart_medicos.ax.set_xlabel(self._axis_label(base))
            for rect in bars:
                w = rect.get_width(); self.chart_medicos.ax.text(w*1.01, rect.get_y()+rect.get_height()/2, self._fmt_value_by_base(w, base), va='center', fontsize=8)
            self.chart_medicos.draw()
        # tabela
        if base == "receita":
            gtab = df.groupby("Medico").agg(Metrica=("Valor Medico","sum"), Qtd=("Medico","count")).reset_index().sort_values("Metrica", ascending=False)
        else:
            gtab = df.groupby("Medico").size().reset_index(name="Metrica").sort_values("Metrica", ascending=False); gtab["Qtd"] = gtab["Metrica"]
        for _, row in gtab.iterrows():
            r = self.tbl_med.rowCount(); self.tbl_med.insertRow(r)
            self.tbl_med.setItem(r, 0, QTableWidgetItem(str(row["Medico"])));
            self.tbl_med.setItem(r, 1, QTableWidgetItem(fmt_money(row["Metrica"]) if base=='receita' else str(int(row["Metrica"]))));
            self.tbl_med.setItem(r, 2, QTableWidgetItem(str(int(row.get("Qtd", row["Metrica"])))))
        self.tbl_med.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)

    # -------------- Procedimentos --------------
    def _montar_procedimentos(self):
        root = QVBoxLayout(self.tab_proc); root.setContentsMargins(20, 20, 20, 20); root.setSpacing(18)
        filtros = self._make_filtros_bar(self.tab_proc); root.addLayout(filtros)

        self.chart_mix = Chart("Procedimentos – Receita / Volume"); root.addWidget(self.chart_mix)
        self.comp_proc_box, self.comp_proc_refs = self._make_comp_box(); root.addWidget(self.comp_proc_box)

        self.tbl_proc = QTableWidget(); self.tbl_proc.setColumnCount(3)
        self.tbl_proc.setHorizontalHeaderLabels(["Procedimento", "Métrica", "# Exames"])
        self.tbl_proc.verticalHeader().setVisible(False)
        self.tbl_proc.setEditTriggers(QAbstractItemView.NoEditTriggers)
        self.tbl_proc.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.tbl_proc.setSelectionMode(QAbstractItemView.SingleSelection)
        self.tbl_proc.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        root.addWidget(self.tbl_proc)

        self._update_procedimentos(); self._update_comparativos(self.comp_proc_refs)

    def _update_procedimentos(self):
        df = self._filtrar_df(); self.chart_mix.clear(); self.tbl_proc.setRowCount(0)
        comp_sel, conv_sel, _ = self._current_ctrls(); base = self._base_kind()
        if df.empty or "Procedimento" not in df.columns:
            self.chart_mix.ax.text(0.5, 0.5, "Sem dados", ha='center', va='center'); self.chart_mix.draw(); return

        if comp_sel == "Todas as Competências" and conv_sel != "Todos":
            if base == "receita":
                g = df.groupby("competencia")["Valor Convenio"].sum().reset_index()
            else:
                g = df.groupby("competencia").size().reset_index(name="Valor Convenio")
            g = g.sort_values("competencia", key=lambda s: s.map(_competencia_key))
            x = pd.to_datetime(g["competencia"].astype(str), format="%m/%Y", errors="coerce"); mask = ~x.isna()
            self.chart_mix.ax.plot(x[mask], g["Valor Convenio"][mask], marker='o')
            self.chart_mix.ax.set_title(f"{conv_sel} – {('Receita' if base=='receita' else 'Volume')} por Competência")
            self.chart_mix.draw()
        else:
            if base == "receita":
                g = df.groupby("Procedimento").agg(Metrica=("Valor Convenio","sum"), Qtd=("Procedimento","count")).reset_index().sort_values("Metrica", ascending=False)
            else:
                g = df.groupby("Procedimento").size().reset_index(name="Metrica").sort_values("Metrica", ascending=False); g["Qtd"] = g["Metrica"]
            top = g.head(15)
            bars = self.chart_mix.ax.barh(top["Procedimento"][::-1], top["Metrica"][::-1])
            self.chart_mix.ax.set_xlabel(self._axis_label(base))
            for rect in bars:
                w = rect.get_width(); self.chart_mix.ax.text(w*1.01, rect.get_y()+rect.get_height()/2, self._fmt_value_by_base(w, base), va='center', fontsize=8)
            self.chart_mix.draw()

        gtab = (df.groupby("Procedimento").agg(Metrica=("Valor Convenio","sum"), Qtd=("Procedimento","count")).reset_index() if base=='receita' else df.groupby("Procedimento").size().reset_index(name="Metrica"))
        if base != 'receita': gtab["Qtd"] = gtab["Metrica"]
        gtab = gtab.sort_values("Metrica", ascending=False)
        for _, row in gtab.iterrows():
            r = self.tbl_proc.rowCount(); self.tbl_proc.insertRow(r)
            self.tbl_proc.setItem(r, 0, QTableWidgetItem(str(row["Procedimento"])));
            self.tbl_proc.setItem(r, 1, QTableWidgetItem(fmt_money(row["Metrica"]) if base=='receita' else str(int(row["Metrica"]))));
            self.tbl_proc.setItem(r, 2, QTableWidgetItem(str(int(row.get("Qtd", row["Metrica"])))))
        self.tbl_proc.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)

    # -------------- Regiões --------------
    def _montar_regioes(self):
        root = QVBoxLayout(self.tab_reg); root.setContentsMargins(20, 20, 20, 20); root.setSpacing(18)
        filtros = self._make_filtros_bar(self.tab_reg); root.addLayout(filtros)

        # Controles específicos
        geo = QHBoxLayout()
        geo.addWidget(QLabel("Nível:"))
        self.combo_geo = QComboBox();
        # Opções conforme colunas disponíveis (agora podem vir da tabela de convênios)
        has_cidade = "Cidade" in self._df.columns and self._df["Cidade"].notna().any()
        has_uf = "UF" in self._df.columns and self._df["UF"].notna().any()
        if has_cidade: self.combo_geo.addItem("Cidade")
        if has_uf: self.combo_geo.addItem("UF")
        if has_uf: self.combo_geo.addItem("Região (macro)")
        if self.combo_geo.count() == 0:
            self.combo_geo.addItem("(Sem dados geográficos)")
        geo.addWidget(self.combo_geo)
        geo.addSpacing(12)
        geo.addWidget(QLabel("Top N:"))
        self.spin_geo_topn = QSpinBox(); self.spin_geo_topn.setRange(3, 50); self.spin_geo_topn.setValue(10)
        geo.addWidget(self.spin_geo_topn)
        geo.addSpacing(12)
        self.chk_geo_outros = QCheckBox("Agrupar restantes em 'Outros'"); self.chk_geo_outros.setChecked(True)
        geo.addWidget(self.chk_geo_outros)
        geo.addStretch(1)
        self.combo_geo.currentIndexChanged.connect(self._update_regioes)
        self.spin_geo_topn.valueChanged.connect(self._update_regioes)
        self.chk_geo_outros.toggled.connect(self._update_regioes)
        root.addLayout(geo)

        self.chart_regioes = Chart("Regiões – Receita / Volume"); root.addWidget(self.chart_regioes)

        self.tbl_reg = QTableWidget(); self.tbl_reg.setColumnCount(4)
        self.tbl_reg.setHorizontalHeaderLabels(["Região", "Métrica", "% do Total", "# Exames"])
        self.tbl_reg.verticalHeader().setVisible(False)
        self.tbl_reg.setEditTriggers(QAbstractItemView.NoEditTriggers)
        self.tbl_reg.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.tbl_reg.setSelectionMode(QAbstractItemView.SingleSelection)
        self.tbl_reg.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        root.addWidget(self.tbl_reg)

        # Comparativo (usar o mesmo padrão para dar contexto)
        self.comp_reg_box, self.comp_reg_refs = self._make_comp_box(); root.addWidget(self.comp_reg_box)

        self._update_regioes(); self._update_comparativos(self.comp_reg_refs)

    def _update_regioes(self):
        df = self._filtrar_df(); self.chart_regioes.clear(); self.tbl_reg.setRowCount(0)
        base = self._base_kind(); nivel = self.combo_geo.currentText() if hasattr(self, 'combo_geo') else "Cidade"

        if df.empty:
            self.chart_regioes.ax.text(0.5, 0.5, "Sem dados", ha='center', va='center'); self.chart_regioes.draw(); return

        # Construção da dimensão de agrupamento
        if nivel.startswith("Região"):
            if "UF" not in df.columns:
                self.chart_regioes.ax.text(0.5, 0.5, "Sem coluna UF para calcular Região", ha='center', va='center'); self.chart_regioes.draw(); return
            dim_col = "__Regiao__"
            df = df.copy(); df[dim_col] = df["UF"].map(BR_UF_TO_REGIAO).fillna("Outros")
        elif nivel == "UF":
            if "UF" not in df.columns:
                self.chart_regioes.ax.text(0.5, 0.5, "Sem coluna UF", ha='center', va='center'); self.chart_regioes.draw(); return
            dim_col = "UF"
        else:
            if "Cidade" not in df.columns:
                self.chart_regioes.ax.text(0.5, 0.5, "Sem coluna Cidade", ha='center', va='center'); self.chart_regioes.draw(); return
            dim_col = "Cidade"

        # Métrica
        if base == "receita":
            g = df.groupby(dim_col)["Valor Convenio"].sum().reset_index(name="Metrica")
        else:
            g = df.groupby(dim_col).size().reset_index(name="Metrica")
        g = g.sort_values("Metrica", ascending=False)

        total = float(g["Metrica"].sum()) or 1.0
        topn = self.spin_geo_topn.value() if hasattr(self, 'spin_geo_topn') else 15
        if len(g) > topn and self.chk_geo_outros.isChecked():
            top = g.iloc[:topn].copy()
            outros_sum = g.iloc[topn:]["Metrica"].sum()
            if outros_sum > 0:
                top = pd.concat([top, pd.DataFrame({dim_col: ["Outros"], "Metrica": [outros_sum]})], ignore_index=True)
        else:
            top = g.iloc[:topn].copy()

        # Gráfico: barras horizontais
        y = top[dim_col].iloc[::-1]; xvals = top["Metrica"].iloc[::-1]
        bars = self.chart_regioes.ax.barh(y, xvals)
        self.chart_regioes.ax.set_xlabel(self._axis_label(base)); self.chart_regioes.ax.set_ylabel("")
        self.chart_regioes.ax.yaxis.set_tick_params(labelsize=8)
        if base == "receita":
            self.chart_regioes.ax.xaxis.set_major_formatter(FuncFormatter(lambda x, p: (f"R$ {x:,.0f}".replace(",","X").replace(".",",").replace("X","."))))
        for rect in bars:
            w = rect.get_width(); self.chart_regioes.ax.text(w*1.01, rect.get_y()+rect.get_height()/2, self._fmt_value_by_base(w, base), va='center', fontsize=8)
        self.chart_regioes.ax.grid(True, axis='y', linestyle='--', alpha=0.3)
        self.chart_regioes.ax.set_title(f"{('Receita' if base=='receita' else 'Volume')} por {nivel}")
        self.chart_regioes.draw()

        # Tabela
        self.tbl_reg.setHorizontalHeaderLabels([nivel, ("Receita" if base=='receita' else "Qtde Exames"), "% do Total", "# Exames"])
        # Para a última coluna, sempre mostramos volume
        vol = df.groupby(dim_col).size().reset_index(name="Qtd")
        merged = g.merge(vol, on=dim_col, how='left')
        for _, row in merged.iterrows():
            r = self.tbl_reg.rowCount(); self.tbl_reg.insertRow(r)
            self.tbl_reg.setItem(r, 0, QTableWidgetItem(str(row[dim_col])))
            self.tbl_reg.setItem(r, 1, QTableWidgetItem(fmt_money(row["Metrica"]) if base=='receita' else str(int(row["Metrica"])) ))
            self.tbl_reg.setItem(r, 2, QTableWidgetItem(f"{row['Metrica']/total*100:.1f}%"))
            self.tbl_reg.setItem(r, 3, QTableWidgetItem(str(int(row.get("Qtd", 0)))))
        self.tbl_reg.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)

        if hasattr(self, "comp_reg_refs"): self._update_comparativos(self.comp_reg_refs)

    # -------------- Previsões --------------
    def _montar_previsoes(self):
        root = QVBoxLayout(self.tab_prev); root.setContentsMargins(20, 20, 20, 20); root.setSpacing(18)
        filtros = self._make_filtros_bar(self.tab_prev); root.addLayout(filtros)
        self.chart_prev = Chart("Tendência (Série Mensal)"); root.addWidget(self.chart_prev)
        self._update_previsoes()

    def _update_previsoes(self):
        comp_sel, conv_sel, med_sel = self._current_ctrls(); base = self._base_kind()
        df = self._df_no_comp(conv_sel, med_sel)
        self.chart_prev.clear()
        if df.empty:
            self.chart_prev.ax.text(0.5, 0.5, "Sem dados", ha='center', va='center'); self.chart_prev.draw(); return
        if base == "receita":
            g = df.groupby("competencia")["Valor Convenio"].sum().reset_index()
        else:
            g = df.groupby("competencia").size().reset_index(name="Valor Convenio")
        g = g.sort_values("competencia", key=lambda s: s.map(_competencia_key))
        x = pd.to_datetime(g["competencia"].astype(str), format="%m/%Y", errors="coerce"); mask = ~x.isna()
        self.chart_prev.ax.plot(x[mask], g["Valor Convenio"][mask], marker='o')
        self.chart_prev.ax.set_ylabel(self._axis_label(base))
        self.chart_prev.ax.xaxis.set_major_locator(mdates.MonthLocator(interval=1))
        self.chart_prev.ax.xaxis.set_major_formatter(mdates.DateFormatter("%m/%Y"))
        self.chart_prev.draw()

    # -------------- utilidades --------------
    def _refresh_all(self):
        tab = self.currentWidget()
        if tab is self.tab_overview:
            self._update_overview(); self._update_comparativos(self.comp_overview_refs)
        elif tab is self.tab_convenio:
            self._update_convenios(); self._update_comparativos(self.comp_convenios_refs)
        elif tab is self.tab_medicos:
            self._update_medicos(); self._update_comparativos(self.comp_medicos_refs)
        elif tab is self.tab_proc:
            self._update_procedimentos(); self._update_comparativos(self.comp_proc_refs)
        elif tab is self.tab_reg:
            self._update_regioes(); self._update_comparativos(self.comp_reg_refs)
        elif tab is self.tab_prev:
            self._update_previsoes()
        elif tab is self.tab_empresas:
            self._update_empresas()
            self._update_comparativos(self.comp_emp_refs)


    def _exportar_chart(self, chart: Chart):
        path, _ = QFileDialog.getSaveFileName(self, "Salvar gráfico", "grafico.png", "PNG (*.png)")
        if not path:
            return
        try:
            chart.fig.savefig(path, dpi=150, bbox_inches='tight')
        except Exception as e:
            QMessageBox.warning(self, "Erro ao salvar", str(e))

    def _exportar_excel_overview(self):
        df = self._filtrar_df()
        if df.empty:
            QMessageBox.information(self, "Exportar", "Não há dados para exportar no filtro atual."); return
        path, _ = QFileDialog.getSaveFileName(self, "Exportar Excel", "dados_overview.xlsx", "Excel (*.xlsx)")
        if not path:
            return
        try:
            df.to_excel(path, index=False)
        except Exception as e:
            QMessageBox.warning(self, "Erro ao exportar", str(e))

    def _aplicar_estilo(self):
        self.setStyleSheet("""
            QTabWidget::pane { border: 1px solid #c7d7e5; border-radius: 8px; padding: 0; background: #f7f9fc; }
            QTabBar::tab { background: #e6edf5; color: #0f172a; border: 1px solid #d7e1ee; border-bottom: none; border-top-left-radius: 8px; border-top-right-radius: 8px; padding: 6px 12px; height: 30px; font-weight: 600; }
            QTabBar::tab:selected { background: #0ea5e9; color: #ffffff; border: 1px solid #0284c7; }
        """)
