# modulo3_cadastro_convenio.py

from PyQt5.QtWidgets import (
    QWidget, QVBoxLayout, QPushButton, QHBoxLayout,
    QTableWidget, QTableWidgetItem, QMessageBox, QDialog,
    QFormLayout, QLineEdit, QComboBox, QLabel, QFileDialog,
    QDialogButtonBox
)
from PyQt5.QtCore import Qt
import sqlite3
import pandas as pd
import requests
from PyQt5.QtWidgets import QHeaderView
from PIL import Image
import unicodedata
import unicodedata, re
import os
import tempfile
import unicodedata
import re
import difflib
import logging
from datetime import date
from pathlib import Path
import pdfkit
import shutil, platform
from pyhanko.sign import signers, fields
from pyhanko.pdf_utils.incremental_writer import IncrementalPdfFileWriter
from pyhanko.sign.fields import SigFieldSpec, append_signature_field
from pyhanko.pdf_utils.incremental_writer import IncrementalPdfFileWriter
from pyhanko.sign import signers
from pyhanko.sign.fields import SigFieldSpec
import traceback
# --- Assinatura digital (pyHanko) ---
try:
    from pyhanko.sign import signers, fields
    from pyhanko.pdf_utils.incremental_writer import IncrementalPdfFileWriter
    _PYHANKO_OK = True
except Exception:
    _PYHANKO_OK = False

CERT_P12_PATH = Path(r"D:/MEDICAL LAUDOS/CERTIFICADOS/medical_laudos.pfx")



# --- Helpers ---------------------------------------------------------------


# helper: assina o PDF com certificado A1 (.pfx/.p12)
# --- Catálogo de procedimentos (valor padrão) e médicos por especialidade ---
PROC_CATALOG = {
    "Laudos Oftalmológicos": [
        ("Acuidade Visual", 4.90),
        ("Avaliação Oftalmológica", 4.90),
        ("Teste de Ishihara", 4.90),
        ("Campimetria", 4.90),
    ],
    "Laudos Cardiológicos": [
        ("ECG: Eletrocardiograma", 4.90),
        ("Holter 24 horas", 15.00),
        ("Mapa 24 horas", 15.00),
    ],
    "Laudos Neurológicos": [
        ("EEG: Eletroencefalograma", 4.90),
        ("EEG - Eletroencefalograma Clínico", 4.90),
    ],
    "Laudos Pneumológicos": [
        ("Espirometria", 4.90),
        ("Espirometria com prova broncodilatadora", 4.90),
    ],
    "Laudos Radiológicos": [
        ("Raio-X diversos", 5.90),
        ("Mamografia", 15.00),
    ],
    "Laudos OIT": [
        ("Raio-X Tórax no padrão OIT 1 Assinatura", 5.90),
        ("Raio-X Tórax no padrão OIT 2 Assinaturas", 8.00),
    ],
    "Outros Laudos": [
        ("Risco Cirúrgico", 40.00),
    ],
}

DOCTORS_BY_CATEGORY = {
    "Laudos Oftalmológicos": [
        "Dr. Ariel Tavares Alves - CRM: 204116-SP - RQE Nº: 80120 - Oftalmologista",
    ],
    "Laudos Cardiológicos": [
        "Dr. Henrique Typaldo Caritatos Fontenelle - CRM: 1039261-RJ - RQE Nº: 34756 - Cardiologista",
        "Dra. Stella Mariana Ferreira Giolo - CRM: 140546-SP - RQE Nº: 89343 - Cardiologista",
        "Dra. Rita de Cássia Fialho Alvim Sadra - CRM: 23833-MG - RQE Nº: 23583 - Cardiologista",
    ],
    "Laudos Neurológicos": [
        "Dra. Cristiane Fiquene Conti - CRM: 119265-SP - RQE Nº: 74269 - Neurologista",
    ],
    "Laudos Pneumológicos": [
        "Dra. Sarah Veiga Medrado - CRM: 78528-MG - RQE Nº 57044 - Pneumologista",
        "Dr. Sergio Renato Nahid - CRM: 37928-MG - RQE Nº: 11763 - Pneumologista",
    ],
    "Laudos Radiológicos": [
        "Dr. Henrique Trigo bianchessi - CRM: 95422-SP - RQE Nº: 26347 - Radiologista",
        "Dra. Susana Trigo Bianchessi - CRM: 97179-SP - RQE Nº: 23856 - Radiologista",
        "Dr. Carlos Eduardo Passos - CRM: 87895-SP - RQE Nº: 52700 - Radiologista/Leiturista OIT",
        "Dr. Ronaldo Lo Russo Zupo - CRM: 13252-MG - RQE Nº: 555 - Radiologista/Leiturista OIT",
    ],
    "Laudos OIT": [
        "Dr. Henrique Trigo bianchessi - CRM: 95422-SP - RQE Nº: 26347 - Radiologista",
        "Dra. Susana Trigo Bianchessi - CRM: 97179-SP - RQE Nº: 23856 - Radiologista",
        "Dr. Carlos Eduardo Passos - CRM: 87895-SP - RQE Nº: 52700 - Radiologista/Leiturista OIT",
        "Dr. Ronaldo Lo Russo Zupo - CRM: 13252-MG - RQE Nº: 555 - Radiologista/Leiturista OIT",
        "Dr. Sergio Renato Nahid - CRM: 37928-MG - RQE Nº: 11763 - Pneumologista / Leitor OIT",
    ],
    "Outros Laudos": [
        "Dr. Henrique Typaldo Caritatos Fontenelle - CRM: 1039261-RJ - RQE Nº: 34756 - Cardiologista",
    ],
}

# mapa rápido procedimento -> categoria
PROC_TO_CATEGORY = {
    proc: cat
    for cat, items in PROC_CATALOG.items()
    for (proc, _price) in items
}

class DialogProcedimentosContrato(QDialog):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Procedimentos do Contrato")
        self.setMinimumWidth(560)

        layout = QVBoxLayout(self)

        # tabela com 2 colunas
        self.table = QTableWidget(0, 2)
        self.table.setHorizontalHeaderLabels(["Procedimento", "Valor (R$)"])
        self.table.horizontalHeader().setSectionResizeMode(0, QHeaderView.Stretch)
        self.table.horizontalHeader().setSectionResizeMode(1, QHeaderView.ResizeToContents)
        self.table.verticalHeader().setVisible(False)
        layout.addWidget(self.table)

        btns_line = QHBoxLayout()
        self.btn_add = QPushButton("+ Adicionar Procedimento")
        self.btn_rem = QPushButton("Remover selecionado")
        btns_line.addWidget(self.btn_add)
        btns_line.addWidget(self.btn_rem)
        btns_line.addStretch()
        layout.addLayout(btns_line)

        self.btn_add.clicked.connect(self._add_row)
        self.btn_rem.clicked.connect(self._remove_selected)

        # botões OK/Cancelar
        self.buttons = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        self.buttons.accepted.connect(self._validate_and_accept)
        self.buttons.rejected.connect(self.reject)
        layout.addWidget(self.buttons)

        # adiciona pelo menos 1 linha
        self._add_row()

    def _add_row(self):
        row = self.table.rowCount()
        self.table.insertRow(row)

        # combobox com todos os procedimentos
        cb = QComboBox()
        all_procs = []
        for cat, items in PROC_CATALOG.items():
            for proc, price in items:
                all_procs.append((f"{proc}  —  {cat}", proc, cat, price))
        # ordena por nome do procedimento
        all_procs.sort(key=lambda x: x[1].lower())

        for label, proc, cat, price in all_procs:
            cb.addItem(label, (proc, cat, price))

        cb.currentIndexChanged.connect(lambda _i, r=row, combo=cb: self._on_proc_change(r, combo))
        self.table.setCellWidget(row, 0, cb)

        # campo valor
        val = QLineEdit()
        val.setPlaceholderText("ex.: 4,90")
        val.setAlignment(Qt.AlignRight)
        self.table.setCellWidget(row, 1, val)

        # preencher valor padrão do primeiro item
        self._on_proc_change(row, cb)

    def _on_proc_change(self, row, combo):
        data = combo.currentData()
        if not data:
            return
        _proc, _cat, price = data
        self.table.cellWidget(row, 1).setText(self._fmt_brl(price))

    def _remove_selected(self):
        r = self.table.currentRow()
        if r >= 0:
            self.table.removeRow(r)

    def _fmt_brl(self, v):
        try:
            return f"R${float(v):.2f}".replace(".", ",")
        except Exception:
            return str(v)

    def _parse_brl_to_float(self, s: str) -> float:
        s = (s or "").strip()
        s = s.replace("R$", "").replace(" ", "")
        s = s.replace(".", "").replace(",", ".")
        try:
            return round(float(s), 2)
        except Exception:
            return 0.0

    def _validate_and_accept(self):
        itens = self.get_itens()
        if not itens:
            QMessageBox.warning(self, "Atenção", "Adicione ao menos um procedimento.")
            return
        self.accept()

    def get_itens(self):
        """Retorna lista de dicts: [{'procedimento','categoria','valor_float','valor_str'}]"""
        itens = []
        for r in range(self.table.rowCount()):
            combo = self.table.cellWidget(r, 0)
            val_edit = self.table.cellWidget(r, 1)
            if not combo or not val_edit:
                continue
            data = combo.currentData()
            if not data:
                continue
            proc, cat, _price = data
            v_str = val_edit.text().strip()
            v_float = self._parse_brl_to_float(v_str) if v_str else float(_price)
            itens.append({
                "procedimento": proc,
                "categoria": cat,
                "valor_float": v_float,
                "valor_str": self._fmt_brl(v_float),
            })

        # dedup por procedimento (mantém o último)
        dedup = {}
        for item in itens:
            dedup[item["procedimento"]] = item
        return list(dedup.values())


def _normalize_nome(s: str) -> str:
    if not s:
        return ""
    s = unicodedata.normalize("NFKD", s)
    s = "".join(c for c in s if not unicodedata.combining(c))
    s = re.sub(r"[^a-z0-9 ]+", " ", s.lower())
    s = re.sub(r"\s+", " ", s).strip()
    return s

def obter_codigo_ibge_por_cidade_uf(cidade: str, uf: str) -> str | None:
    if not cidade or not uf:
        return None
    try:
        uf = uf.strip().upper()
        r = requests.get(
            f"https://servicodados.ibge.gov.br/api/v1/localidades/estados/{uf}/municipios",
            timeout=10
        )
        r.raise_for_status()
        municipios = r.json()  # [{'id':3303500,'nome':'Nova Iguaçu',...}, ...]

        alvo = _normalize_nome(cidade)
        # 1) match exato (normalizado)
        for m in municipios:
            if _normalize_nome(m["nome"]) == alvo:
                return str(m["id"]).zfill(7)
        # 2) fuzzy simples
        nomes = [m["nome"] for m in municipios]
        for cutoff in (0.9, 0.8):
            cand = difflib.get_close_matches(cidade, nomes, n=1, cutoff=cutoff)
            if cand:
                m = next(x for x in municipios if x["nome"] == cand[0])
                return str(m["id"]).zfill(7)
    except Exception as e:
        logging.exception("Falha ao obter codigo IBGE: %s", e)
    return None

def _autopreencher_ibge(self):
    cidade = (self.cidade.text() or "").strip()
    uf = (self.uf.text() or "").strip()
    if not cidade or not uf:
        return
    ibge = obter_codigo_ibge_por_cidade_uf(cidade, uf)
    if ibge:
        self.cod_ibge.setText(ibge)   # <-- aqui

def tratar_duplicados(df):
    

    def normalizar_medico(nome):
        if pd.isna(nome):
            return ''
        nome = str(nome).strip().lower()
        nome = unicodedata.normalize('NFKD', nome).encode('ASCII', 'ignore').decode('utf-8')
        return nome

    df["Duplicado"] = False
    df["NaoDuplicadoOIT"] = False

    if df.empty or not all(c in df.columns for c in ["Nome", "Empresa", "Convenio", "Tipo de Exame", "Procedimento", "Data Exame", "Médico"]):
        return df

    # Separa registros OIT
    oit_df = df[df["Procedimento"].str.upper().str.contains("RX - TÓRAX OIT - 2 ASSINATURAS", na=False)].copy()
    restantes_df = df[~df.index.isin(oit_df.index)].copy()

    # 🔒 Blindagem: padroniza Empresa
    for df_padronizar in [oit_df, restantes_df]:
        if "Empresa" in df_padronizar.columns:
            df_padronizar["Empresa"] = df_padronizar["Empresa"].fillna("").astype(str).str.strip()

    colunas_chave = ["Nome", "Empresa", "Convenio", "Tipo de Exame", "Procedimento", "Data Exame"]

    # 🟢 Lógica OIT
    grupo_oit = oit_df.groupby(colunas_chave)
    for _, grupo in grupo_oit:
        grupo = grupo.copy()
        indices = grupo.index.tolist()
        medicos_norm = grupo["Médico"].apply(normalizar_medico)
        medicos_unicos = medicos_norm.unique()

        if len(grupo) == 2 and len(medicos_unicos) == 2:
            oit_df.loc[indices, "NaoDuplicadoOIT"] = True
        else:
            encontrados = set()
            validos = []
            for idx, nome in zip(grupo.index, medicos_norm):
                if nome not in encontrados and len(validos) < 2:
                    validos.append(idx)
                    encontrados.add(nome)
            for idx in validos:
                oit_df.loc[idx, "NaoDuplicadoOIT"] = True
            duplicados = grupo.index.difference(validos)
            oit_df.loc[duplicados, "Duplicado"] = True

    # 🟡 Lógica geral de duplicados
    if all(c in restantes_df.columns for c in colunas_chave):
        duplicatas = restantes_df.duplicated(subset=colunas_chave, keep='first')
        restantes_df.loc[duplicatas, "Duplicado"] = True

    df_final = pd.concat([oit_df, restantes_df]).sort_index()

    # 🔁 Zera valores apenas dos duplicados
    if "Valor Convenio" in df_final.columns:
        df_final.loc[df_final["Duplicado"] == True, "Valor Convenio"] = 0

    return df_final




CAMINHO_BANCO = "db/sistema_financeiro.db"

class DialogSelecionarCompetencia(QDialog):
    def __init__(self):
        super().__init__()
        self.setWindowTitle("Selecionar Competência")
        self.setFixedSize(300, 150)

        layout = QFormLayout(self)
        self.combo_mes = QComboBox()
        self.combo_ano = QComboBox()

        self.combo_mes.addItems(["01", "02", "03", "04", "05", "06",
                                 "07", "08", "09", "10", "11", "12"])
        self.combo_ano.addItems(["2025", "2026", "2027", 
                                 "2028", "2029", "2030", "2031", "2032", "2033"])

        layout.addRow("Mês:", self.combo_mes)
        layout.addRow("Ano:", self.combo_ano)

        botoes = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        botoes.accepted.connect(self.accept)
        botoes.rejected.connect(self.reject)
        layout.addWidget(botoes)

    def get_competencia(self):
        return f"{self.combo_mes.currentText()}/{self.combo_ano.currentText()}"

class DialogNovoConvenio(QDialog):
    def __init__(self, parent=None, dados_existentes=None):
        super().__init__(parent)
        self.setWindowTitle("Convênio")
        self.setMinimumWidth(400)

        self.layout = QFormLayout(self)
        self.nome_combo = QComboBox()
        self.nome_combo.setEditable(True)

        try:
            conn = sqlite3.connect(CAMINHO_BANCO)
            cursor = conn.cursor()
            cursor.execute("SELECT DISTINCT Convenio FROM registros_financeiros ORDER BY Convenio")
            nomes = [row[0] for row in cursor.fetchall() if row[0]]
            self.nome_combo.addItems(nomes)
            conn.close()
        except Exception as e:
            print(f"Erro ao buscar convênios: {e}")

        self.razao = QLineEdit()
        self.cnpj = QLineEdit()
        self.cnpj.editingFinished.connect(self.buscar_dados_cnpj)
        self.cep = QLineEdit()
        self.cep.editingFinished.connect(self.buscar_cep)
        self.logradouro = QLineEdit()
        self.numero = QLineEdit()
        self.complemento = QLineEdit()
        self.bairro = QLineEdit()
        self.cidade = QLineEdit()
        self.uf = QLineEdit()
        self.email = QLineEdit()
        self.telefone = QLineEdit()
        self.ie = QLineEdit()
        self.im = QLineEdit()        
        self.obs = QLineEdit()
        self.dia_vencimento = QLineEdit()
        self.cod_ibge = QLineEdit()
        

        self.layout.addRow("Nome do Convênio:", self.nome_combo)
        self.layout.addRow("Razão Social:", self.razao)
        self.layout.addRow("CNPJ:", self.cnpj)
        self.layout.addRow("CEP:", self.cep)
        self.layout.addRow("Logradouro:", self.logradouro)
        self.layout.addRow("Número:", self.numero)
        self.layout.addRow("Complemento:", self.complemento)
        self.layout.addRow("Bairro:", self.bairro)
        self.layout.addRow("Cidade:", self.cidade)
        self.layout.addRow("UF:", self.uf)
        self.layout.addRow("E-mail:", self.email)
        self.layout.addRow("Telefone:", self.telefone)
        self.layout.addRow("Inscrição Estadual:", self.ie)
        self.layout.addRow("Inscrição Municipal:", self.im)        
        self.layout.addRow("Observações:", self.obs)
        self.layout.addRow("Dia de Vencimento:", self.dia_vencimento)
        
        # depois de self.im / self.obs / self.dia_vencimento...
        self.cod_ibge = QLineEdit()
        self.cod_ibge.setPlaceholderText("ex.: 3303500")
        self.layout.addRow("Código IBGE:", self.cod_ibge)


        self.botao_salvar = QPushButton("Salvar Informações do Convênio")
        self.botao_salvar.clicked.connect(self.salvar_convenio)
        self.layout.addRow(self.botao_salvar)

        self.dados_existentes = dados_existentes
        if dados_existentes:
            self.carregar_dados(dados_existentes)

    def carregar_dados(self, dados):
        self.nome_combo.setCurrentText(dados[1])
        self.razao.setText(dados[2])
        self.cnpj.setText(dados[3])
        self.cep.setText(dados[4])
        self.logradouro.setText(dados[5])
        self.numero.setText(dados[6])
        self.complemento.setText(dados[7])
        self.bairro.setText(dados[8])
        self.cidade.setText(dados[9])
        self.uf.setText(dados[10])
        self.email.setText(dados[11])
        self.telefone.setText(dados[12])
        self.ie.setText(dados[13])
        self.im.setText(dados[14])        
        self.obs.setText(dados[15])
        self.dia_vencimento.setText(str(dados[16]))
        # ... depois de self.dia_vencimento.setText(...)
        # supondo que a coluna nova ficou no final da tabela (índice 17)
        self.cod_ibge.setText(str(dados[17]) if len(dados) > 17 and dados[17] is not None else "")

        
        self.id_editar = dados[0]

    def buscar_cep(self):
        cep = self.cep.text().strip().replace("-", "")
        if not cep or len(cep) != 8:
            return
        try:
            r = requests.get(f"https://viacep.com.br/ws/{cep}/json/", timeout=10)
            if r.status_code == 200:
                dados = r.json()
                self.logradouro.setText(dados.get("logradouro", ""))
                self.bairro.setText(dados.get("bairro", ""))
                self.cidade.setText(dados.get("localidade", ""))
                self.uf.setText(dados.get("uf", ""))
                ibge = (dados.get("ibge") or "").strip()
                if ibge and re.fullmatch(r"\d{7}", ibge):
                    self.cod_ibge.setText(ibge)   # <-- aqui
                else:
                    _autopreencher_ibge(self)
        except Exception:
            pass

    def buscar_dados_cnpj(self):
        cnpj = ''.join(filter(str.isdigit, self.cnpj.text().strip()))
        if len(cnpj) != 14:
            return
        try:
            r = requests.get(
                f"https://www.receitaws.com.br/v1/cnpj/{cnpj}",
                headers={"User-Agent": "Mozilla/5.0"},
                timeout=15
            )
            if r.status_code == 200:
                dados = r.json()
                if dados.get("status") == "ERROR":
                    QMessageBox.warning(self, "CNPJ não encontrado", dados.get("message", "Erro desconhecido."))
                    return
                self.razao.setText(dados.get("nome", ""))
                self.cep.setText((dados.get("cep") or "").replace(".", "").replace("-", ""))
                self.logradouro.setText(dados.get("logradouro", ""))
                self.numero.setText(dados.get("numero", ""))
                self.complemento.setText(dados.get("complemento", ""))
                self.bairro.setText(dados.get("bairro", ""))
                self.cidade.setText(dados.get("municipio", ""))
                self.uf.setText(dados.get("uf", ""))
                self.email.setText(dados.get("email", ""))
                self.telefone.setText(dados.get("telefone", ""))
                _autopreencher_ibge(self)  # preenche self.codigo_municipio_ibge
            else:
                QMessageBox.warning(self, "Erro", f"Erro ao consultar CNPJ: código {r.status_code}")
        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao buscar dados do CNPJ:\n{e}")


    def salvar_convenio(self):
        try:
            dados = (
                self.nome_combo.currentText(), self.razao.text(), self.cnpj.text(), self.cep.text(),
                self.logradouro.text(), self.numero.text(), self.complemento.text(),
                self.bairro.text(), self.cidade.text(), self.uf.text(),
                self.email.text(), self.telefone.text(), self.ie.text(),
                self.im.text(), self.obs.text(), self.dia_vencimento.text(),
                self.cod_ibge.text().strip()
            )
            conn = sqlite3.connect(CAMINHO_BANCO)
            cursor = conn.cursor()

            if hasattr(self, 'id_editar'):
                cursor.execute("""
                    UPDATE convenios SET
                        nome=?, razao_social=?, cnpj=?, cep=?, logradouro=?, numero=?, complemento=?,
                        bairro=?, cidade=?, uf=?, email=?, telefone=?, inscricao_estadual=?,
                        inscricao_municipal=?, observacoes=?, dia_vencimento=?, 
                        codigo_municipio_ibge=?
                    WHERE id=?
                """, dados + (self.id_editar,))

            else:
                cursor.execute("""
                INSERT INTO convenios (
                    nome, razao_social, cnpj, cep, logradouro, numero, complemento,
                    bairro, cidade, uf, email, telefone, inscricao_estadual, inscricao_municipal, 
                    observacoes, dia_vencimento, codigo_municipio_ibge
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            """, dados)


            conn.commit()
            conn.close()
            QMessageBox.information(self, "Sucesso", "Convênio salvo com sucesso.")
            self.accept()
        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao salvar convênio:\n{e}")


class ModuloCadastroConvenio(QWidget):   
    

    def _assinar_pdf(self, pdf_path: str | Path, empresa: dict) -> str | None:
        if not _PYHANKO_OK:
            QMessageBox.warning(self, "Assinatura digital", "pyHanko não disponível.")
            return None

        from pyhanko.sign import signers, fields
        from pyhanko.pdf_utils.incremental_writer import IncrementalPdfFileWriter
        try:
            from pyhanko.pdf_utils.annotations import BorderStyle
        except Exception:
            BorderStyle = None
        import traceback

        ANCHOR_URI = "https://sig-anchor.local/assin-left"   # deve bater com o href do HTML

        cert_path = CERT_P12_PATH
        if not cert_path.exists():
            QMessageBox.warning(self, "Assinatura digital", f"Certificado não encontrado:\n{cert_path}")
            return None

        senha = self._get_cert_password()
        if not senha:
            QMessageBox.critical(self, "Assinatura digital", "Senha do certificado não encontrada.")
            return None
        if isinstance(senha, str):
            senha = senha.encode("utf-8")

        pdf_in  = Path(pdf_path)
        pdf_out = pdf_in.with_name(pdf_in.stem + "_assinado.pdf")

        # ---- fallback em pontos (caso não ache o anchor) ----
        def mm(v): return v * 72.0 / 25.4
        LEFT_MRG, RIGHT_MRG, BOTTOM_MRG = mm(12), mm(12), mm(22)
        A4_W_PT = 595.28
        content_w = A4_W_PT - LEFT_MRG - RIGHT_MRG
        left_col_w = content_w / 2.0
        PAD_MM, LINE_BOTTOM_GAP_MM, SPACE_TOP_MM, SPACE_HEIGHT_MM = 12, 4, 8, 32
        x1_fb = LEFT_MRG + mm(PAD_MM)
        x2_fb = LEFT_MRG + left_col_w - mm(PAD_MM)
        y1_fb = BOTTOM_MRG + mm(LINE_BOTTOM_GAP_MM + SPACE_TOP_MM)
        y2_fb = y1_fb + mm(SPACE_HEIGHT_MM)

        # ---- tenta achar o anchor no PDF (retorna página e retângulo) ----
        def _find_anchor_rect(path: Path, uri: str):
            try:
                import PyPDF2
                r = PyPDF2.PdfReader(str(path))
                for page_ix, page in enumerate(r.pages):
                    annots = page.get("/Annots") or []
                    for ref in annots:
                        annot = ref.get_object()
                        a = annot.get("/A")
                        if a and a.get("/URI") == uri:
                            llx, lly, urx, ury = [float(v) for v in annot["/Rect"]]
                            x1, y1 = min(llx, urx), min(lly, ury)
                            x2, y2 = max(llx, urx), max(lly, ury)
                            return page_ix, (x1, y1, x2, y2)
            except Exception:
                pass
            return None, None

        try:
            with open(pdf_in, "rb") as inf:
                w = IncrementalPdfFileWriter(inf)

                page_ix, anchor_rect = _find_anchor_rect(pdf_in, ANCHOR_URI)
                print("ANCHOR:", page_ix, anchor_rect)  # debug

                if anchor_rect:
                    x1, y1, x2, y2 = anchor_rect
                    on_page = page_ix            # página onde o anchor está
                else:
                    x1, y1, x2, y2 = x1_fb, y1_fb, x2_fb, y2_fb
                    on_page = -1                  # fallback: última página

                sig_name = "Assinatura_MedicalLaudos"
                spec_kwargs = dict(on_page=on_page, box=(x1, y1, x2, y2))
                if BorderStyle:
                    spec_kwargs["widget_annot_kwargs"] = {"border_style": BorderStyle(width=0)}  # sem borda

                try:
                    spec = fields.SigFieldSpec(sig_name, **spec_kwargs)
                except TypeError:
                    spec = fields.SigFieldSpec(sig_name, on_page=on_page, box=(x1, y1, x2, y2))

                fields.append_signature_field(w, spec)

                signer = signers.SimpleSigner.load_pkcs12(str(cert_path), passphrase=senha)
                meta = signers.PdfSignatureMetadata(
                    field_name=sig_name,
                    reason="Assinatura eletrônica do contrato",
                    location=f"{empresa.get('cidade','')}-{empresa.get('uf','')}".strip("-"),
                )

                pdf_signer = signers.PdfSigner(meta, signer)
                with open(pdf_out, "wb") as outf:
                    pdf_signer.sign_pdf(w, output=outf)

            return str(pdf_out)

        except Exception as e:
            traceback.print_exc()
            QMessageBox.critical(self, "Assinatura digital", f"Falha ao assinar PDF:\n{e}")
            return None






    
    def _get_cert_password(self) -> bytes | None:
        """Lê a senha A1 do BD (medical_laudos.senha_certificado) ou da env P12_PASSWORD.
        Garante retorno em bytes e detecta se por engano gravaram o PFX na coluna."""
        pw: bytes | None = None
        try:
            con = sqlite3.connect(CAMINHO_BANCO)
            cur = con.cursor()
            cur.execute("SELECT senha_certificado FROM medical_laudos LIMIT 1")
            row = cur.fetchone()
            con.close()
            if row is not None:
                val = row[0]
                if isinstance(val, (bytes, bytearray)):
                    pw = bytes(val)
                elif val is not None:
                    pw = str(val).strip().encode("utf-8")
        except Exception:
            pw = None

        if not pw:
            env = os.environ.get("P12_PASSWORD")
            if env:
                pw = env.encode("utf-8")

        # ⚠️ proteção: se isso aqui "parecer" um PFX (0x30 0x82 no começo), não é senha!
        if pw and len(pw) > 64 and pw[:2] in (b"\x30\x82", b"\x30\x83"):
            QMessageBox.critical(
                self, "Assinatura digital",
                "O campo medical_laudos.senha_certificado parece conter o ARQUIVO PFX "
                "(dados binários) em vez da SENHA. Grave apenas a senha (texto) nessa coluna."
            )
            return None
        return pw



    
    def _build_header_footer(self, empresa: dict):
        """Cria arquivos temporários HTML para header/footer e retorna (header_path, footer_path)."""
        

        logo_tag = self._logo_tag()  # já devolve <img src="file://..."> se existir

        header_html = f"""<!DOCTYPE html>
    <html><head><meta charset="utf-8">
    <style>
    html,body{{margin:0;padding:0;font-family:'Calibri','Carlito','Segoe UI',Arial,Helvetica,sans-serif;font-size:11pt;}}
    .hdr{{width:100%; border-bottom:1px solid #ddd; padding:6px 6mm; display:table; table-layout:fixed;}}
    .left{{display:table-cell; vertical-align:middle; width:70%;}}
    .right{{display:table-cell; vertical-align:middle; width:30%; text-align:right; white-space:nowrap;}}
    .brand{{font-weight:700; margin-left:8px; vertical-align:middle; display:inline-block;}}
    .logo img{{height:24px; vertical-align:middle;}}
    </style>
    </head><body>
    <div class="hdr">
        <div class="left">
        <span class="logo">{logo_tag}</span>        
        </div>        
    </div>
    </body></html>"""

        # Rodapé centralizado (endereço + paginação)
        rodape_texto = (
            f"Rua Itaguaí 746, Bairro Caiçaras – Belo Horizonte - MG, "
            f"CEP: {empresa.get('cep','')} - Telefone: {empresa.get('telefone','')} (WhatsApp) "
            f"E-mail: {empresa.get('email','')}"
        )

        footer_html = f"""<!DOCTYPE html>
    <html><head><meta charset="utf-8">
    <style>
    html,body{{margin:0;padding:0;font-family:'Calibri','Carlito','Segoe UI',Arial,Helvetica,sans-serif;font-size:11pt;}}
    .ftr{{width:100%; border-top:1px solid #ddd; padding:6px 6mm; text-align:center;}}
    .addr{{display:block;}}
    .pageno{{display:block; margin-top:2px;}}
    </style>
    </head><body>
    <div class="ftr">
        <span class="addr">{rodape_texto}</span>        
    </div>
    </body></html>"""

        fh = tempfile.NamedTemporaryFile(delete=False, suffix=".html")
        fh.write(header_html.encode("utf-8")); fh.close()
        ff = tempfile.NamedTemporaryFile(delete=False, suffix=".html")
        ff.write(footer_html.encode("utf-8")); ff.close()
        return fh.name, ff.name

    
    def _pdfkit_config(self):
        """
        Tenta localizar o wkhtmltopdf.exe no Windows (ou wkhtmltopdf em outros SOs)
        em locais comuns, PATH e variável de ambiente WKHTMLTOPDF_PATH.
        Retorna pdfkit.configuration(...) ou None.
        """
        # 1) PATH do sistema
        found = shutil.which("wkhtmltopdf")
        try_paths = [Path(found)] if found else []

        # 2) Variável de ambiente
        env = os.environ.get("WKHTMLTOPDF_PATH")
        if env:
            try_paths.append(Path(env))

        # 3) Locais padrão (Windows) + pasta do app (bin/)
        try_paths += [
            Path("bin") / "wkhtmltopdf.exe",
            Path("wkhtmltopdf.exe"),
            Path(r"C:\Program Files\wkhtmltopdf\bin\wkhtmltopdf.exe"),
            Path(r"C:\Program Files (x86)\wkhtmltopdf\bin\wkhtmltopdf.exe"),
        ]

        for p in try_paths:
            try:
                if p and p.exists():
                    return pdfkit.configuration(wkhtmltopdf=str(p))
            except Exception:
                pass
        return None

    
    def _logo_tag(self) -> str:
        try_paths = [
            Path("imagens") / "medical_laudos_logo.png",
            Path("icones") / "medical_laudos_logo.png",  # fallback opcional
        ]
        for p in try_paths:
            try:
                if p.exists():
                    # usa file:// absoluto (wkhtmltopdf carrega com enable-local-file-access)
                    return f'<img src="{p.resolve().as_uri()}" style="height:42px;">'
            except Exception:
                pass
        return ""

    
    
    def _fmt_brl(self, v) -> str:
        try:
            return f"R${float(v):.2f}".replace(".", ",")
        except Exception:
            return str(v)

    def _money_rows_from_itens(self, itens):
        """Converte itens [{procedimento, valor_str}] em linhas HTML."""
        return "".join(
            f"<tr><td>{i['procedimento']}</td><td style='text-align:right'>{i['valor_str']}</td></tr>"
            for i in itens
        )

    
    # -------------------------------
    # GERAR CONTRATO (PDF)
    # -------------------------------
    def gerar_contrato(self):
        row = self.tabela.currentRow()
        if row < 0:
            QMessageBox.warning(self, "Atenção", "Selecione um convênio na tabela.")
            return

        nome = self.tabela.item(row, 0).text()

        # Busca dados completos do convênio
        try:
            con = sqlite3.connect(CAMINHO_BANCO)
            cur = con.cursor()
            cur.execute("""
                SELECT id, nome, razao_social, cnpj, cep, logradouro, numero, complemento,
                    bairro, cidade, uf, email, telefone, inscricao_estadual, inscricao_municipal,
                    observacoes, dia_vencimento, COALESCE(codigo_municipio_ibge,'')
                FROM convenios
                WHERE nome = ?
            """, (nome,))
            r = cur.fetchone()
            con.close()
            if not r:
                QMessageBox.warning(self, "Aviso", "Convênio não encontrado.")
                return
        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Falha ao consultar convênio:\n{e}")
            return

        (_, nome_c, razao, cnpj, cep, logradouro, numero, compl,
        bairro, cidade, uf, email, telefone, ie, im, obs, dia_venc, cod_ibge) = r

        conv = {
            "nome": nome_c or "",
            "razao_social": razao or "",
            "cnpj": cnpj or "",
            "cep": cep or "",
            "logradouro": logradouro or "",
            "numero": (numero or ""),
            "complemento": (compl or ""),
            "bairro": bairro or "",
            "cidade": cidade or "",
            "uf": uf or "",
            "email": email or "",
            "telefone": telefone or "",
            "ie": ie or "",
            "im": im or "",
            "obs": obs or "",
            "dia_venc": str(dia_venc or "").strip(),
            "cod_ibge": str(cod_ibge or "").strip(),
        }

        empresa = self._dados_empresa_prestadora()

        # >>> NOVO PASSO: escolher procedimentos/valores
        dlg_procs = DialogProcedimentosContrato(self)
        if dlg_procs.exec_() != QDialog.Accepted:
            return
        itens = dlg_procs.get_itens()
        if not itens:
            QMessageBox.warning(self, "Atenção", "Adicione ao menos um procedimento.")
            return

        # categorias selecionadas (para montar bloco de médicos)
        categorias_escolhidas = sorted({i["categoria"] for i in itens})


        # Onde salvar
        nome_sugestao = f"CONTRATO_{self._slug(conv['nome'])}_{date.today().isoformat()}.pdf"
        destino, _ = QFileDialog.getSaveFileName(self, "Salvar contrato", nome_sugestao, "PDF (*.pdf)")
        if not destino:
            return

        html = self._render_html_contrato_modelo(empresa, conv, itens, categorias_escolhidas)

        # Gerar PDF (fallback para HTML)

        options = {
            "page-size": "A4",
            "margin-top": "12mm",
            "margin-right": "12mm",
            "margin-bottom": "14mm",
            "margin-left": "12mm",
            "encoding": "UTF-8",
            "enable-local-file-access": "",  # para carregar o logo local
        }

        # Gerar PDF (fallback para HTML)
        config = self._pdfkit_config()

        # cria header/footer temporários
        header_path, footer_path = self._build_header_footer(empresa)

        options = {
            "page-size": "A4",
            # margens maiores pra dar espaço ao cabeçalho/rodapé
            "margin-top": "24mm",
            "margin-right": "12mm",
            "margin-bottom": "22mm",
            "margin-left": "12mm",
            "encoding": "UTF-8",
            "enable-local-file-access": "",  # necessário para logo local
            "header-html": header_path,
            "footer-html": footer_path,
            "header-spacing": "4",
            "footer-spacing": "4",
            # opcional: deixa a saída mais silenciosa
            "quiet": "",
        }

        # config já criado acima: config = self._pdfkit_config()
        config = self._pdfkit_config()
        header_path = footer_path = None

        try:
            # cria header/footer temporários
            header_path, footer_path = self._build_header_footer(empresa)

            options = {
                "page-size": "A4",
                "margin-top": "24mm",
                "margin-right": "12mm",
                "margin-bottom": "22mm",
                "margin-left": "12mm",
                "encoding": "UTF-8",
                "enable-local-file-access": "",
                "header-html": header_path,
                "footer-html": footer_path,
                "header-spacing": "4",
                "footer-spacing": "4",
                # "quiet": "",  # opcional
            }

            pdfkit.from_string(html, destino, options=options, configuration=config)

            # tentar assinar
            signed_path = self._assinar_pdf(destino, empresa)
            if signed_path:
                QMessageBox.information(self, "Sucesso", f"Contrato gerado e ASSINADO em:\n{signed_path}")
            else:
                QMessageBox.information(self, "Sucesso", f"Contrato gerado em:\n{destino}\n(Assinatura digital não aplicada.)")

        except Exception as e:
            alt = Path(destino).with_suffix(".html")
            try:
                with open(alt, "w", encoding="utf-8") as f:
                    f.write(html)
                msg_extra = ""
                if config is None:
                    msg_extra = (
                        "\nDica: instale o wkhtmltopdf ou coloque o executável em "
                        "'bin/wkhtmltopdf.exe' ao lado do app, ou defina a variável "
                        "de ambiente WKHTMLTOPDF_PATH com o caminho completo."
                    )
                QMessageBox.warning(
                    self, "Atenção",
                    f"Não foi possível gerar o PDF automaticamente ({e}).{msg_extra}\n"
                    f"Salvei o HTML em:\n{alt}\nAbra no navegador e imprima em PDF."
                )
            except Exception as e2:
                QMessageBox.critical(self, "Erro", f"Falha no fallback HTML:\n{e2}")
        finally:
            # limpa temporários
            for p in (header_path, footer_path):
                try:
                    if p and os.path.exists(p):
                        os.unlink(p)
                except Exception:
                    pass






    # ------------------ helpers ------------------
        # -------- helpers contrato --------
    def _only_digits(self, s: str) -> str:
        
        return re.sub(r"\D", "", str(s or ""))

    def _fmt_cpf_cnpj(self, v: str) -> str:
        d = self._only_digits(v)
        if len(d) == 14:  # CNPJ
            return f"{d[:2]}.{d[2:5]}.{d[5:8]}/{d[8:12]}-{d[12:]}"
        if len(d) == 11:  # CPF
            return f"{d[:3]}.{d[3:6]}.{d[6:9]}-{d[9:]}"
        return v or ""

    def _fmt_cep(self, v: str) -> str:
        d = self._only_digits(v)
        return f"{d[:5]}-{d[5:]}" if len(d) == 8 else (v or "")

    def _slug(self, s: str) -> str:
        
        s = unicodedata.normalize("NFKD", s or "")
        s = "".join(c for c in s if not unicodedata.combining(c))
        s = re.sub(r"[^a-zA-Z0-9]+", "_", s).strip("_")
        return s[:60] or "CONVENIO"

    def _data_extenso(self, d: date) -> str:
        meses = ["janeiro","fevereiro","março","abril","maio","junho",
                 "julho","agosto","setembro","outubro","novembro","dezembro"]
        return f"{d.day} de {meses[d.month-1]} de {d.year}"

    def _dados_empresa_prestadora(self) -> dict:
        """Lê configuracoes_empresa (se existir) para preencher o bloco da Medical Laudos."""
        empresa = {
            "razao_social": "MEDICAL LAUDOS LTDA",
            "nome_fantasia": "Medical Laudos",
            "cnpj": "48.036.500/0001-00",
            "endereco": "Rua Itaguaí 746, Bairro Caiçaras – Belo Horizonte - MG",
            "cep": "30.775-110",
            "telefone": "(31) 9 8848-0823",
            "email": "administrativo@medicallaudos.com.br",
            "cidade": "Belo Horizonte",
            "uf": "MG",
        }
        try:
            con = sqlite3.connect(CAMINHO_BANCO)
            cur = con.cursor()
            cur.execute("SELECT name FROM sqlite_master WHERE type='table' AND name='configuracoes_empresa'")
            if cur.fetchone():
                # ajuste estas colunas se o seu schema for diferente
                cur.execute("""
                    SELECT COALESCE(razao_social,''), COALESCE(nome_fantasia,''), COALESCE(cnpj,''),
                           COALESCE(endereco,''), COALESCE(cidade,''), COALESCE(uf,''),
                           COALESCE(cep,''), COALESCE(telefone,''), COALESCE(email,'')
                    FROM configuracoes_empresa LIMIT 1
                """)
                r = cur.fetchone()
                if r:
                    empresa = {
                        "razao_social": r[0] or empresa["razao_social"],
                        "nome_fantasia": r[1] or empresa["nome_fantasia"],
                        "cnpj": self._fmt_cpf_cnpj(r[2]),
                        "endereco": r[3] or empresa["endereco"],
                        "cidade": r[4] or empresa["cidade"],
                        "uf": r[5] or empresa["uf"],
                        "cep": self._fmt_cep(r[6]),
                        "telefone": r[7] or empresa["telefone"],
                        "email": r[8] or empresa["email"],
                    }
            con.close()
        except Exception:
            pass
        return empresa


    def _render_html_contrato_modelo(self, empresa: dict, conv: dict, itens=None, categorias_escolhidas=None) -> str:
        """HTML baseado no 'MODELO DE CONTRATO.doc' com procedimentos/valores escolhidos."""
        hoje_ext = self._data_extenso(date.today())
        logo_html = self._logo_tag()

        contratante_nome = (conv["razao_social"] or conv["nome"] or "").strip()
        cnpj = self._fmt_cpf_cnpj(conv["cnpj"])
        endereco = ", ".join([p for p in [
            conv["logradouro"], str(conv["numero"] or "").strip(),
            f"Bairro {conv['bairro']}" if conv["bairro"] else "",
            f"{conv['cidade']} - {conv['uf']}".strip()
        ] if p])
        cep = self._fmt_cep(conv["cep"])
        email = conv["email"] or ""
        tel = conv["telefone"] or ""
        email_fin = email
        cidade_forum = conv["cidade"] or "Belo Horizonte"
        uf_forum = conv["uf"] or "MG"
        dia_venc = conv["dia_venc"] or "15"

        # tabela default (se nada for passado)
        if not itens:
            default = [
                ("Mapa 24 horas", 15.00),
                ("Holter 24 horas", 15.00),
                ("Eletrocardiograma - ECG", 4.90),
                ("Eletroencefalograma Ocupacional", 4.90),
                ("Eletroencefalograma Clínico", 4.90),
                ("Espirometria", 4.90),
                ("Espirometria com Broncodilatador", 4.90),
            ]
            itens = [{"procedimento": p, "valor_float": v, "valor_str": self._fmt_brl(v),
                    "categoria": PROC_TO_CATEGORY.get(p, "")} for p, v in default]

        linhas_tabela = self._money_rows_from_itens(itens)

        # Médicos por categoria (somente das categorias escolhidas)
        medicos_html = ""
        if categorias_escolhidas:
            blocos = []
            for cat in categorias_escolhidas:
                docs = DOCTORS_BY_CATEGORY.get(cat, [])
                if not docs:
                    continue
                items = "".join(f"<li>{d}</li>" for d in docs)
                blocos.append(f"<div class='tit'>Médicos Responsáveis — {cat}</div><ul>{items}</ul>")
            if blocos:
                medicos_html = f"<div class='box'>{''.join(blocos)}</div>"

        css = """
        <style>
            /* Fonte padrão do contrato */
            html, body{
            font-family: 'Calibri','Carlito','Segoe UI',Arial,Helvetica,sans-serif;
            font-size: 11pt;        /* Calibri 11 */
            line-height: 1.35;
            color:#222;
            }

            /* Título */
            h1{
            font-size: 11pt;
            font-weight: 700;
            text-align: center;
            margin: 8px 0 12px;
            }

            .header{display:flex;align-items:center;gap:12px;margin-bottom:6px;}
            .brand{font-weight:700;}

            .box{border:1px solid #ddd;border-radius:6px;padding:10px;margin:10px 0;}
            .tit{font-weight:700;margin:6px 0;color:#333;}

            table{width:100%;border-collapse:collapse;margin:6px 0;}
            th,td{border:1px solid #ccc;padding:6px;text-align:left;}
            th{background:#f0f0f0}

            /* ===================== Assinaturas ===================== */
            /* Tabela das assinaturas */
            .assin-table{
            width:100%;
            margin-top:36px;
            border-collapse:separate; 
            border-spacing:0; 
            table-layout:fixed;
            }
            .assin-table, .assin-table tr, .assin-table td { page-break-inside: avoid; }

            /* Cada célula de assinatura */
            .assin-cell{
            width:50%;
            text-align:center; 
            vertical-align:top;
            padding:0 12mm;        /* ⇐ PAD_MM */
            height:46mm;           /* altura total da célula */
            position:relative;
            }

            /* Faixa onde o carimbo deve ficar (acima da linha) */
            .assin-space{
            position:absolute;
            left:12mm;             /* ⇐ PAD_MM */
            right:12mm;            /* ⇐ PAD_MM */
            top:8mm;               /* ⇐ SPACE_TOP_MM */
            height:32mm;           /* ⇐ SPACE_HEIGHT_MM */
            }

            /* Anchor invisível ocupando 100% da faixa — vira anotação no PDF */
            .sig-anchor{
            position:absolute;
            inset:0;
            display:block;
            border:0;
            text-decoration:none;
            color:transparent;
            }
            .sig-anchor img{ display:block; width:100%; height:100%; opacity:0.01; }

            /* Linha de assinatura no rodapé da célula */
            .assin-cell .line{
            position:absolute;
            left:12mm;             /* ⇐ PAD_MM */
            right:12mm;            /* ⇐ PAD_MM */
            bottom:4mm;            /* ⇐ LINE_BOTTOM_GAP_MM */
            display:block;
            border-top:1px solid #333;
            padding-top:8px;
            }
            /* ======================================================= */
            </style>

"""

        html = f"""<!DOCTYPE html>
<html lang="pt-br"><head><meta charset="utf-8">{css}<title>Contrato</title></head>
<body>

<h1>CONTRATO DE PRESTAÇÃO DE SERVIÇOS QUE ENTRE SI FAZEM {contratante_nome} E MEDICAL LAUDOS LTDA</h1>

<p><b>CONTRATANTE:</b> {contratante_nome}, inscrito no CNPJ sob o número {cnpj}, estabelecido na {endereco} CEP: {cep} — E-mail: {email} / Tel: {tel}</p>
<p><b>CONTRATADO — PRESTADOR DO SERVIÇO:</b> {empresa['razao_social']}, inscrito no CNPJ: {empresa['cnpj']}, estabelecida na {empresa['endereco']} — CEP: {empresa['cep']} — Telefone: {empresa['telefone']} — E-mail: {empresa['email']}.</p>

<p>RESOLVEM firmar o presente contrato de prestação de serviços, observadas e obedecidas as cláusulas e condições a seguir:</p>

<div class="box">
<div class="tit">1. CLÁUSULA PRIMEIRA — DO OBJETO DO CONTRATO</div>
<p>1.1 — O presente contrato tem por objeto a prestação, pela CONTRATADA à CONTRATANTE, de serviços de elaboração de laudos médicos com base nos exames transmitidos eletronicamente pela CONTRATANTE.</p>
</div>

<div class="box">
<div class="tit">2. CLÁUSULA SEGUNDA — DOS PRAZOS E DIRETRIZES PARA A PRESTAÇÃO DOS SERVIÇOS</div>
<p>2.1 — O CONTRATANTE enviará à CONTRATADA os exames médicos sob a forma de arquivo digital através do sistema de gestão on-line ou através do e-mail {empresa['email']}.</p>
<p>2.2 — A CONTRATADA enviará o exame transmitido pelo CONTRATANTE, em forma de digital, a um de seus médicos credenciados, que confeccionará o respectivo laudo.</p>
<p>2.3 — Os laudos serão entregues ao CONTRATANTE com a assinatura em certificado digital.</p>
<p>2.4	O CONTRATANTE terá acesso ao nosso sistema de Gestão Online, onde o contratante terá uma senha de acesso para enviar e receber os exames laudados, no prazo Máximo de 24 horas após o envio do exame.</p>
<p>2.5	Os exames médicos enviados pela Contratante após as 20h00min horas terão inicio da contagem do prazo estipulado na clausula 2.3 a partir das 08h00min horas da manhã do dia subsequente sendo que sábados, domingos e feriados nacionais não serão incluídos na contagem do tempo;</p>
<p>2.6	O CONTRATANTE utilizará exclusivamente o sistema de gestão online ou e-mail para envio de exames e recebimento dos laudos médicos.</p>
</div>

<div class="box">
<div class="tit">3. CLÁUSULA TERCEIRA — DAS OBRIGAÇÕES DA CONTRATADA</div>
<p>3.1	A CONTRATADA prestará os serviços com estrita observância dos preceitos éticos e profissionais relacionados ao trabalho a ser desenvolvido, que deverá ser realizado dentro dos padrões e prazos especificados;</p>
<p>3.2	Atender prontamente as observações e solicitações apresentadas pela CONTRATANTE, no que compatível com o objeto do presente contrato.</p>
<p>3.3	A CONTRATADA compromete-se envidar seus melhores esforços a fim de utilizar, na prestação dos serviços, profissionais qualificados e especializados.</p>
<p>3.3.1	A responsabilidade pelo conteúdo dos laudos é exclusiva do médico signatário por qualquer tipo de erro ou vicio aparente ou não;</p>
<p>3.3.2	A titulo explicativo entende-se como erro ou vicio:</p>
<p>3.3.2.1	Erro de diagnostico</p>
<p>3.3.2.2	Erro de ortografia;</p>
<p>3.3.3	Garantir a utilização de profissionais qualificados e aptos pelas normas do Conselho Regional de Medicina e demais legislações aplicáveis à espécie a elaborar os laudos médicos respectivos a cada espécie de exame.</p>
<p>3.4	A CONTRATADA encaminhará à CONTRATANTE, mensalmente, um relatório constando todas as informações e serviços por ela prestados no mês de referência, para conferencia pela CONTRATANTE.</p>
<p>3.4.1    O relatório de que trata o item 3.4 será encaminhado à CONTRATANTE até o quinto dia útil do mês subsequente.</p>
</div>

<div class="box">
<div class="tit">4. CLÁUSULA QUARTA — DAS OBRIGAÇÕES DA CONTRATANTE</div>
<p>4.1	A CONTRATANTE deverá efetuar os pagamentos devidos à CONTRATADA uma vez executados qualquer dos serviços, observando estritamente o cumprimento dos prazos estabelecidos neste instrumento;</p>
<p>4.2	A CONTRATANTE obriga-se a colocar a disposição da CONTRATADA todas as informações necessárias ao desenvolvimento dos serviços nos prazos previamente definidos, garantindo a qualidade dos exames médicos enviados, viabilizando a perfeita elaboração dos laudos médicos.</p>
<p>4.2.1	Deve a CONTRATANTE realizar de maneira correta os cadastros dos pacientes no sistema bem como enviar os exames médicos de modo que a sua apresentação esteja de acordo com os requisitos técnicos adequados, para análise inequívoca do medico que confeccionará o laudo.</p>
<p>4.2.2	Caso o cadastro do paciente e os exames inseridos no sistema pela CONTRATANTE não se enquadrem no item 4.2.1, a CONTRATADA comunicará o fato à CONTRATANTE para que sejam procedidas as devidas correções/alterações. Nesta hipótese, o cadastro inicialmente inserido pela CONTRATANTE será excluído do sistema e a CONTRATANTE deverá inseri-lo novamente com as correções necessárias.</p>
<p>4.2.3	Nos casos em que persistirem as inconsistências informadas, sem qualquer providência da CONTRATANTE, a CONTRATADA ficará desobrigada de confeccionar o respectivo laudo, por estar comprometida a qualidade e confiabilidade do serviço.</p>
<p>4.2.4	Nas hipóteses de reenvio de exame para correções/retificações pela CONTRATANTE, o prazo para entrega do serviço pela CONTRATADA, relativo aos exames devolvidos à CONTRATANTE para correções/retificações se iniciará a partir do envio do novo cadastro, conforme item 2.4 do presente contrato.</p>
<p>4.3	É de responsabilidade exclusiva do contratante a observância dos seguintes itens:</p>
<p>4.3.1	Colher a identificação do paciente corretamente, tais como os seguintes: Nome completo / Identidade / CPF/Idade/Data Nascimento/Função.</p>
<p>4.3.2	As informações obtidas ou extraídas na vigência deste contrato deverão ter tratamento de “sigilosas ou confidenciais”, e são propriedade da CONTRATADA, sendo vedada qualquer divulgação a terceiros ou utilização para fins pessoais, zelando a CONTRATANTE, por si, por seus empregados, prepostos, subcontratados ou representantes a qualquer titulo, incluindo-se, mas não limitando a manutenção do sigilo absoluto sobre dados, materiais, informações, documentos, especificações técnicas e comerciais, cadastro de clientes, lista de preços e produtos de que eventualmente tenham conhecimento ou acesso em razão do presente contrato.</p>
<p>4.3.3	É vedado a CONTRATANTE entrar em contato direto, bem como celebrar qualquer tipo de negocio jurídico com os médicos vinculados a CONTRATADA, sem a participação ou autorização desta;</p>
<p>4.3.4	O dever de sigilo previsto nos itens acima deverá ser observado mesmo após o termino deste contrato.</p>
<p>4.3.5	O descumprimento das obrigações de sigilo e de confidencialidade poderá importar:</p>
<p>a)	Na rescisão contratual, vigente o contrato;</p>
<p>b)	Na responsabilidade da parte faltosa por perdas e danos;</p>
<p>c)	Na adoção dos remédios jurídicos e sanções cabíveis.</p>
<p>4.3.6	E VEDADO A CONTRATANTE, promover qualquer alteração nos laudos médicos emitidos pelo CONTRATADO, sendo que qualquer modificação que entenda necessária devera ser solicitada por escrito, sendo objeto de analise pelo CONTRATADO, que não ficará obrigado a modifica-lo caso esteja de acordo com o exame realizado.</p>
<p>4.3.7	Caso a CONTRATANTE promova alguma alteração em laudo emitido pelo CONTRATADO, sem o consentimento deste, será responsabilizada cível e criminalmente pelos danos que venha a causar ao CONTRATADO ou a terceiros, sem prejuízos das sanções previstas em Lei.</p>
</div>

<div class="box">
<div class="tit">5.	CLAUSULA QUINTA – DO PREÇO E DAS CONDIÇÕES DE PAGAMENTO</div>
<p>5.1 — Tabela pactuada:</p>
<table>
    <tr><th>Exame</th><th>Valor</th></tr>
    {linhas_tabela}
</table>
<p>5.2 — Pagamentos até o dia <b>{dia_venc}</b> de cada mês, relativos ao mês anterior; boleto enviado para <b>{email_fin}</b>.</p>
<p>5.2.1 - O pagamento a que se refere o item 5.2 se processará da seguinte forma: após o transcurso integral do mês em que o serviço está sendo prestado (ou seja, do primeiro dia do mês até o último dia do mesmo mês), a CONTRATANTE efetuará o pagamento referente ao mês em questão até o dia <b>{dia_venc}</b> do mês subsequente</b>.</p>
<p><i>Parágrafo único</i> — O não pagamento no prazo poderá suspender os serviços até regularização.</p>
<p>5.3 O Pagamento dos serviços prestados pela contratada e descritos neste contrato serão realizados pela CONTRATANTE através de boleto bancário, que será disponibilizado por e-mail em até 5 (cinco) dias antes do vencimento</b>.</p>
</div>

{medicos_html}

<div class="box">
<div class="tit">6. DO PRAZO</div>
<p>O presente contrato terá prazo de duração de 12(doze) meses, passando a valer a partir da assinatura pelas partes e podendo ser prorrogado desde que não haja manifestação em contrário dentro de 30(trinta) dias antes do fim deste prazo</p>
</div>

<div class="box">
<div class="tit">7. DO FORO</div>
<p>Fica eleito o foro da Comarca de {cidade_forum}/{uf_forum} para dirimir eventuais controvérsias.</p>
</div>

<p>Por estarem assim justos e contratados, firmam o presente instrumento em duas vias de igual teor.</p>
<p><b>{empresa['cidade']}, {hoje_ext}.</b></p>

<table class="assin-table">
  <tr>
    <td class="assin-cell">
      <div class="assin-space">
        <!-- anchor com pixel transparente ocupando toda a faixa -->
        <a class="sig-anchor" href="https://sig-anchor.local/assin-left">
          <img alt="" src="data:image/png;base64,
iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAQAAAC1HAwCAAAAC0lEQVR42mP8/x8AAwMB/azkZxkAAAAASUVORK5CYII=">
        </a>
      </div>
      <div class="line">{empresa['razao_social']}<br/>CNPJ: {empresa['cnpj']}</div>
    </td>
    <td class="assin-cell">
      <div class="assin-space"></div>
      <div class="line">{contratante_nome}<br/>CNPJ: {cnpj}</div>
    </td>
  </tr>
</table>



</body></html>"""
        return html




    def __init__(self):
        super().__init__()
        self.layout = QVBoxLayout(self)

        # Botões de ação
        self.botao_novo = QPushButton("+ Novo Convênio")
        self.botao_editar = QPushButton("✏️ Editar Convênio")
        self.botao_excluir = QPushButton("🗑️ Excluir Convênio")
        self.botao_exportar = QPushButton("Exportar Excel")
        self.botao_contrato = QPushButton("Gerar Contrato")

        # Estilo visual dos botões (igual ao módulo 2)
        estilo_botao = """
            QPushButton {
                background-color: #3498db;
                color: white;
                padding: 6px 12px;
                font-weight: bold;
                border-radius: 4px;
            }
            QPushButton:hover {
                background-color: #2980b9;
            }
        """

        for botao in [self.botao_novo, self.botao_editar, self.botao_excluir, self.botao_exportar, self.botao_contrato]:
            botao.setStyleSheet(estilo_botao)

        # Conexões dos botões
        self.botao_novo.clicked.connect(self.abrir_modal)
        self.botao_editar.clicked.connect(self.editar_convenio)
        self.botao_excluir.clicked.connect(self.excluir_convenio)
        self.botao_exportar.clicked.connect(self.exportar_excel)
        self.botao_contrato.clicked.connect(self.gerar_contrato)

        # Layout dos botões
        botoes = QHBoxLayout()
        botoes.addWidget(self.botao_novo)
        botoes.addWidget(self.botao_editar)
        botoes.addWidget(self.botao_excluir)
        botoes.addWidget(self.botao_exportar)
        botoes.addWidget(self.botao_contrato)
        self.layout.addLayout(botoes)

        # Campo de busca
        filtro_layout = QHBoxLayout()
        self.campo_busca = QLineEdit()
        self.campo_busca.setPlaceholderText("🔍 Buscar por nome do convênio...")
        self.campo_busca.textChanged.connect(self.filtrar_convenios)
        filtro_layout.addWidget(self.campo_busca)
        self.layout.addLayout(filtro_layout)


        # Tabela
        self.tabela = QTableWidget()
        self.tabela.setColumnCount(9)
        self.tabela.setHorizontalHeaderLabels([
            "Nome", "Razão Social", "CNPJ", "Cidade", "UF", "E-mail", "Telefone", "Vencimento", "Cód. IBGE"
        ])

        self.tabela.setSelectionBehavior(QTableWidget.SelectRows)
        self.tabela.setEditTriggers(QTableWidget.NoEditTriggers)
        self.tabela.setAlternatingRowColors(True)
        self.tabela.setStyleSheet("alternate-background-color: #f2f2f2; background-color: white;")
        self.tabela.horizontalHeader().setStyleSheet("font-weight: bold;")
        self.tabela.verticalHeader().setVisible(False)
        self.tabela.itemDoubleClicked.connect(self.editar_convenio)
        self.layout.addWidget(self.tabela)

        self.carregar_convenios()

        # Distribuição proporcional de colunas
        header = self.tabela.horizontalHeader()
        header.setStretchLastSection(False)

        # Aplicar redimensionamento proporcional
        header.setSectionResizeMode(0, QHeaderView.Stretch)  # Nome
        header.setSectionResizeMode(1, QHeaderView.Stretch)  # Razão Social
        header.setSectionResizeMode(2, QHeaderView.ResizeToContents)  # CNPJ
        header.setSectionResizeMode(3, QHeaderView.ResizeToContents)  # Cidade
        header.setSectionResizeMode(4, QHeaderView.ResizeToContents)  # UF
        header.setSectionResizeMode(5, QHeaderView.Stretch)  # E-mail
        header.setSectionResizeMode(6, QHeaderView.ResizeToContents)  # Telefone
        header.setSectionResizeMode(7, QHeaderView.ResizeToContents)  # Vencimento
        header.setSectionResizeMode(8, QHeaderView.ResizeToContents)  # Cód. Trib.


    def filtrar_convenios(self):
        texto = self.campo_busca.text().lower()
        for row in range(self.tabela.rowCount()):
            item_nome = self.tabela.item(row, 0)  # Coluna "Nome"
            if item_nome and texto in item_nome.text().lower():
                self.tabela.setRowHidden(row, False)
            else:
                self.tabela.setRowHidden(row, True)


    def carregar_convenios(self):
        try:
            conn = sqlite3.connect(CAMINHO_BANCO)
            cursor = conn.cursor()
            cursor.execute("""
                SELECT nome, razao_social, cnpj, cidade, uf, email, telefone, dia_vencimento,
                    COALESCE(codigo_municipio_ibge, '')
                FROM convenios
                ORDER BY nome
            """)

            dados = cursor.fetchall()
            conn.close()

            self.tabela.setRowCount(0)
            for row_idx, row_data in enumerate(dados):
                self.tabela.insertRow(row_idx)
                for col_idx, value in enumerate(row_data):
                    item = QTableWidgetItem(str(value))
                    item.setTextAlignment(Qt.AlignCenter)
                    self.tabela.setItem(row_idx, col_idx, item)

        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao carregar convênios:\n{e}")


    def abrir_modal(self):
        dialog = DialogNovoConvenio(self)
        if dialog.exec_() == QDialog.Accepted:
            self.carregar_convenios()

    def editar_convenio(self):
        selected = self.tabela.currentRow()
        if selected < 0:
            QMessageBox.warning(self, "Atenção", "Selecione um convênio para editar.")
            return

        nome = self.tabela.item(selected, 0).text()
        try:
            conn = sqlite3.connect(CAMINHO_BANCO)
            cursor = conn.cursor()
            cursor.execute("SELECT * FROM convenios WHERE nome = ?", (nome,))
            dados = cursor.fetchone()
            conn.close()
            if not dados:
                QMessageBox.warning(self, "Aviso", "Convênio não encontrado.")
                return

            dialog = DialogNovoConvenio(self, dados_existentes=dados)
            if dialog.exec_() == QDialog.Accepted:
                self.carregar_convenios()

        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao editar convênio:\n{e}")

    def excluir_convenio(self):
        selected = self.tabela.currentRow()
        if selected < 0:
            QMessageBox.warning(self, "Atenção", "Selecione um convênio para excluir.")
            return

        nome = self.tabela.item(selected, 0).text()
        confirm = QMessageBox.question(
            self, "Confirmar Exclusão",
            f"Tem certeza que deseja excluir o convênio '{nome}'?",
            QMessageBox.Yes | QMessageBox.No
        )
        if confirm != QMessageBox.Yes:
            return

        try:
            conn = sqlite3.connect(CAMINHO_BANCO)
            cursor = conn.cursor()
            cursor.execute("DELETE FROM convenios WHERE nome = ?", (nome,))
            conn.commit()
            conn.close()
            QMessageBox.information(self, "Sucesso", f"Convênio '{nome}' excluído com sucesso.")
            self.carregar_convenios()
        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao excluir convênio:\n{e}")

    def exportar_excel(self):
        selected = self.tabela.currentRow()
        if selected < 0:
            QMessageBox.warning(self, "Atenção", "Selecione um convênio na tabela.")
            return

        nome = self.tabela.item(selected, 0).text()
        nome_maiusculo = nome.upper()

        dialog = DialogSelecionarCompetencia()
        if dialog.exec_() != QDialog.Accepted:
            return
        competencia = dialog.get_competencia()

        try:
            conn = sqlite3.connect(CAMINHO_BANCO)
            df = pd.read_sql_query(
                "SELECT * FROM registros_financeiros WHERE Convenio = ? AND Competência = ?",
                conn, params=(nome, competencia)
            )
            conn.close()

            df = tratar_duplicados(df)

            # 🟡 Zera "Valor Convenio" se duplicado
            if "Duplicado" in df.columns and "Valor Convenio" in df.columns:
                df.loc[df["Duplicado"] == True, "Valor Convenio"] = 0

            if df.empty:
                QMessageBox.information(self, "Aviso", f"Nenhum registro encontrado para {nome} na competência {competencia}.")
                return

            if "Médico" in df.columns:
                df["Médico"] = df["Médico"].str.extract(r"^(.*?)(?:\s*-\s*CRM|$)")[0].str.strip()

            colunas_para_remover = ["Convenio"]
            df.drop(columns=[col for col in colunas_para_remover if col in df.columns], inplace=True)

            nome_arquivo = f"{nome}_{competencia.replace('/', '-')}.xlsx"
            caminho, _ = QFileDialog.getSaveFileName(
                self, "Salvar como", nome_arquivo, "Excel Files (*.xlsx)"
            )
            if not caminho:
                return

            colunas_visiveis = [
                col for col in df.columns if col not in [
                    'Cod Registro', 'Duplicado', 'NaoDuplicadoOIT',
                    'Competência', 'Data Cadastro', 'Data Recebimento', 'Valor Médico', 'Competencia'
                ]
            ]

            writer = pd.ExcelWriter(caminho, engine='xlsxwriter')
            workbook = writer.book
            df.to_excel(writer, index=False, sheet_name='Registros', columns=colunas_visiveis, startrow=2)
            worksheet = writer.sheets['Registros']

            # Estilos
            azul_cabecalho = '#B7D6F4'
            header_format = workbook.add_format({'bold': True, 'bg_color': '#D9D9D9', 'border': 1, 'align': 'center', 'valign': 'vcenter'})
            center_format = workbook.add_format({'align': 'center', 'valign': 'vcenter'})
            money_format = workbook.add_format({'num_format': '"R$" #,##0.00', 'align': 'center'})
            amarelo_center = workbook.add_format({'align': 'center', 'bg_color': '#FFF59D'})
            amarelo_money = workbook.add_format({'num_format': '"R$" #,##0.00', 'align': 'center', 'bg_color': '#FFF59D'})
            verde_center = workbook.add_format({'align': 'center', 'bg_color': '#C8E6C9'})
            verde_money = workbook.add_format({'num_format': '"R$" #,##0.00', 'align': 'center', 'bg_color': '#C8E6C9'})
            bold_blue = workbook.add_format({'bold': True, 'font_color': 'black', 'bg_color': azul_cabecalho, 'border': 1, 'align': 'center'})
            bold_right = workbook.add_format({'bold': True, 'align': 'right', 'bg_color': azul_cabecalho, 'border': 1})
            value_right = workbook.add_format({'align': 'right', 'bg_color': azul_cabecalho, 'border': 1})
            money_right = workbook.add_format({'num_format': '"R$" #,##0.00', 'align': 'right', 'bg_color': azul_cabecalho, 'border': 1})
            logo_background = workbook.add_format({'bg_color': azul_cabecalho, 'border': 1})

            total_laudos = len(df)
            total_valor = df["Valor Convenio"].apply(
                lambda v: float(str(v).replace("R$", "").replace(".", "").replace(",", ".")) if isinstance(v, str) else v
            ).sum()

            # Cabeçalho ajustado
            worksheet.merge_range("A1:A2", "", logo_background)
            worksheet.merge_range("B1:E1", f"FATURAMENTO {nome_maiusculo}", bold_blue)
            worksheet.merge_range("B2:E2", f"Competência: {competencia}", bold_blue)
            worksheet.write("F1", "TOTAL", bold_right)
            worksheet.write("G1", total_valor, money_right)
            worksheet.write("F2", "LAUDOS", bold_right)
            worksheet.write("G2", total_laudos, value_right)

            # Logo
            caminho_logo = os.path.join("icones", "medical_laudos_logo.png")
            if os.path.exists(caminho_logo):
                worksheet.insert_image("A1", caminho_logo, {'x_scale': 0.45, 'y_scale': 0.45, 'x_offset': 2, 'y_offset': 2})

            # Cabeçalhos da tabela (linha 3)
            for col_num, nome_coluna in enumerate(colunas_visiveis):
                worksheet.write(2, col_num, nome_coluna, header_format)
                largura_max = max(
                    [len(str(cell)) for cell in [nome_coluna] + df[nome_coluna].astype(str).tolist()]
                ) + 2
                worksheet.set_column(col_num, col_num, largura_max)

            # Ativar filtro automático na linha dos cabeçalhos
            worksheet.autofilter(2, 0, 2 + len(df), len(colunas_visiveis) - 1)

            def str_to_bool(valor):
                return str(valor).strip().lower() in ["1", "true", "verdadeiro"]

            for row_num, row_data in df.iterrows():
                duplicado = str_to_bool(row_data.get("Duplicado", ""))
                nao_oit = str_to_bool(row_data.get("NaoDuplicadoOIT", ""))

                for col_num, nome_coluna in enumerate(colunas_visiveis):
                    valor = row_data[nome_coluna]
                    if duplicado:
                        formato = amarelo_money if nome_coluna == 'Valor Convenio' else amarelo_center
                    elif nao_oit:
                        formato = verde_money if nome_coluna == 'Valor Convenio' else verde_center
                    else:
                        formato = money_format if nome_coluna == 'Valor Convenio' else center_format

                    linha_excel = row_num + 3
                    if nome_coluna == 'Valor Convenio':
                        try:
                            worksheet.write_number(linha_excel, col_num, float(valor), formato)
                        except:
                            worksheet.write(linha_excel, col_num, str(valor), formato)
                    else:
                        worksheet.write(linha_excel, col_num, str(valor), formato)

            writer.close()
            QMessageBox.information(self, "Sucesso", f"Arquivo exportado com sucesso para:\n{caminho}")

        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao exportar Excel:\n{e}")



