# modulo5_faturamento_convenios.py

from PyQt5.QtWidgets import (
    QWidget, QVBoxLayout, QHBoxLayout, QLabel, QPushButton,
    QComboBox, QTableWidget, QTableWidgetItem, QMessageBox, QDialog,
    QFormLayout, QFileDialog, QSizePolicy, QSpacerItem, QHeaderView, QLineEdit, QInputDialog
)
import sqlite3
import pandas as pd
import os
import unicodedata
from gerar_nfse_xml import gerar_xml_nfse_por_convenio
from PyQt5.QtGui import QIcon
from PyQt5.QtCore import Qt, pyqtSignal, QTimer
import re
from pathlib import Path
from cancelar_nfse import cancelar_nfse_por_convenio, CANCEL_MOTIVOS

STATUS_MAP = {
    1: "Não recebido",
    2: "Não processado",
    3: "Processando",
    4: "Processado com sucesso",
    5: "Processado com erro",
}


def _only_digits(s: str) -> str:
    return re.sub(r"\D", "", s or "")

def _obter_numero_nfse_para_cancelar(id_convenio: int, competencia_servico: str) -> str | None:
    """
    Tenta descobrir automaticamente o número da NFS-e do convênio/competência atuais.
    1) notas_emitidas_status (por nome do convênio + competência)
    2) nfse_notas JOIN nfse_lotes (por id_convenio + competência)
    Retorna o número (string) ou None.
    """
    with sqlite3.connect(CAMINHO_BANCO) as conn:
        cur = conn.cursor()

        # Descobre o nome do convênio (notas_emitidas_status usa 'convenio' por nome)
        row = cur.execute("SELECT nome FROM convenios WHERE id = ?", (id_convenio,)).fetchone()
        if not row:
            return None
        nome_convenio = row[0]

        # 1) Tenta pegar do espelho de status
        row = cur.execute("""
            SELECT numero_nfse
              FROM notas_emitidas_status
             WHERE convenio = ?
               AND (competencia_servico = ? OR competencia = ?)
               AND numero_nfse IS NOT NULL
               AND TRIM(numero_nfse) <> ''
             ORDER BY data_emissao DESC
             LIMIT 1
        """, (nome_convenio, competencia_servico, competencia_servico)).fetchone()
        if row and str(row[0]).strip():
            return _only_digits(str(row[0]).strip())  # garante dígitos

        # 2) Fallback: tabelas "de origem"
        row = cur.execute("""
            SELECT n.numero_nfse
              FROM nfse_notas n
              JOIN nfse_lotes l ON l.protocolo = n.protocolo
             WHERE l.convenio_id = ?
               AND n.competencia = ?
               AND n.numero_nfse IS NOT NULL
             ORDER BY n.id DESC
             LIMIT 1
        """, (id_convenio, competencia_servico)).fetchone()
        if row and str(row[0]).strip():
            return _only_digits(str(row[0]).strip())

    return None

def formatar_situacao(cod):
    if cod is None or str(cod).strip() == "":
        return "—"
    try:
        cod_int = int(cod)
    except:
        return str(cod)
    return f"{cod_int} - {STATUS_MAP.get(cod_int, 'Desconhecida')}"

def carregar_situacoes_por_competencia(competencia: str) -> pd.DataFrame:
    with sqlite3.connect(CAMINHO_BANCO) as conn:
        df_lotes = pd.read_sql_query(
            """
            SELECT convenio_nome, protocolo, situacao, mensagem_retorno, data_envio, id,
                   COALESCE(boleto_pago, 0) AS boleto_pago
            FROM nfse_lotes
            WHERE competencia = ?
            ORDER BY data_envio DESC, id DESC
            """,
            conn, params=(competencia,)
        )
    if df_lotes.empty:
        return df_lotes
    return df_lotes.drop_duplicates(subset=["convenio_nome"], keep="first").set_index("convenio_nome")




def tratar_duplicados(df):
    import unicodedata

    def normalizar_medico(nome):
        if pd.isna(nome):
            return ''
        nome = str(nome).strip().lower()
        nome = unicodedata.normalize('NFKD', nome).encode('ASCII', 'ignore').decode('utf-8')
        return nome

    df["Duplicado"] = False
    df["NaoDuplicadoOIT"] = False

    if df.empty or not all(c in df.columns for c in ["Nome", "Empresa", "Convenio", "Tipo de Exame", "Procedimento", "Data Exame", "Médico"]):
        return df

    # Separa registros OIT
    oit_df = df[df["Procedimento"].str.upper().str.contains("RX - TÓRAX OIT - 2 ASSINATURAS", na=False)].copy()
    restantes_df = df[~df.index.isin(oit_df.index)].copy()

    # 🔒 Blindagem: padroniza Empresa
    for df_padronizar in [oit_df, restantes_df]:
        if "Empresa" in df_padronizar.columns:
            df_padronizar["Empresa"] = df_padronizar["Empresa"].fillna("").astype(str).str.strip()

    colunas_chave = ["Nome", "Empresa", "Convenio", "Tipo de Exame", "Procedimento", "Data Exame"]

    # 🟢 Lógica OIT
    grupo_oit = oit_df.groupby(colunas_chave)
    for _, grupo in grupo_oit:
        grupo = grupo.copy()
        indices = grupo.index.tolist()
        medicos_norm = grupo["Médico"].apply(normalizar_medico)
        medicos_unicos = medicos_norm.unique()

        if len(grupo) == 2 and len(medicos_unicos) == 2:
            oit_df.loc[indices, "NaoDuplicadoOIT"] = True
        else:
            encontrados = set()
            validos = []
            for idx, nome in zip(grupo.index, medicos_norm):
                if nome not in encontrados and len(validos) < 2:
                    validos.append(idx)
                    encontrados.add(nome)
            for idx in validos:
                oit_df.loc[idx, "NaoDuplicadoOIT"] = True
            duplicados = grupo.index.difference(validos)
            oit_df.loc[duplicados, "Duplicado"] = True

    # 🟡 Lógica geral de duplicados
    if all(c in restantes_df.columns for c in colunas_chave):
        duplicatas = restantes_df.duplicated(subset=colunas_chave, keep='first')
        restantes_df.loc[duplicatas, "Duplicado"] = True

    df_final = pd.concat([oit_df, restantes_df]).sort_index()

    # 🔁 Zera valores apenas dos duplicados
    if "Valor Convenio" in df_final.columns:
        df_final.loc[df_final["Duplicado"] == True, "Valor Convenio"] = 0

    return df_final

CAMINHO_BANCO = os.path.join("db", "sistema_financeiro.db")

def remover_acentos(texto):
    if not isinstance(texto, str):
        return ""
    return ''.join(c for c in unicodedata.normalize('NFD', texto) if unicodedata.category(c) != 'Mn')

class DialogFaturamentoConvenio(QDialog):  
    
    def _set_status_nf(self, status: str):
        self.status_nf_atual = status
        self.lbl_status_nf.setText(status)
        self._pintar_status_nf(status)
        self._aplicar_regra_ui_por_status_nf(status) 
    
    
    atualizar_tabela_signal = pyqtSignal()  # ✅ Sinal personalizado
    def __init__(self, convenio_nome, competencia, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Faturamento do Convênio")        
        modo_teste = False  # ✅ Se True, desabilita as travas dos botões
        self.setMinimumWidth(500)
        self.convenio_nome = convenio_nome
        self.competencia = competencia

        layout = QVBoxLayout(self)

        # Estilo normal
        estilo_botao = """
            QPushButton {
                background-color: #3498db;
                color: white;
                padding: 6px 12px;
                font-weight: bold;
                border-radius: 4px;
            }
            QPushButton:hover {
                background-color: #2980b9;
            }
        """

        # Estilo desativado
        estilo_cinza = """
            QPushButton {
                background-color: #bdc3c7;
                color: white;
                padding: 6px 12px;
                font-weight: bold;
                border-radius: 4px;
            }
        """

    

        try:
            conn = sqlite3.connect(CAMINHO_BANCO)
            cursor = conn.cursor()

            cursor.execute("SELECT * FROM convenios WHERE nome = ?", (convenio_nome,))
            dados_convenio = cursor.fetchone()
            colunas = [desc[0] for desc in cursor.description]

            if dados_convenio:
                self.dados_convenio = dict(zip(colunas, dados_convenio))  # ✅ Salva como dicionário
                self.id_convenio = self.dados_convenio.get("id")           # ✅ ID necessário para a função de XML

            colunas = [desc[0] for desc in cursor.description]

            if dados_convenio:
                form_layout = QFormLayout()
                for i, valor in enumerate(dados_convenio):
                    form_layout.addRow(colunas[i].capitalize(), QLabel(str(valor)))

                # 🔵 Cabeçalho DINÂMICO: guarde as labels como atributos e atualize via _refresh_header()
                self.lbl_situacao_pbh = QLabel("—")
                self.lbl_protocolo     = QLabel("—")
                self.lbl_numero_nf     = QLabel("—")
                self.lbl_status_nf = QLabel("—")

                form_layout.addRow("Situação do lote (PBH):", self.lbl_situacao_pbh)
                form_layout.addRow("Protocolo:",              self.lbl_protocolo)
                form_layout.addRow("Número NFS-e:",           self.lbl_numero_nf)
                form_layout.addRow("Status da NF:", self.lbl_status_nf)

                layout.addLayout(form_layout)

                # carrega estado atual do banco ao abrir o modal
                #self._refresh_header()

            self.df = pd.read_sql_query(
                "SELECT * FROM registros_financeiros WHERE Convenio = ? AND Competência = ?",
                conn, params=(convenio_nome, competencia)
            )
            conn.close()

            # ✅ Aplica o tratamento de duplicados
            self.df = tratar_duplicados(self.df)

            # ✅ Zera os valores duplicados
            if "Duplicado" in self.df.columns and "Valor Convenio" in self.df.columns:
                self.df.loc[self.df["Duplicado"] == True, "Valor Convenio"] = 0.0

            # ✅ Agora sim calcula o total
            qtd = len(self.df)
            total = pd.to_numeric(self.df['Valor Convenio'], errors='coerce').fillna(0).sum()


            layout.addWidget(QLabel(f"\nQuantidade de exames: {qtd}"))
            layout.addWidget(QLabel(f"Valor total a receber: R$ {total:,.2f}".replace(",", "X").replace(".", ",").replace("X", ".")))

            # 🔍 Verifica status da nota fiscal no banco
            conn = sqlite3.connect(CAMINHO_BANCO)
            cursor = conn.cursor()
            cursor.execute("""
                SELECT status FROM notas_emitidas_status
                WHERE convenio = ? AND competencia_servico = ?
                ORDER BY id DESC LIMIT 1
            """, (convenio_nome, competencia))
            resultado_status = cursor.fetchone()
            conn.close()

            status_nf = (resultado_status[0].strip().lower() if resultado_status else "pendente")
            tem_nf_emitida = (status_nf == "emitida")


            # guarde os estilos pra reusar em outros métodos
            self.estilo_botao = estilo_botao
            self.estilo_cinza = estilo_cinza

            # ---------- Verifica se já existe protocolo (lote enviado) ----------
            tem_protocolo = False
            situacao_lote = None
            boleto_flag = 0
            try:
                conn_chk = sqlite3.connect(CAMINHO_BANCO)
                cur_chk = conn_chk.cursor()
                cur_chk.execute("""
                    SELECT protocolo, COALESCE(situacao, 0), COALESCE(boleto_pago, 0)
                    FROM nfse_lotes
                    WHERE convenio_id = ? AND competencia = ?
                    ORDER BY id DESC LIMIT 1
                """, (self.id_convenio, self.competencia))
                row_chk = cur_chk.fetchone()
                if row_chk and row_chk[0]:
                    tem_protocolo = True
                    situacao_lote = int(row_chk[1] or 0)
                    boleto_flag   = int(row_chk[2] or 0)
            finally:
                try: conn_chk.close()
                except: pass            

            # ---------- Linha principal: ações ----------
            botoes_layout = QHBoxLayout()

            # crie os botões como ATRIBUTOS da instância
            self.btn_exportar = QPushButton("Exportar Excel")
            self.btn_emitir   = QPushButton("Emitir Nota Fiscal")
            self.btn_cancelar = QPushButton("Cancelar Nota Fiscal")

            self.btn_consultar_nfse = QPushButton("🔄 Consultar NFS-e")
            self.btn_consultar_nfse.setToolTip("Consulta a situação do lote e baixa a NFS-e se já estiver processada.")
            self.btn_consultar_nfse.clicked.connect(self.consultar_nfse)

            # habilitação/estilo
            pode_emitir    = not tem_protocolo          # travar emissão se já tem protocolo
            pode_cancelar  = tem_nf_emitida             # sua regra atual
            pode_consultar = tem_protocolo              # só consulta se já foi emitido (tem protocolo)

            self.btn_emitir.setEnabled(pode_emitir)
            self.btn_cancelar.setEnabled(pode_cancelar)
            self.btn_consultar_nfse.setEnabled(pode_consultar)

            self.btn_exportar.setStyleSheet(self.estilo_botao)
            self.btn_emitir.setStyleSheet(   self.estilo_botao if pode_emitir    else self.estilo_cinza)
            self.btn_cancelar.setStyleSheet( self.estilo_botao if pode_cancelar  else self.estilo_cinza)
            self.btn_consultar_nfse.setStyleSheet(self.estilo_botao if pode_consultar else self.estilo_cinza)       

            # conexões
            self.btn_emitir.clicked.connect(self.emitir_nota_fiscal)
            self.btn_cancelar.clicked.connect(self.cancelar_nota_fiscal)
            self.btn_exportar.clicked.connect(self.exportar_excel)

            # ordem na barra
            botoes_layout.addWidget(self.btn_cancelar)
            botoes_layout.addWidget(self.btn_exportar)
            botoes_layout.addWidget(self.btn_emitir)
            botoes_layout.addWidget(self.btn_consultar_nfse)
            layout.addLayout(botoes_layout)

            # ---------- Segunda linha: financeiro (pago / em aberto) ----------
            botoes_fin = QHBoxLayout()

            self.botao_marcar_pago = QPushButton("✔ Marcar como Pago")
            self.botao_marcar_em_aberto = QPushButton("❌ Marcar como Em Aberto")

            self.botao_marcar_pago.clicked.connect(self.marcar_como_pago)
            self.botao_marcar_em_aberto.clicked.connect(self.marcar_como_em_aberto)

            # habilita apenas se já houver NF emitida
            self._habilitar_pagamento(tem_nf_emitida)
            
            # ✅ Agora sim, aplique a sua regra por situação (4 = libera cancelar/pagamento e trava consulta)
            self._aplicar_regra_ui_por_situacao(situacao_lote, boleto_flag)
            
            self._aplicar_regra_ui_por_status_nf()

            botoes_fin.addWidget(self.botao_marcar_pago)
            botoes_fin.addWidget(self.botao_marcar_em_aberto)
            layout.addLayout(botoes_fin)
            
            # ---- Estados usados durante cancelamento/refresh automático ----
            self._timer_status = None
            self._numero_nf_cancelando = None
            self._cancelamento_em_andamento = False
            
            # ✅ Agora pode atualizar o cabeçalho e aplicar as regras (botões já existem)
            self._refresh_header()
            self._aplicar_regra_ui_por_status_nf()



        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao carregar faturamento: {e}")
            
    def _habilitar_pagamento(self, on: bool):
        self.botao_marcar_pago.setEnabled(on)
        self.botao_marcar_em_aberto.setEnabled(on)
        self.botao_marcar_pago.setStyleSheet(self.estilo_botao if on else self.estilo_cinza)
        self.botao_marcar_em_aberto.setStyleSheet(self.estilo_botao if on else self.estilo_cinza)
        
    def _aplicar_regra_ui_por_situacao(self, situacao, boleto_pago=0):
        # situação 4 => desabilita "Consultar NFS-e", habilita "Cancelar",
        # e deixa "Pago/Em aberto" mutuamente exclusivos.
        try:
            sit = int(situacao or 0)
        except:
            sit = 0
        is4 = (sit == 4)

        # Cancelar NF
        self.btn_cancelar.setEnabled(is4)
        self.btn_cancelar.setStyleSheet(self.estilo_botao if is4 else self.estilo_cinza)

        # Consultar NFS-e (desabilita quando 4)
        self.btn_consultar_nfse.setEnabled(not is4)
        self.btn_consultar_nfse.setStyleSheet(self.estilo_botao if not is4 else self.estilo_cinza)

        # Pagamento
        if not is4:
            # antes de processado, ambos off
            self.botao_marcar_pago.setEnabled(False)
            self.botao_marcar_em_aberto.setEnabled(False)
            self.botao_marcar_pago.setStyleSheet(self.estilo_cinza)
            self.botao_marcar_em_aberto.setStyleSheet(self.estilo_cinza)
        else:
            if int(boleto_pago or 0) == 1:
                # Já pago: só pode "Marcar como em aberto"
                self.botao_marcar_pago.setEnabled(False)
                self.botao_marcar_em_aberto.setEnabled(True)
                self.botao_marcar_pago.setStyleSheet(self.estilo_cinza)
                self.botao_marcar_em_aberto.setStyleSheet(self.estilo_botao)
            else:
                # Em aberto: só pode "Marcar como pago"
                self.botao_marcar_pago.setEnabled(True)
                self.botao_marcar_em_aberto.setEnabled(False)
                self.botao_marcar_pago.setStyleSheet(self.estilo_botao)
                self.botao_marcar_em_aberto.setStyleSheet(self.estilo_cinza)
                
                
    def _status_nf(self) -> str:
        """Lê o último status textual de notas_emitidas_status para este convênio/competência."""
        try:
            with sqlite3.connect(CAMINHO_BANCO) as conn:
                cur = conn.cursor()
                cur.execute("""
                    SELECT status
                    FROM notas_emitidas_status
                    WHERE convenio = ? AND competencia_servico = ?
                ORDER BY id DESC LIMIT 1
                """, (self.convenio_nome, self.competencia))
                row = cur.fetchone()
                return (row[0] or "").strip().lower() if row else "pendente"
        except Exception:
            return "pendente"

    # substitua as duas por esta única versão
    def _aplicar_regra_ui_por_status_nf(self, status_txt: str | None = None):
        # se ninguém passou, pega do label atual
        if status_txt is None:
            status_txt = getattr(self, "status_nf_atual", None)
            if status_txt is None and hasattr(self, "lbl_status_nf"):
                status_txt = self.lbl_status_nf.text()

        st = (status_txt or "").strip().lower()

        # mapeia pros nomes REAIS dos seus botões
        def set_enabled(emitir, consultar, cancelar, pago, aberto):
            self.btn_emitir.setEnabled(emitir)
            self.btn_consultar_nfse.setEnabled(consultar)
            self.btn_cancelar.setEnabled(cancelar)
            self.botao_marcar_pago.setEnabled(pago)
            self.botao_marcar_em_aberto.setEnabled(aberto)

            self.btn_emitir.setStyleSheet(self.estilo_botao if emitir else self.estilo_cinza)
            self.btn_consultar_nfse.setStyleSheet(self.estilo_botao if consultar else self.estilo_cinza)
            self.btn_cancelar.setStyleSheet(self.estilo_botao if cancelar else self.estilo_cinza)
            self.botao_marcar_pago.setStyleSheet(self.estilo_botao if pago else self.estilo_cinza)
            self.botao_marcar_em_aberto.setStyleSheet(self.estilo_botao if aberto else self.estilo_cinza)

        # ====== SUA REGRA "CORRETA" ======
        is_inicial = (st in {"", "—", "nao emitida", "não emitida", "sem emissao", "sem emissão", "pendente"})
        is_processando = (
            "processando" in st or st.startswith("3 -")
            or st.startswith("1 -") or st.startswith("2 -")
            or "solicitado" in st or "análise" in st or "analise" in st
        )
        is_emitida = ("emitida" in st) and not is_processando
        is_cancelada = ("cancelad" in st)
        is_erro = ("erro" in st or st.startswith("5 -"))

        if is_inicial or is_erro:
            # Estado inicial: só emitir
            set_enabled(True, False, False, False, False); return
        if is_processando:
            # Processando: só consultar
            set_enabled(False, True, False, False, False); return
        if is_emitida:
            # Emitida: cancelar + marcar pago/aberto
            set_enabled(False, False, True, True, True); return
        if is_cancelada:
            # Cancelada: só emitir
            set_enabled(True, False, False, False, False); return

        # fallback conservador = inicial
        set_enabled(True, False, False, False, False)




    def _recarregar_botoes(self):
        """Atualiza a habilitação dos botões sem fechar o diálogo."""
        if not all(hasattr(self, a) for a in (
            "btn_emitir", "btn_cancelar", "btn_consultar_nfse",
            "botao_marcar_pago", "botao_marcar_em_aberto"
        )):
            return

        # ---- Lê protocolo (existe lote?) ----
        tem_protocolo = False
        try:
            conn = sqlite3.connect(CAMINHO_BANCO)
            cur = conn.cursor()
            cur.execute("""
                SELECT protocolo FROM nfse_lotes
                WHERE convenio_id = ? AND competencia = ?
                ORDER BY id DESC LIMIT 1
            """, (self.id_convenio, self.competencia))
            row = cur.fetchone()
            tem_protocolo = bool(row and row[0])
        finally:
            try: conn.close()
            except: pass

        # ---- Lê status textual ----
        try:
            conn = sqlite3.connect(CAMINHO_BANCO)
            cur = conn.cursor()
            cur.execute("""
                SELECT status
                FROM notas_emitidas_status
                WHERE convenio = ? AND competencia_servico = ?
                ORDER BY id DESC LIMIT 1
            """, (self.convenio_nome, self.competencia))
            r = cur.fetchone()
            status_txt = (r[0].strip().lower() if r and r[0] else "pendente")
        finally:
            try: conn.close()
            except: pass

        # 🔶 1) Cancelamento em curso -> tudo travado
        if ("cancelamento solicitado" in status_txt) or ("cancelamento em análise" in status_txt):
            for w in (self.btn_emitir, self.btn_cancelar, self.btn_consultar_nfse):
                w.setEnabled(False); w.setStyleSheet(self.estilo_cinza)
            self._habilitar_pagamento(False)
            return

        # 🔷 2) Cancelada -> TRATAR COMO “SEM PROTOCOLO ATIVO”
        if "cancelada" in status_txt:
            # habilita reemissão e consulta; bloqueia cancelamento/pagamento
            self.btn_emitir.setEnabled(True);  self.btn_emitir.setStyleSheet(self.estilo_botao)
            self.btn_consultar_nfse.setEnabled(False); self.btn_consultar_nfse.setStyleSheet(self.estilo_cinza)
            self.btn_cancelar.setEnabled(False); self.btn_cancelar.setStyleSheet(self.estilo_cinza)
            self._habilitar_pagamento(False)
            return

        # ---- 3) Regra padrão (emitida x não emitida) ----
        tem_nf_emitida = (status_txt == "emitida")
        pode_emitir    = not tem_protocolo
        pode_cancelar  = tem_nf_emitida
        pode_consultar = tem_protocolo

        self.btn_emitir.setEnabled(pode_emitir)
        self.btn_cancelar.setEnabled(pode_cancelar)
        self.btn_consultar_nfse.setEnabled(pode_consultar)

        self.btn_emitir.setStyleSheet(self.estilo_botao if pode_emitir else self.estilo_cinza)
        self.btn_cancelar.setStyleSheet(self.estilo_botao if pode_cancelar else self.estilo_cinza)
        self.btn_consultar_nfse.setStyleSheet(self.estilo_botao if pode_consultar else self.estilo_cinza)

        # Pagamento segue regra “emitida”
        self._habilitar_pagamento(tem_nf_emitida)

        # 📌 4) Sobrepor com a situação do lote (somente se NÃO estiver cancelada)
        try:
            with sqlite3.connect(CAMINHO_BANCO) as conn_:
                cur_ = conn_.cursor()
                cur_.execute("""
                    SELECT COALESCE(situacao,0), COALESCE(boleto_pago,0)
                    FROM nfse_lotes
                    WHERE convenio_id = ? AND competencia = ?
                    ORDER BY id DESC LIMIT 1
                """, (self.id_convenio, self.competencia))
                row_ = cur_.fetchone()
                situacao_lote = row_[0] if row_ else None
                boleto_flag   = row_[1] if row_ else 0
                self._aplicar_regra_ui_por_situacao(situacao_lote, boleto_flag)
        except Exception:
            pass


     
        
        
    def _refresh_header(self):
        protocolo = None
        situacao = None
        numero_nf = None
        status_nf_txt = None
        comp_emissao_pbh = None  # ⇦ nova

        try:
            with sqlite3.connect(CAMINHO_BANCO) as conn:
                cur = conn.cursor()

                # Lote + situação
                cur.execute("""
                    SELECT protocolo, situacao
                    FROM nfse_lotes
                    WHERE convenio_nome = ? AND competencia = ?
                ORDER BY id DESC
                    LIMIT 1
                """, (self.convenio_nome, self.competencia))
                row = cur.fetchone()
                if row:
                    protocolo, situacao = row[0], row[1]

                # Número da NFS-e (se já existir)
                if protocolo:
                    cur.execute("""
                        SELECT numero_nfse
                        FROM nfse_notas
                        WHERE protocolo = ?
                    ORDER BY id DESC
                        LIMIT 1
                    """, (protocolo,))
                    r2 = cur.fetchone()
                    if r2:
                        numero_nf = r2[0]

                # Status + competência da emissão (PBH) gravados em notas_emitidas_status
                cur.execute("""
                    SELECT status, competencia_nfse
                    FROM notas_emitidas_status
                    WHERE convenio = ? AND competencia_servico = ?
                ORDER BY id DESC
                    LIMIT 1
                """, (self.convenio_nome, self.competencia))
                r3 = cur.fetchone()
                if r3:
                    status_nf_txt = r3[0]
                    comp_emissao_pbh = r3[1]

                # Fallback: se não veio competencia_nfse, tenta derivar da data_emissao_nfse
                if not comp_emissao_pbh and numero_nf:
                    cur.execute("""
                        SELECT data_emissao_nfse
                        FROM nfse_notas
                        WHERE numero_nfse = ?
                    ORDER BY id DESC
                        LIMIT 1
                    """, (str(numero_nf),))
                    r4 = cur.fetchone()
                    if r4 and r4[0]:
                        d = str(r4[0])           # esperado YYYY-MM-DD...
                        if len(d) >= 7 and "-" in d:
                            comp_emissao_pbh = f"{d[5:7]}/{d[0:4]}"
        except Exception:
            pass

        # Atualiza cabeçalho
        self.lbl_protocolo.setText(protocolo or "—")
        if situacao is None and protocolo:
            self.lbl_situacao_pbh.setText("Aguardando processamento")
        else:
            self.lbl_situacao_pbh.setText(formatar_situacao(situacao))
        self.lbl_numero_nf.setText(str(numero_nf) if numero_nf else "—")

        # Status + regras de UI
        self.lbl_status_nf.setText(status_nf_txt or "—")
        self._pintar_status_nf(self.lbl_status_nf.text())
        self._aplicar_regra_ui_por_status_nf(self.lbl_status_nf.text())

        # Competência da emissão (PBH) — só atualiza se a label existir
        if hasattr(self, "lbl_competencia_nfse"):
            self.lbl_competencia_nfse.setText(comp_emissao_pbh or "—")


        
        
    def _ler_status_nf_banco(self, numero_nf=None):
        """Lê o status textual mais recente em notas_emitidas_status.
        Se numero_nf for passado, filtra por ele.
        """
        status_nf_txt = None
        try:
            with sqlite3.connect(CAMINHO_BANCO) as conn:
                cur = conn.cursor()
                if numero_nf:
                    cur.execute("""
                        SELECT status
                        FROM notas_emitidas_status
                        WHERE convenio = ? AND competencia = ? AND numero_nfse = ?
                        ORDER BY id DESC LIMIT 1
                    """, (self.convenio_nome, self.competencia, str(numero_nf)))
                else:
                    cur.execute("""
                        SELECT status
                        FROM notas_emitidas_status
                        WHERE convenio = ? AND competencia = ?
                        ORDER BY id DESC LIMIT 1
                    """, (self.convenio_nome, self.competencia))
                row_status = cur.fetchone()
                if row_status:
                    status_nf_txt = row_status[0]
        except Exception:
            pass
        return status_nf_txt          
        
        
    def _pintar_status_nf(self, texto: str):
        t = (texto or "").lower()
        cor = "#7f8c8d"  # cinza
        if "cancelada" in t:
            cor = "#2ecc71"  # verde
        elif "cancelamento" in t:  # solicitado / em análise
            cor = "#e67e22"  # laranja
        elif "emitida" in t or "processado (ok)" in t:
            cor = "#2980b9"  # azul
        elif "erro" in t:
            cor = "#e74c3c"  # vermelho
        self.lbl_status_nf.setStyleSheet(f"color: {cor}; font-weight: bold;")
        
    def _aplicar_fluxo_geral(self):
        """Habilita/Desabilita botões conforme estado (inicial, 1/2/3, emitida, cancelada, cancelando)."""
        if not all(hasattr(self, a) for a in (
            "btn_emitir", "btn_cancelar", "btn_consultar_nfse",
            "botao_marcar_pago", "botao_marcar_em_aberto"
        )):
            return

        status_txt = "pendente"
        situacao_lote = None
        boleto_flag = 0
        tem_protocolo = False

        try:
            with sqlite3.connect(CAMINHO_BANCO) as conn:
                cur = conn.cursor()
                # status textual da competência de serviço
                cur.execute("""
                    SELECT status
                    FROM notas_emitidas_status
                    WHERE convenio = ? AND competencia_servico = ?
                    ORDER BY id DESC LIMIT 1
                """, (self.convenio_nome, self.competencia))
                r = cur.fetchone()
                status_txt = (r[0].strip().lower() if r and r[0] else "pendente")

                # lote + boleto
                cur.execute("""
                    SELECT protocolo, COALESCE(situacao,0), COALESCE(boleto_pago,0)
                    FROM nfse_lotes
                    WHERE convenio_id = ? AND competencia = ?
                    ORDER BY id DESC LIMIT 1
                """, (self.id_convenio, self.competencia))
                row = cur.fetchone()
                if row:
                    tem_protocolo = bool(row[0])
                    situacao_lote = int(row[1] or 0)
                    boleto_flag   = int(row[2] or 0)
        except Exception:
            pass

        def on(w, ok: bool):
            w.setEnabled(ok)
            w.setStyleSheet(self.estilo_botao if ok else self.estilo_cinza)

        # Cancelamento em andamento
        if "cancelamento solicitado" in status_txt or "cancelamento em an" in status_txt:
            on(self.btn_emitir, False)
            on(self.btn_consultar_nfse, False)
            on(self.btn_cancelar, False)
            self._habilitar_pagamento(False)
            return

        # Cancelada -> estado inicial
        if "cancelada" in status_txt:
            on(self.btn_emitir, True)
            on(self.btn_consultar_nfse, True)   # pode deixar ativo
            on(self.btn_cancelar, False)
            self._habilitar_pagamento(False)
            return

        # Emitida / OK -> cancelar e pagamento
        if ("emitida" in status_txt) or ("processado" in status_txt and "ok" in status_txt) or (situacao_lote == 4):
            on(self.btn_emitir, False)
            on(self.btn_consultar_nfse, False)
            on(self.btn_cancelar, True)
            if boleto_flag == 1:
                self.botao_marcar_pago.setEnabled(False); self.botao_marcar_pago.setStyleSheet(self.estilo_cinza)
                self.botao_marcar_em_aberto.setEnabled(True); self.botao_marcar_em_aberto.setStyleSheet(self.estilo_botao)
            else:
                self.botao_marcar_pago.setEnabled(True); self.botao_marcar_pago.setStyleSheet(self.estilo_botao)
                self.botao_marcar_em_aberto.setEnabled(False); self.botao_marcar_em_aberto.setStyleSheet(self.estilo_cinza)
            return

        # Processando 1/2/3 -> só consultar
        if situacao_lote in (1, 2, 3):
            on(self.btn_emitir, False)
            on(self.btn_cancelar, False)
            self._habilitar_pagamento(False)
            on(self.btn_consultar_nfse, True)
            return

        # Estado inicial (nunca emitida / sem protocolo)
        on(self.btn_emitir, True)
        on(self.btn_consultar_nfse, False)
        on(self.btn_cancelar, False)
        self._habilitar_pagamento(False)
        return
     
    
        
    def _iniciar_watch_status_cancelamento(self, numero_nf):
        """Começa a checar no banco, a cada 5s, se o cancelamento foi confirmado."""
        self._numero_nf_cancelando = str(numero_nf)
        self._cancelamento_em_andamento = True

        if self._timer_status is None:
            self._timer_status = QTimer(self)
            self._timer_status.timeout.connect(self._tick_watch_status)

        # dispara já e depois a cada 5s
        self._tick_watch_status()
        self._timer_status.start(5000)

    def _parar_watch_status_cancelamento(self):
        self._cancelamento_em_andamento = False
        if self._timer_status:
            self._timer_status.stop()

    def _tick_watch_status(self):
        status_txt = self._ler_status_nf_banco(numero_nf=self._numero_nf_cancelando)
        if status_txt:
            self.lbl_status_nf.setText(status_txt)
            self._pintar_status_nf(status_txt)
            self._aplicar_regra_ui_por_status_nf(status_txt)

            t = status_txt.lower()
            if ("cancelada" in t) or ("erro" in t):
                self._parar_watch_status_cancelamento()
                self._refresh_header()
                self._recarregar_botoes()
                # ⬇️ garante que a grade principal também atualize para o status final
                self.atualizar_tabela_signal.emit()
        else:
            self._refresh_header()


            
            
    def consultar_nfse(self):
        try:
            if not hasattr(self, "id_convenio"):
                QMessageBox.warning(self, "Aviso", "ID do convênio não encontrado.")
                return

            from gerar_nfse_xml import consultar_e_atualizar_lote_por_convenio
            res = consultar_e_atualizar_lote_por_convenio(
                id_convenio=self.id_convenio,
                competencia=self.competencia
            )

            protocolo    = res.get("protocolo") or "—"
            situacao     = res.get("situacao")
            situacao_txt = res.get("situacao_txt") or "—"
            numeros_nf   = res.get("numeros_nf") or []

            # lê o boleto_pago atual no lote
            try:
                with sqlite3.connect(CAMINHO_BANCO) as c_:
                    k_ = c_.cursor()
                    k_.execute("""
                        SELECT COALESCE(boleto_pago,0)
                        FROM nfse_lotes
                        WHERE convenio_id = ? AND competencia = ?
                        ORDER BY id DESC LIMIT 1
                    """, (self.id_convenio, self.competencia))
                    boleto_flag = (k_.fetchone() or [0])[0]
            except:
                boleto_flag = 0

            # aplica a regra UMA vez (com situacao já definida)
            self._aplicar_regra_ui_por_situacao(situacao, boleto_flag)

            linhas = [f"Protocolo: {protocolo}",
                    f"Situação do lote (PBH): {situacao_txt}"]
            if numeros_nf:
                linhas.append(f"Número(s) da NFS-e: {', '.join(map(str, numeros_nf))}")

            QMessageBox.information(self, "Consulta realizada", "\n".join(linhas))
            self._refresh_header()
            self._recarregar_botoes()
            self._aplicar_regra_ui_por_status_nf()
            self.atualizar_tabela_signal.emit()

        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Falha na consulta: {e}")




    def cancelar_nota_fiscal(self):
        try:
            if not hasattr(self, 'id_convenio'):
                QMessageBox.critical(self, "Erro", "ID do convênio não encontrado.")
                return
            if not hasattr(self, 'competencia') or not str(self.competencia).strip():
                QMessageBox.critical(self, "Erro", "Competência não informada na tela.")
                return

            # 1) Descobre automaticamente o número da NF
            numero_nfse = _obter_numero_nfse_para_cancelar(self.id_convenio, self.competencia)
            if not numero_nfse:
                QMessageBox.warning(self, "Não encontrado",
                                    "Não encontrei número de NFS-e para este convênio/competência.")
                return

            # 2) Pergunta o motivo do cancelamento
            #    (ordena pelos códigos 1..5 e deixa "2 - Serviço não prestado" como default)
            itens_ordenados = sorted(CANCEL_MOTIVOS.items(), key=lambda kv: int(kv[0]))
            opcoes = [f"{k} - {v}" for k, v in itens_ordenados]
            idx_default = next((i for i, (k, _) in enumerate(itens_ordenados) if k == "2"), 0)

            escolha, ok = QInputDialog.getItem(
                self, "Motivo do cancelamento", "Selecione o motivo:",
                opcoes, idx_default, False
            )
            if not ok:
                return
            m = re.match(r"(\d+)", escolha or "")
            codigo_cancelamento = m.group(1) if m else "2"

            # 3) Confirmação
            confirmacao = QMessageBox.question(
                self,
                "Confirmar Cancelamento",
                f"Cancelar a NFS-e nº {numero_nfse}?\nMotivo: {CANCEL_MOTIVOS.get(codigo_cancelamento, '—')}",
                QMessageBox.Yes | QMessageBox.No
            )
            if confirmacao != QMessageBox.Yes:
                return

            # 4) Dispara o cancelamento (agora passando o código!)
            sucesso, mensagem = cancelar_nfse_por_convenio(
                numero_nfse=numero_nfse,
                id_convenio=self.id_convenio,
                competencia_servico=self.competencia,
                codigo_cancelamento=codigo_cancelamento
            )

            if sucesso:
                self.lbl_status_nf.setText("Cancelamento solicitado")
                self._pintar_status_nf("Cancelamento solicitado")
                self._aplicar_regra_ui_por_status_nf("Cancelamento solicitado")
                self._iniciar_watch_status_cancelamento(numero_nfse)
                self.atualizar_tabela_signal.emit()
                QMessageBox.information(self, "Cancelamento", f"✅ {mensagem}")
            else:
                QMessageBox.critical(self, "Erro", f"❌ {mensagem}")

        except Exception as e:
            QMessageBox.critical(self, "Erro", f"❌ Falha ao cancelar nota fiscal:\n{str(e)}")


    def emitir_nota_fiscal(self):
        try:
            if not hasattr(self, 'id_convenio'):
                QMessageBox.critical(self, "Erro", "ID do convênio não encontrado.")
                return

            pasta_saida = "notas_emitidas"
            os.makedirs(pasta_saida, exist_ok=True)
            nome_arquivo = f"NF_{self.convenio_nome}_{self.competencia.replace('/', '-')}.xml"
            caminho_xml = os.path.join(pasta_saida, nome_arquivo)

            ret = gerar_xml_nfse_por_convenio(
                id_convenio=self.id_convenio,
                competencia=self.competencia,
                caminho_saida=caminho_xml
            )

            # interpreta o retorno novo (dict) ou antigo (bool/str)
            ok = False
            msg_sucesso = ""
            msg_erro = None

            if isinstance(ret, dict):
                status = (ret.get("status") or "").lower()
                ok = status in ("emitida", "gerado")
                lotes = ret.get("lotes") or []
                msg_sucesso = "Lote(s) gerado(s)/enviado(s):\n" + "\n".join(lotes)
                if not ok:
                    msg_erro = f"Status: {ret.get('status')}\nDetalhes: {ret}"
            elif ret is True:
                ok = True
                msg_sucesso = f"XML salvo em:\n{caminho_xml}"
            else:
                msg_erro = str(ret)

            self.atualizar_tabela_signal.emit()  # atualiza a tela principal

            if ok:
                # ✅ GERA O EXCEL AUTOMATICAMENTE NA PASTA "Nota Fiscal"
                self.exportar_excel()

                QMessageBox.information(self, "Sucesso", f"✅ Nota Fiscal gerada/enviada.\n\n{msg_sucesso}")
                self._refresh_header()
                self._recarregar_botoes()
            else:
                QMessageBox.critical(self, "Erro ao Enviar", f"❌ Nota não enviada.\n\n{msg_erro or 'Sem detalhes.'}")

        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao gerar Nota Fiscal:\n{str(e)}")


    def exportar_excel(self, mostrar_msg=True):
        df = self.df.copy()

        if df.empty:
            self.atualizar_tabela_signal.emit()
            QMessageBox.information(self, "Aviso", f"Nenhum registro encontrado para {self.convenio_nome} na competência {self.competencia}.")
            return

        try:
            nome = self.convenio_nome
            nome_maiusculo = nome.upper()
            competencia = self.competencia

            df = tratar_duplicados(df)

            # 🟡 Zera "Valor Convenio" se duplicado
            if "Duplicado" in df.columns and "Valor Convenio" in df.columns:
                df.loc[df["Duplicado"] == True, "Valor Convenio"] = 0

            if "Médico" in df.columns:
                df["Médico"] = df["Médico"].str.extract(r"^(.*?)(?:\s*-\s*CRM|$)")[0].str.strip()

            colunas_para_remover = ["Convenio"]
            df.drop(columns=[col for col in colunas_para_remover if col in df.columns], inplace=True)

            # monta caminho: notas_emitidas/<CONVENIO>_<MM-AAAA>/Nota Fiscal/...
            safe_conv = re.sub(r"[^0-9A-Za-z_-]+", "_", nome.strip())
            safe_comp = competencia.replace("/", "-").strip()

            dest_dir = Path("notas_emitidas") / f"{safe_conv}_{safe_comp}" / "Nota Fiscal"
            dest_dir.mkdir(parents=True, exist_ok=True)

            caminho = str(dest_dir / f"Faturamento_{safe_conv}_{safe_comp}.xlsx")


            colunas_visiveis = [
                col for col in df.columns if col not in [
                    'Cod Registro', 'Duplicado', 'NaoDuplicadoOIT',
                    'Competência', 'Data Cadastro', 'Data Recebimento', 'Valor Médico', 'Competencia'
                ]
            ]

            writer = pd.ExcelWriter(caminho, engine='xlsxwriter')
            workbook = writer.book
            df.to_excel(writer, index=False, sheet_name='Registros', columns=colunas_visiveis, startrow=2)
            worksheet = writer.sheets['Registros']

            # Estilos
            azul_cabecalho = '#B7D6F4'
            header_format = workbook.add_format({'bold': True, 'bg_color': '#D9D9D9', 'border': 1, 'align': 'center', 'valign': 'vcenter'})
            center_format = workbook.add_format({'align': 'center', 'valign': 'vcenter'})
            money_format = workbook.add_format({'num_format': '"R$" #,##0.00', 'align': 'center'})
            amarelo_center = workbook.add_format({'align': 'center', 'bg_color': '#FFF59D'})
            amarelo_money = workbook.add_format({'num_format': '"R$" #,##0.00', 'align': 'center', 'bg_color': '#FFF59D'})
            verde_center = workbook.add_format({'align': 'center', 'bg_color': '#C8E6C9'})
            verde_money = workbook.add_format({'num_format': '"R$" #,##0.00', 'align': 'center', 'bg_color': '#C8E6C9'})
            bold_blue = workbook.add_format({'bold': True, 'font_color': 'black', 'bg_color': azul_cabecalho, 'border': 1, 'align': 'center'})
            bold_right = workbook.add_format({'bold': True, 'align': 'right', 'bg_color': azul_cabecalho, 'border': 1})
            value_right = workbook.add_format({'align': 'right', 'bg_color': azul_cabecalho, 'border': 1})
            money_right = workbook.add_format({'num_format': '"R$" #,##0.00', 'align': 'right', 'bg_color': azul_cabecalho, 'border': 1})
            logo_background = workbook.add_format({'bg_color': azul_cabecalho, 'border': 1})

            total_laudos = len(df)
            total_valor = df["Valor Convenio"].apply(
                lambda v: float(str(v).replace("R$", "").replace(".", "").replace(",", ".")) if isinstance(v, str) else v
            ).sum()

            # Cabeçalho ajustado
            worksheet.merge_range("A1:A2", "", logo_background)
            worksheet.merge_range("B1:E1", f"FATURAMENTO {nome_maiusculo}", bold_blue)
            worksheet.merge_range("B2:E2", f"Competência: {competencia}", bold_blue)
            worksheet.write("F1", "TOTAL", bold_right)
            worksheet.write("G1", total_valor, money_right)
            worksheet.write("F2", "LAUDOS", bold_right)
            worksheet.write("G2", total_laudos, value_right)

            # Logo
            caminho_logo = os.path.join("icones", "medical_laudos_logo.png")
            if os.path.exists(caminho_logo):
                worksheet.insert_image("A1", caminho_logo, {'x_scale': 0.45, 'y_scale': 0.45, 'x_offset': 2, 'y_offset': 2})

            # Cabeçalhos da tabela (linha 3)
            for col_num, nome_coluna in enumerate(colunas_visiveis):
                worksheet.write(2, col_num, nome_coluna, header_format)
                largura_max = max(
                    [len(str(cell)) for cell in [nome_coluna] + df[nome_coluna].astype(str).tolist()]
                ) + 2
                worksheet.set_column(col_num, col_num, largura_max)

            # Ativar filtro automático na linha dos cabeçalhos
            worksheet.autofilter(2, 0, 2 + len(df), len(colunas_visiveis) - 1)

            def str_to_bool(valor):
                return str(valor).strip().lower() in ["1", "true", "verdadeiro"]

            for row_num, row_data in df.iterrows():
                duplicado = str_to_bool(row_data.get("Duplicado", ""))
                nao_oit = str_to_bool(row_data.get("NaoDuplicadoOIT", ""))

                for col_num, nome_coluna in enumerate(colunas_visiveis):
                    valor = row_data[nome_coluna]
                    if duplicado:
                        formato = amarelo_money if nome_coluna == 'Valor Convenio' else amarelo_center
                    elif nao_oit:
                        formato = verde_money if nome_coluna == 'Valor Convenio' else verde_center
                    else:
                        formato = money_format if nome_coluna == 'Valor Convenio' else center_format

                    linha_excel = row_num + 3
                    if nome_coluna == 'Valor Convenio':
                        try:
                            worksheet.write_number(linha_excel, col_num, float(valor), formato)
                        except:
                            worksheet.write(linha_excel, col_num, str(valor), formato)
                    else:
                        worksheet.write(linha_excel, col_num, str(valor), formato)

            writer.close()
            QMessageBox.information(self, "Sucesso", f"Arquivo exportado com sucesso para:\n{caminho}")

        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao exportar Excel:\n{e}")

    def marcar_como_pago(self):
        try:
            with sqlite3.connect(CAMINHO_BANCO, timeout=10) as conn:
                cur = conn.cursor()
                cur.execute("""
                    UPDATE nfse_lotes
                    SET boleto_pago = 1
                    WHERE id = (
                        SELECT id FROM nfse_lotes
                        WHERE convenio_id = ? AND competencia = ?
                        ORDER BY id DESC LIMIT 1
                    )
                """, (self.id_convenio, self.competencia))
                afetadas = cur.rowcount

            if afetadas == 0:
                QMessageBox.warning(self, "Aviso", "Nenhum lote encontrado para este convênio/competência.")
                return

            # situação 4 + agora pago
            self._aplicar_regra_ui_por_situacao(4, 1)
            self._refresh_header()
            self.atualizar_tabela_signal.emit()
            QMessageBox.information(self, "Pronto", "Boleto marcado como pago.")

        except sqlite3.OperationalError as e:
            QMessageBox.critical(self, "Erro", f"Banco ocupado. Tente novamente em instantes.\n\n{e}")
        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao atualizar: {e}")



    def marcar_como_em_aberto(self):
        try:
            with sqlite3.connect(CAMINHO_BANCO, timeout=10) as conn:
                cur = conn.cursor()
                cur.execute("""
                    UPDATE nfse_lotes
                    SET boleto_pago = 0
                    WHERE id = (
                        SELECT id FROM nfse_lotes
                        WHERE convenio_id = ? AND competencia = ?
                        ORDER BY id DESC LIMIT 1
                    )
                """, (self.id_convenio, self.competencia))
                afetadas = cur.rowcount

            if afetadas == 0:
                QMessageBox.warning(self, "Aviso", "Nenhum lote encontrado para este convênio/competência.")
                return

            # situação 4 + agora em aberto
            self._aplicar_regra_ui_por_situacao(4, 0)
            self._refresh_header()
            self.atualizar_tabela_signal.emit()
            QMessageBox.information(self, "Pronto", "Boleto marcado como em aberto.")

        except sqlite3.OperationalError as e:
            QMessageBox.critical(self, "Erro", f"Banco ocupado. Tente novamente em instantes.\n\n{e}")
        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao atualizar: {e}")


class ModuloFaturamentoConvenios(QWidget):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.layout = QVBoxLayout(self)

        # ✅ Estilo unificado para os botões (como os de outros módulos)
        estilo_botao = """
            QPushButton {
                background-color: #3498db;
                color: white;
                padding: 6px 12px;
                font-weight: bold;
                border-radius: 4px;
            }
            QPushButton:hover {
                background-color: #2980b9;
            }
        """

        # Título
        titulo = QLabel("Faturamento de Convênios")
        titulo.setAlignment(Qt.AlignCenter)
        titulo.setStyleSheet("font-size: 20px; font-weight: bold; margin-bottom: 15px;")
        self.layout.addWidget(titulo)

        # Filtros
        self.combo_mes = QComboBox()
        self.combo_ano = QComboBox()
        self.botao_carregar = QPushButton("Carregar Convênios")

        self.combo_mes.addItems([f"{i:02d}" for i in range(1, 13)])
        self.combo_ano.addItems(["2025", "2026", "2027", 
                                 "2028", "2029", "2030", "2031", "2032", "2033"])

        seletor_layout = QHBoxLayout()
        seletor_layout.setSpacing(10)
        seletor_layout.addWidget(QLabel("Mês:"))
        seletor_layout.addWidget(self.combo_mes)
        seletor_layout.addWidget(QLabel("Ano:"))
        seletor_layout.addWidget(self.combo_ano)
        seletor_layout.addSpacerItem(QSpacerItem(40, 20, QSizePolicy.Expanding, QSizePolicy.Minimum))
        seletor_layout.addWidget(self.botao_carregar)

        self.layout.addLayout(seletor_layout)

        # Campo de busca
        self.input_busca = QLineEdit()
        self.input_busca.setPlaceholderText("🔍 Buscar por nome do convênio...")
        self.input_busca.textChanged.connect(self.aplicar_filtro_busca)
        self.layout.addWidget(self.input_busca)

        # Tabela
        self.tabela = QTableWidget()
        self.tabela.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)
        self.tabela.setAlternatingRowColors(True)
        self.tabela.setSelectionBehavior(QTableWidget.SelectRows)
        self.tabela.setSelectionMode(QTableWidget.SingleSelection)
        self.tabela.horizontalHeader().setStretchLastSection(True)
        self.tabela.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        self.layout.addWidget(self.tabela)


        # Novo layout de rodapé com totais e botão
        rodape_layout = QHBoxLayout()

        self.label_totais = QLabel("Convênios: 0 | Total de exames: 0 | Total a receber: R$ 0,00 | ✅ Total Pago: R$ 0,00 | ❌ Total em Aberto: R$ 0,00")
        self.label_totais.setStyleSheet("""
            font-weight: bold;
            padding: 6px 12px;
            color: #2c3e50;
        """)

        self.botao_exibir = QPushButton("Exibir Faturamento do Convênio")
        self.botao_exibir.setEnabled(False)
        self.botao_exibir.setStyleSheet(estilo_botao)

        rodape_layout.addWidget(self.label_totais)
        rodape_layout.addSpacerItem(QSpacerItem(40, 20, QSizePolicy.Expanding, QSizePolicy.Minimum))
        rodape_layout.addWidget(self.botao_exibir)

        self.layout.addLayout(rodape_layout)

        self.botao_carregar.clicked.connect(self.carregar_convenios_por_competencia)
        self.botao_exibir.clicked.connect(self.exibir_modal_faturamento)
        self.tabela.cellClicked.connect(self.habilitar_botao)

        
        self.botao_carregar.setStyleSheet(estilo_botao)
        self.botao_exibir.setStyleSheet(estilo_botao)
        
    def _obter_status_nf_convenio(self, convenio_nome: str) -> str:
        try:
            with sqlite3.connect(CAMINHO_BANCO) as conn:
                cur = conn.cursor()
                cur.execute("""
                    SELECT status
                    FROM notas_emitidas_status
                    WHERE convenio = ? AND competencia_servico = ?
                ORDER BY id DESC LIMIT 1
                """, (convenio_nome, self.competencia))

                row = cur.fetchone()
                return row[0] if row else "Pendente"
        except Exception:
            return "Pendente"

    def _obter_numero_nf_convenio(self, convenio_nome: str) -> str:
        try:
            with sqlite3.connect(CAMINHO_BANCO) as conn:
                cur = conn.cursor()
                cur.execute("""
                    SELECT n.numero_nfse
                    FROM nfse_notas n
                    JOIN nfse_lotes l ON l.protocolo = n.protocolo
                    WHERE l.convenio_nome = ? AND l.competencia = ?
                    ORDER BY n.id DESC
                    LIMIT 1
                """, (convenio_nome, self.competencia))
                row = cur.fetchone()
                return row[0] if row else ""
        except Exception:
            return ""



    def carregar_convenios_por_competencia(self):
        self.df_completo = pd.DataFrame()  # 🧼 Limpa dados antigos

        self.botao_exibir.setEnabled(False)
        mes = self.combo_mes.currentText()
        ano = self.combo_ano.currentText()
        self.competencia = f"{mes}/{ano}"

        self.df_completo = pd.DataFrame()  # 🧼 Limpa qualquer cache antigo


        try:
            conn = sqlite3.connect(CAMINHO_BANCO)
            df = pd.read_sql_query(
                "SELECT * FROM registros_financeiros WHERE Competência = ?", conn, params=(self.competencia,)
            )

            df = tratar_duplicados(df)
            if "Duplicado" in df.columns and "Valor Convenio" in df.columns:
                df.loc[df["Duplicado"] == True, "Valor Convenio"] = 0.0

            df["Valor Convenio"] = pd.to_numeric(df["Valor Convenio"], errors='coerce').fillna(0)
            df_grouped = df.groupby("Convenio").agg(
                Quantidade=("Convenio", "count"),
                Total=("Valor Convenio", "sum")
            ).reset_index()

           
            # 🔵 Situação do lote (PBH) por convênio
            df_situacoes = carregar_situacoes_por_competencia(self.competencia)

            def obter_situacao(convenio_nome):
                if df_situacoes.empty or convenio_nome not in df_situacoes.index:
                    return None
                return df_situacoes.loc[convenio_nome].get("situacao")

            def obter_protocolo(convenio_nome):
                if df_situacoes.empty or convenio_nome not in df_situacoes.index:
                    return ""
                return df_situacoes.loc[convenio_nome].get("protocolo", "")

            df_grouped["Situação Lote"] = df_grouped["Convenio"].apply(lambda n: formatar_situacao(obter_situacao(n)))
            df_grouped["Protocolo"] = df_grouped["Convenio"].apply(obter_protocolo)



            def obter_status(convenio_nome):
                mapa = {
                    1: "1 - Não recebido",
                    2: "2 - Não processado",
                    3: "3 - Processando",
                    4: "4 - Processado (OK)",
                    5: "5 - Processado com erro",
                    None: "⏳ Aguardando"
                }
                try:
                    conn_ = sqlite3.connect(CAMINHO_BANCO)
                    cur = conn_.cursor()
                    cur.execute("""
                        SELECT situacao
                        FROM nfse_lotes
                        WHERE convenio_nome = ? AND competencia = ?
                        ORDER BY id DESC LIMIT 1
                    """, (convenio_nome, self.competencia))
                    row = cur.fetchone()
                    conn_.close()
                    sit = row[0] if row else None
                    return mapa.get(sit, "⏳ Aguardando")
                except:
                    return "⏳ Aguardando"
            
            def obter_boleto(convenio_nome):
                if df_situacoes.empty or convenio_nome not in df_situacoes.index:
                    return 0
                return int(df_situacoes.loc[convenio_nome].get("boleto_pago") or 0)

            df_grouped["Boleto Pago"] = df_grouped["Convenio"].apply(obter_boleto)

            df_grouped["Status NF"] = df_grouped["Convenio"].apply(self._obter_status_nf_convenio)
            self.df_completo = df_grouped
            conn.close()            


            df_grouped["Número NF"] = df_grouped["Convenio"].apply(self._obter_numero_nf_convenio)


            self.atualizar_tabela(self.df_completo)

            self.df_visualizada = self.df_completo.copy()  # 🔄 Garante que a visualização reflita os dados novos


            


        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao carregar convênios: {e}")






    def aplicar_filtro_busca(self):
        texto = remover_acentos(self.input_busca.text().strip().lower())

        df_filtrado = self.df_completo.copy()

        if texto:
            df_filtrado["Convenio_normalizado"] = df_filtrado["Convenio"].astype(str).apply(lambda x: remover_acentos(x).strip().lower())
            df_filtrado = df_filtrado[df_filtrado["Convenio_normalizado"].str.contains(texto, na=False)]
            df_filtrado.drop(columns=["Convenio_normalizado"], inplace=True)

        self.atualizar_tabela(df_filtrado)

    def atualizar_tabela(self, df):
        self.tabela.setRowCount(0)
        self.tabela.setColumnCount(0)
        self.tabela.clearContents()

        self.df_visualizada = df

        if df.empty:
            return

        colunas = ["Convenio", "Quantidade", "Total", "Status NF", "Situação Lote", "Protocolo", "Número NF", "Boleto Pago"]
        self.tabela.setColumnCount(len(colunas))
        self.tabela.setRowCount(len(df))
        self.tabela.setHorizontalHeaderLabels([
            "Convênio", "Qtd Exames", "Valor a Receber", "Status NF",
            "Situação Lote (PBH)", "Protocolo", "Número NF", "Boleto"
        ])


        for i in range(len(df)):
            for j, coluna in enumerate(colunas):
                valor = df.iloc[i][coluna] if coluna in df.columns else "⏳ Pendente"
                
                if coluna == "Total":
                    try:
                        valor_float = float(valor)
                        texto = f"R$ {valor_float:,.2f}".replace(",", "X").replace(".", ",").replace("X", ".")
                    except:
                        texto = str(valor)
                elif coluna == "Status NF":
                    texto = str(valor)
                elif coluna == "Boleto Pago":
                    texto = "✅ Pago" if valor == 1 else "❌ Em aberto"

                else:
                    texto = str(valor)

                item = QTableWidgetItem(texto)
                item.setTextAlignment(Qt.AlignCenter)

                # Apenas no campo Status NF, adiciona ícone
                if coluna == "Status NF":
                    t = texto.lower()
                    if "cancelada" in t:
                        item.setIcon(QIcon("icones/icone_sucesso.png"))
                    elif "cancelamento" in t:  # "Cancelamento solicitado" ou "Cancelamento em análise"
                        item.setIcon(QIcon("icones/icone_aguarde.png"))
                    elif "emitida" in t or "processado (ok)" in t:
                        item.setIcon(QIcon("icones/icone_sucesso.png"))
                    elif "erro" in t:
                        item.setIcon(QIcon("icones/icone_erro.png"))
                    elif "pendente" in t:
                        item.setIcon(QIcon("icones/icone_aguarde.png"))


                self.tabela.setItem(i, j, item)

        self.tabela.resizeColumnsToContents()
        self.tabela.resizeRowsToContents()
        self.tabela.horizontalHeader().setStretchLastSection(True)

        # Atualizar rodapé com totais
        total_convenios = len(df)
        total_exames = df["Quantidade"].sum() if "Quantidade" in df.columns else 0
        total_valor = df["Total"].sum() if "Total" in df.columns else 0.0

        # 🟢 Somar Total Pago e Total Em Aberto
        df["Total"] = pd.to_numeric(df["Total"], errors="coerce").fillna(0)
        valor_pago = df[df["Boleto Pago"] == 1]["Total"].sum()
        valor_aberto = df[df["Boleto Pago"] == 0]["Total"].sum()

        total_valor_str = f"R$ {total_valor:,.2f}".replace(",", "X").replace(".", ",").replace("X", ".")
        valor_pago_str = f"R$ {valor_pago:,.2f}".replace(",", "X").replace(".", ",").replace("X", ".")
        valor_aberto_str = f"R$ {valor_aberto:,.2f}".replace(",", "X").replace(".", ",").replace("X", ".")

        self.label_totais.setText(
            f"Convênios: {total_convenios} | Total de exames: {total_exames} | "
            f"Total a receber: {total_valor_str} | ✅ Total Pago: {valor_pago_str} | ❌ Total em aberto: {valor_aberto_str}"
        )

    def habilitar_botao(self):
        self.botao_exibir.setEnabled(True)

    def exibir_modal_faturamento(self):
        selected = self.tabela.currentRow()
        item = self.tabela.item(selected, 0)
        if selected < 0 or item is None:
            QMessageBox.warning(self, "Aviso", "Nenhum convênio selecionado.")
            return
        convenio = item.text()
        dialog = DialogFaturamentoConvenio(convenio, self.competencia, self)
        dialog.atualizar_tabela_signal.connect(self.carregar_convenios_por_competencia)  # ✅ Conecta o sinal
        dialog.exec_()


