from PyQt5.QtWidgets import (
    QWidget, QVBoxLayout, QHBoxLayout, QPushButton, QTableWidget,
    QTableWidgetItem, QDialog, QFormLayout, QLineEdit, QComboBox,
    QMessageBox, QLabel, QDateEdit, QTabWidget, QHeaderView
)
from PyQt5.QtCore import Qt, QDate
import sqlite3
import os
from datetime import datetime
from PyQt5.QtGui import QDoubleValidator

CAMINHO_BANCO = os.path.join("db", "sistema_financeiro.db")

# ----------------------------
# Aba 1 - Cadastro de Funcionários
# ----------------------------

class DialogFuncionario(QDialog):
    def __init__(self, funcionario=None, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Funcionário")
        self.setFixedSize(400, 350)
        layout = QFormLayout()

        self.input_nome = QLineEdit()
        self.input_idade = QLineEdit()
        self.input_cpf = QLineEdit()
        self.input_pis = QLineEdit()
        self.input_funcao = QLineEdit()
        self.input_valor = QLineEdit()
        

        validator = QDoubleValidator(0.00, 99999.99, 2)
        validator.setNotation(QDoubleValidator.StandardNotation)
        self.input_valor.setValidator(validator)
        self.data_admissao = QDateEdit(calendarPopup=True)
        self.data_admissao.setDate(QDate.currentDate())
        self.data_demissao = QDateEdit(calendarPopup=True)
        self.data_demissao.setDate(QDate.currentDate())

        self.data_demissao.setSpecialValueText("Não informada")
        self.data_demissao.setDate(QDate(2000, 1, 1))  # Valor mínimo especial
        self.data_demissao.setMinimumDate(QDate(2000, 1, 1))


        self.combo_status = QComboBox()
        self.combo_status.addItems(["Ativo", "Inativo"])
        


        layout.addRow("Nome:", self.input_nome)
        layout.addRow("Idade:", self.input_idade)
        layout.addRow("CPF:", self.input_cpf)
        layout.addRow("PIS:", self.input_pis)
        layout.addRow("Função:", self.input_funcao)
        layout.addRow("Valor Mensal (R$):", self.input_valor)
        layout.addRow("Admissão:", self.data_admissao)
        layout.addRow("Demissão:", self.data_demissao)

        btn_salvar = QPushButton("Salvar")
        btn_salvar.clicked.connect(self.accept)
        layout.addRow("Status:", self.combo_status)
        layout.addRow(btn_salvar)
        self.setLayout(layout)

        if funcionario:
            self.input_nome.setText(funcionario[1])
            self.input_idade.setText(str(funcionario[2]))
            self.input_cpf.setText(funcionario[3])
            self.input_pis.setText(funcionario[4])
            self.input_funcao.setText(funcionario[5])
            self.input_valor.setText(str(funcionario[6]))
            self.data_admissao.setDate(QDate.fromString(funcionario[7], "yyyy-MM-dd"))
            if funcionario[8]:
                self.data_demissao.setDate(QDate.fromString(funcionario[8], "yyyy-MM-dd"))
            else:
                self.data_demissao.setDate(QDate(2000, 1, 1))

            if len(funcionario) > 9:
                self.combo_status.setCurrentText(funcionario[9] or "Ativo")


    def get_data(self):
        try:
            idade = int(self.input_idade.text())
        except ValueError:
            QMessageBox.warning(self, "Erro", "Idade inválida. Digite apenas números inteiros.")
            return None

        try:
            valor = float(self.input_valor.text().replace(",", ".").strip() or 0)
        except ValueError:
            QMessageBox.warning(self, "Erro", "Valor mensal inválido. Digite apenas números, com vírgula ou ponto.")
            return None

        return (
            self.input_nome.text().strip(),
            idade,
            self.input_cpf.text().strip(),
            self.input_pis.text().strip(),
            self.input_funcao.text().strip(),
            valor,
            self.data_admissao.date().toString("yyyy-MM-dd"),
            None if self.data_demissao.date() == QDate(2000, 1, 1) else self.data_demissao.date().toString("yyyy-MM-dd"),

            self.combo_status.currentText()
        )



class AbaCadastroFuncionarios(QWidget):
    def __init__(self):
        super().__init__()
        self.setWindowTitle("Cadastro de Funcionários")
        self.layout = QVBoxLayout(self)

        botoes = QHBoxLayout()
        self.btn_novo = QPushButton("+ Criar Funcionário")
        self.btn_editar = QPushButton("Editar")
        self.btn_excluir = QPushButton("Excluir")
        self.btn_editar.setEnabled(False)
        self.btn_excluir.setEnabled(False)
        botoes.addWidget(self.btn_novo)
        botoes.addWidget(self.btn_editar)
        botoes.addWidget(self.btn_excluir)

        self.layout.addLayout(botoes)

        # Campo de filtro por nome
        filtro_layout = QHBoxLayout()
        filtro_layout.addWidget(QLabel("Filtrar por nome:"))
        self.input_filtro = QLineEdit()
        self.input_filtro.setPlaceholderText("🔍 Digite o nome do funcionário...")
        self.input_filtro.textChanged.connect(self.carregar_dados)
        filtro_layout.addWidget(self.input_filtro)
        self.layout.addLayout(filtro_layout)


        self.tabela = QTableWidget()
        self.tabela.setColumnCount(9)
        self.tabela.setHorizontalHeaderLabels(["Nome", "Idade", "CPF", "PIS", "Função", "Valor", "Admissão", "Demissão", "Status"])
        self.tabela.setSelectionBehavior(QTableWidget.SelectRows)
        self.tabela.horizontalHeader().setStretchLastSection(True)
        self.tabela.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)

        self.layout.addWidget(self.tabela)

        self.label_rodape = QLabel("Total: 0 funcionários | Valor total: R$ 0,00")
        self.layout.addWidget(self.label_rodape)

        self.tabela.itemSelectionChanged.connect(self.atualizar_botoes)
        self.btn_novo.clicked.connect(self.adicionar_funcionario)
        self.btn_editar.clicked.connect(self.editar_funcionario)
        self.btn_excluir.clicked.connect(self.excluir_funcionario)

        self.carregar_dados()

    def conexao(self):
        return sqlite3.connect(CAMINHO_BANCO)

    def carregar_dados(self):
        self.tabela.setRowCount(0)
        conn = self.conexao()
        c = conn.cursor()

        # Cria tabela se ainda não existir
        c.execute("""
            CREATE TABLE IF NOT EXISTS funcionarios (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                nome TEXT,
                idade INTEGER,
                cpf TEXT,
                pis TEXT,
                funcao TEXT,
                valor_mensal REAL,
                admissao TEXT,
                demissao TEXT
            )
        """)

        # 🛠️ Adiciona a coluna 'status' se ainda não existir
        try:
            c.execute("ALTER TABLE funcionarios ADD COLUMN status TEXT DEFAULT 'Ativo'")
            conn.commit()
        except sqlite3.OperationalError as e:
            if "duplicate column" not in str(e):
                raise  # Só ignora erro se for 'coluna já existe'

        filtro_nome = self.input_filtro.text().strip().lower()

        if filtro_nome:
            c.execute("""
                SELECT * FROM funcionarios
                WHERE lower(nome) LIKE ?
                ORDER BY nome
            """, (f"%{filtro_nome}%",))
        else:
            c.execute("SELECT * FROM funcionarios ORDER BY nome")

        funcionarios = c.fetchall()

        conn.close()

        total = 0
        soma = 0
        for row_idx, row in enumerate(funcionarios):
            self.tabela.insertRow(row_idx)
            for col_idx, val in enumerate(row[1:]):
                item = QTableWidgetItem(str(val))
                item.setTextAlignment(Qt.AlignCenter)
                item.setFlags(Qt.ItemIsSelectable | Qt.ItemIsEnabled)
                self.tabela.setItem(row_idx, col_idx, item)

            soma += row[6]
            total += 1

        valor_formatado = f"R$ {soma:,.2f}".replace(",", "X").replace(".", ",").replace("X", ".")
        self.label_rodape.setText(f"<b>Total:</b> {total} &nbsp;&nbsp;|&nbsp;&nbsp; <b>Valor total:</b> {valor_formatado}")


    def atualizar_botoes(self):
        selecionado = self.tabela.currentRow() >= 0
        self.btn_editar.setEnabled(selecionado)
        self.btn_excluir.setEnabled(selecionado)

    def adicionar_funcionario(self):
        dialog = DialogFuncionario()
        if dialog.exec_():
            dados = dialog.get_data()
            conn = self.conexao()
            conn.execute("""
                INSERT INTO funcionarios (nome, idade, cpf, pis, funcao, valor_mensal, admissao, demissao, status)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
            """, dados)
            conn.commit()
            conn.close()
            self.carregar_dados()

    def editar_funcionario(self):
        row = self.tabela.currentRow()
        if row < 0:
            return
        nome = self.tabela.item(row, 0).text()
        conn = self.conexao()
        c = conn.cursor()
        c.execute("SELECT * FROM funcionarios WHERE nome = ?", (nome,))
        funcionario = c.fetchone()
        conn.close()

        dialog = DialogFuncionario(funcionario)
        if dialog.exec_():
            novos_dados = dialog.get_data()
            conn = self.conexao()
            conn.execute("""
                UPDATE funcionarios SET
                    nome=?, idade=?, cpf=?, pis=?, funcao=?, valor_mensal=?,
                    admissao=?, demissao=?, status=?
                WHERE id=?
            """, (*novos_dados, funcionario[0]))


            conn.commit()
            conn.close()
            self.carregar_dados()

    def excluir_funcionario(self):
        row = self.tabela.currentRow()
        if row < 0:
            return
        nome = self.tabela.item(row, 0).text()
        if QMessageBox.question(self, "Excluir", f"Excluir funcionário '{nome}'?", QMessageBox.Yes | QMessageBox.No) == QMessageBox.Yes:
            conn = self.conexao()
            conn.execute("DELETE FROM funcionarios WHERE nome = ?", (nome,))
            conn.commit()
            conn.close()
            self.carregar_dados()

# ----------------------------
# Aba 2 - Pagamentos de Funcionários
# ----------------------------


class DialogPagamentoFuncionario(QDialog):
    def __init__(self, funcionarios_ativos, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Lançar Pagamento")
        self.setMinimumWidth(500)

        layout_principal = QVBoxLayout()
        form_layout = QFormLayout()

        self.combo_funcionario = QComboBox()
        for f in funcionarios_ativos:
            self.combo_funcionario.addItem(f[1], f[0])  # nome, id

        # Competência: mês e ano
        meses = [f"{i:02}" for i in range(1, 13)]
        anos = [str(ano) for ano in range(2024, QDate.currentDate().year() + 2)]

        self.combo_mes = QComboBox()
        self.combo_mes.addItems(meses)
        self.combo_ano = QComboBox()
        self.combo_ano.addItems(anos)

        competencia_layout = QHBoxLayout()
        competencia_layout.addWidget(self.combo_mes)
        competencia_layout.addWidget(QLabel("/"))
        competencia_layout.addWidget(self.combo_ano)

        self.input_valor_pago = QLineEdit()
        self.data_pagamento = QDateEdit(calendarPopup=True)
        self.data_pagamento.setDate(QDate.currentDate())
        self.input_observacao = QLineEdit()

        form_layout.addRow("Funcionário:", self.combo_funcionario)
        form_layout.addRow("Competência:", competencia_layout)
        form_layout.addRow("Valor Pago (R$):", self.input_valor_pago)
        form_layout.addRow("Data Pagamento:", self.data_pagamento)
        form_layout.addRow("Observação:", self.input_observacao)

        layout_principal.addLayout(form_layout)

        btn_salvar = QPushButton("Salvar")
        btn_salvar.setFixedWidth(120)
        btn_salvar.clicked.connect(self.accept)
        btn_layout = QHBoxLayout()
        btn_layout.addStretch()
        btn_layout.addWidget(btn_salvar)
        btn_layout.addStretch()
        layout_principal.addLayout(btn_layout)

        self.setLayout(layout_principal)

    def get_data(self):
        valor_texto = self.input_valor_pago.text().replace(",", ".").strip()

        if not valor_texto:
            QMessageBox.warning(self, "Campo obrigatório", "Informe o valor pago.")
            return None

        try:
            valor_digitado = float(valor_texto)
        except ValueError:
            QMessageBox.warning(self, "Erro", "Valor inválido.")
            return None

        if valor_digitado < 0:
            QMessageBox.warning(self, "Erro", "O valor pago não pode ser negativo.")
            return None

        funcionario_id = self.combo_funcionario.currentData()
        conn = sqlite3.connect(CAMINHO_BANCO)
        cursor = conn.cursor()
        cursor.execute("SELECT valor_mensal FROM funcionarios WHERE id = ?", (funcionario_id,))
        resultado = cursor.fetchone()
        conn.close()

        if resultado:
            valor_mensal = resultado[0]
            if valor_digitado > valor_mensal:
                QMessageBox.warning(
                    self, "Valor excedente",
                    f"O valor pago (R$ {valor_digitado:.2f}) excede o valor mensal do funcionário (R$ {valor_mensal:.2f})."
                )
                return None
        else:
            QMessageBox.warning(self, "Erro", "Funcionário não encontrado.")
            return None

        # 🧠 Determinar o status
        if valor_digitado == 0:
            status = "Não pago"
        elif valor_digitado < valor_mensal:
            status = "Parcial"
        else:
            status = "Pago"

        return {
            "id_funcionario": funcionario_id,
            "competencia": f"{self.combo_mes.currentText()}/{self.combo_ano.currentText()}",
            "valor_pago": valor_digitado,
            "data_pagamento": self.data_pagamento.date().toString("yyyy-MM-dd"),
            "observacao": self.input_observacao.text().strip(),
            "status_pagamento": status
        }


class AbaPagamentosFuncionarios(QWidget):
    def __init__(self):
        super().__init__()
        self.setWindowTitle("Pagamentos de Funcionários")
        self.setLayout(QVBoxLayout())

        filtro_layout = QHBoxLayout()
        filtro_layout.addWidget(QLabel("Competência:"))
        self.combo_competencia = QComboBox()
        filtro_layout.addWidget(QLabel(" | Filtrar por nome:"))
        self.input_filtro_nome = QLineEdit()
        self.input_filtro_nome.setPlaceholderText("🔍 Digite o nome...")
        self.input_filtro_nome.textChanged.connect(self.carregar_pagamentos)
        filtro_layout.addWidget(self.input_filtro_nome)

        self.combo_competencia.currentIndexChanged.connect(self.carregar_pagamentos)
        filtro_layout.addWidget(self.combo_competencia)

        self.btn_lancar = QPushButton("Lançar Pagamento")
        self.btn_editar = QPushButton("Editar Pagamento")
        self.btn_editar.setEnabled(False)
        self.btn_excluir = QPushButton("Excluir Pagamento")
        self.btn_excluir.setEnabled(False)

        filtro_layout.addWidget(self.btn_lancar)
        filtro_layout.addWidget(self.btn_editar)
        filtro_layout.addWidget(self.btn_excluir)

        self.layout().addLayout(filtro_layout)

        self.tabela = QTableWidget()
        self.tabela.setColumnCount(7)
        self.tabela.setHorizontalHeaderLabels([
            "Funcionário", "Valor Pago", "Valor em Aberto", "Status",
            "Data Pagamento", "Observação", "Competência"
        ])
        self.tabela.setSelectionBehavior(QTableWidget.SelectRows)
        self.tabela.itemSelectionChanged.connect(self.habilitar_editar)
        self.tabela.horizontalHeader().setStretchLastSection(True)
        self.tabela.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)

        self.layout().addWidget(self.tabela)

        self.label_rodape = QLabel("Valor Pago: R$ 0,00 | Valor em Aberto: R$ 0,00")
        self.layout().addWidget(self.label_rodape)

        self.btn_lancar.clicked.connect(self.lancar_pagamento)
        self.btn_editar.clicked.connect(self.editar_pagamento)
        self.btn_excluir.clicked.connect(self.excluir_pagamento)


        self.inicializar_tabela()
        self.carregar_competencias()

    def conexao(self):
        return sqlite3.connect(CAMINHO_BANCO)

    def inicializar_tabela(self):
        conn = self.conexao()
        c = conn.cursor()
        c.execute("""
            CREATE TABLE IF NOT EXISTS pagamentos_funcionarios (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                id_funcionario INTEGER,
                competencia TEXT,
                valor_pago REAL,
                data_pagamento TEXT,
                observacao TEXT,
                status_pagamento TEXT
            )
        """)
        conn.commit()
        conn.close()

    def carregar_competencias(self):
        conn = self.conexao()
        c = conn.cursor()
        c.execute("SELECT DISTINCT competencia FROM pagamentos_funcionarios ORDER BY competencia DESC")
        competencias = [row[0] for row in c.fetchall()]
        conn.close()
        self.combo_competencia.clear()
        self.combo_competencia.addItems(competencias if competencias else ["07/2025"])

    def carregar_pagamentos(self):
        competencia = self.combo_competencia.currentText()
        conn = self.conexao()
        c = conn.cursor()
        filtro_nome = self.input_filtro_nome.text().strip().lower()

        query = """
            SELECT f.nome, pg.valor_pago, pg.status_pagamento, pg.data_pagamento, pg.observacao, pg.competencia
            FROM pagamentos_funcionarios pg
            JOIN funcionarios f ON pg.id_funcionario = f.id
            WHERE pg.competencia = ?
        """
        params = [competencia]

        if filtro_nome:
            query += " AND lower(f.nome) LIKE ?"
            params.append(f"%{filtro_nome}%")

        query += " ORDER BY f.nome"
        c.execute(query, params)

        dados = c.fetchall()

        self.tabela.setRowCount(0)
        total_pago = 0.0
        total_em_aberto = 0.0

        for i, row in enumerate(dados):
            nome, valor_pago, status, data, obs, comp = row

            c.execute("SELECT valor_mensal FROM funcionarios WHERE nome = ?", (nome,))
            result = c.fetchone()
            valor_mensal = result[0] if result else 0.0
            valor_em_aberto = max(valor_mensal - valor_pago, 0)

            total_pago += valor_pago
            total_em_aberto += valor_em_aberto

            status_emoji = "✅ Pago" if status == "Pago" else "🟡 Parcial" if status == "Parcial" else "❌ Não pago"

            colunas = [
                nome,
                f"{valor_pago:.2f}",
                f"{valor_em_aberto:.2f}",
                status_emoji,
                data,
                obs,
                comp
            ]

            self.tabela.insertRow(i)
            for j, val in enumerate(colunas):
                item = QTableWidgetItem(val)
                item.setTextAlignment(Qt.AlignCenter)
                self.tabela.setItem(i, j, item)

        conn.close()

        valor_pago_str = f"R$ {total_pago:,.2f}".replace(",", "X").replace(".", ",").replace("X", ".")
        valor_aberto_str = f"R$ {total_em_aberto:,.2f}".replace(",", "X").replace(".", ",").replace("X", ".")
        self.label_rodape.setText(
            f"<b>Valor Pago:</b> {valor_pago_str} &nbsp;&nbsp;|&nbsp;&nbsp; <b>Valor em Aberto:</b> {valor_aberto_str}"
        )

    def habilitar_editar(self):
        ativo = self.tabela.currentRow() >= 0
        self.btn_editar.setEnabled(ativo)
        self.btn_excluir.setEnabled(ativo)


    def obter_funcionarios_ativos(self):
        conn = self.conexao()
        c = conn.cursor()
        c.execute("SELECT id, nome FROM funcionarios WHERE status = 'Ativo'")
        return c.fetchall()

    def lancar_pagamento(self):
        funcionarios = self.obter_funcionarios_ativos()
        if not funcionarios:
            QMessageBox.warning(self, "Aviso", "Nenhum funcionário ativo para lançar pagamento.")
            return

        dialog = DialogPagamentoFuncionario(funcionarios)
        if dialog.exec_():
            dados = dialog.get_data()
            if not dados:
                return

            # 🔒 Verifica duplicidade
            conn = self.conexao()
            c = conn.cursor()
            c.execute("""
                SELECT COUNT(*) FROM pagamentos_funcionarios
                WHERE id_funcionario = ? AND competencia = ?
            """, (dados["id_funcionario"], dados["competencia"]))
            existe = c.fetchone()[0]
            if existe:
                QMessageBox.warning(
                    self, "Duplicado",
                    "Já existe um pagamento registrado para este funcionário nesta competência."
                )
                conn.close()
                return

            # ✅ Insere pagamento
            c.execute("""
                INSERT INTO pagamentos_funcionarios (
                    id_funcionario, competencia, valor_pago, data_pagamento, observacao, status_pagamento
                ) VALUES (?, ?, ?, ?, ?, ?)
            """, (
                dados["id_funcionario"], dados["competencia"], dados["valor_pago"],
                dados["data_pagamento"], dados["observacao"], dados["status_pagamento"]
            ))
            conn.commit()
            conn.close()
            self.carregar_competencias()
            self.carregar_pagamentos()


    def editar_pagamento(self):
        row = self.tabela.currentRow()
        if row < 0:
            return

        nome = self.tabela.item(row, 0).text()
        competencia = self.tabela.item(row, 6).text()  # coluna competência

        conn = self.conexao()
        c = conn.cursor()
        c.execute("""
            SELECT pg.id, pg.id_funcionario, pg.competencia, pg.valor_pago,
                pg.data_pagamento, pg.observacao, pg.status_pagamento
            FROM pagamentos_funcionarios pg
            JOIN funcionarios f ON pg.id_funcionario = f.id
            WHERE pg.competencia = ? AND f.nome = ?
        """, (competencia, nome))
        row_db = c.fetchone()
        conn.close()

        if not row_db:
            QMessageBox.warning(self, "Erro", "Não foi possível localizar o registro para edição.")
            return

        dados = {
            "id": row_db[0],
            "id_funcionario": row_db[1],
            "competencia": row_db[2],
            "valor_pago": row_db[3],
            "data_pagamento": row_db[4],
            "observacao": row_db[5],
            "status_pagamento": row_db[6]
        }

        dialog = DialogPagamentoFuncionario(self.obter_funcionarios_ativos())
        # preencher dados
        dialog.combo_funcionario.setCurrentIndex(
            dialog.combo_funcionario.findData(dados["id_funcionario"]))
        mes, ano = dados["competencia"].split("/")
        dialog.combo_mes.setCurrentText(mes)
        dialog.combo_ano.setCurrentText(ano)
        dialog.input_valor_pago.setText(str(dados["valor_pago"]))
        dialog.data_pagamento.setDate(QDate.fromString(dados["data_pagamento"], "yyyy-MM-dd"))
        dialog.input_observacao.setText(dados["observacao"])

        if dialog.exec_():
            novos_dados = dialog.get_data()
            if not novos_dados:
                return

            conn = self.conexao()
            c = conn.cursor()
            c.execute("""
                UPDATE pagamentos_funcionarios
                SET id_funcionario = ?, competencia = ?, valor_pago = ?, data_pagamento = ?, 
                    observacao = ?, status_pagamento = ?
                WHERE id = ?
            """, (
                novos_dados["id_funcionario"], novos_dados["competencia"], novos_dados["valor_pago"],
                novos_dados["data_pagamento"], novos_dados["observacao"], novos_dados["status_pagamento"],
                dados["id"]
            ))
            conn.commit()
            conn.close()
            self.carregar_pagamentos()

    def excluir_pagamento(self):
        row = self.tabela.currentRow()
        if row < 0:
            return

        nome = self.tabela.item(row, 0).text()
        competencia = self.tabela.item(row, 6).text()

        confirm = QMessageBox.question(
            self,
            "Confirmar Exclusão",
            f"Tem certeza que deseja excluir o pagamento de '{nome}' na competência {competencia}?",
            QMessageBox.Yes | QMessageBox.No
        )

        if confirm == QMessageBox.Yes:
            conn = self.conexao()
            c = conn.cursor()
            c.execute("""
                DELETE FROM pagamentos_funcionarios
                WHERE id_funcionario = (
                    SELECT id FROM funcionarios WHERE nome = ?
                ) AND competencia = ?
            """, (nome, competencia))
            conn.commit()
            conn.close()
            self.carregar_pagamentos()




# ----------------------------
# Módulo Final com Abas
# ----------------------------

class ModuloFuncionarios(QWidget):
    def __init__(self):
        super().__init__()
        layout = QVBoxLayout(self)
        abas = QTabWidget()
        abas.addTab(AbaCadastroFuncionarios(), "Cadastro de Funcionários")
        abas.addTab(AbaPagamentosFuncionarios(), "Pagamentos Mensais")
        layout.addWidget(abas)
        self.setLayout(layout)
