from PyQt5.QtWidgets import (
    QWidget, QVBoxLayout, QHBoxLayout, QPushButton, QComboBox,
    QTableWidget, QTableWidgetItem, QMessageBox, QLabel, QHeaderView, QFrame,
    QAbstractItemView, QLineEdit
)
from PyQt5.QtCore import Qt
from datetime import datetime, date
import sqlite3, os
from PyQt5.QtWidgets import QInputDialog
import calendar

CAMINHO_BANCO = os.path.join("db", "sistema_financeiro.db")


def iso_to_br(s: str) -> str:
    if not s:
        return ""
    try:
        dt = datetime.strptime(s, "%Y-%m-%d")
        return f"{dt.day:02d}/{dt.month:02d}/{dt.year:04d}"
    except Exception:
        return s  # se já vier em DD/MM/AAAA, mantém

def parse_money(txt: str) -> float:
    """Converte 'R$ 1.234,56' ou '1234,56' para float 1234.56."""
    if not txt:
        raise ValueError("vazio")
    txt = txt.replace("R$", "").strip()
    txt = txt.replace(".", "").replace(",", ".")
    return float(txt)


def fmt_money(v):
    s = f"{float(v):,.2f}"
    s = s.replace(",", "X").replace(".", ",").replace("X", ".")
    return f"R$ {s}"


def _ultimo_dia_mes(ano: int, mes: int) -> int:
    return calendar.monthrange(ano, mes)[1]

def _datas_semanais_do_mes(ano: int, mes: int, weekday: int):
    # weekday: 0=Seg .. 6=Dom (compatível com datetime.weekday)
    from datetime import date, timedelta
    d = date(ano, mes, 1)
    # avança até o primeiro 'weekday'
    while d.weekday() != weekday:
        d += timedelta(days=1)
    while d.month == mes:
        yield d
        d += timedelta(days=7)



class ModuloInfoImpostos(QWidget):
    
    # -------- Helpers --------
    def competencia_atual(self) -> str:
        return f"{self.combo_mes.currentText()}/{self.combo_ano.currentText()}"

    def _on_competencia_changed(self, *_):
        self.carregar_dados()

    
    def __init__(self):
        super().__init__()

        root = QVBoxLayout(self)
        root.setContentsMargins(20, 20, 20, 20)
        root.setSpacing(20)

        estilo_botao = """
            QPushButton {
                background-color: #3498db;
                color: white;
                padding: 6px 12px;
                font-weight: bold;
                border-radius: 4px;
            }
            QPushButton:hover { background-color: #2980b9; }
        """

        # =========================
        # Filtro de competência (Mês/Ano) + ações
        # =========================
        topbar = QHBoxLayout()
        topbar.setSpacing(8)

        # --- opções de competência ---
        MESES = ["01","02","03","04","05","06","07","08","09","10","11","12"]
        ANOS  = ["2025","2026","2027"]

        topbar.addWidget(QLabel("Mês:"))
        self.combo_mes = QComboBox()
        self.combo_mes.addItems(MESES)
        self.combo_mes.setMinimumWidth(70)
        topbar.addWidget(self.combo_mes)

        topbar.addWidget(QLabel("Ano:"))
        self.combo_ano = QComboBox()
        self.combo_ano.addItems(ANOS)
        self.combo_ano.setMinimumWidth(90)
        topbar.addWidget(self.combo_ano)

        # Atualiza dados quando mês/ano mudarem
        self.combo_mes.currentIndexChanged.connect(self._on_competencia_changed)
        self.combo_ano.currentIndexChanged.connect(self._on_competencia_changed)

        topbar.addStretch(1)

        # Botões padronizados
        self.btn_editar_valor = QPushButton("✏️ Editar Valor")
        self.btn_pago         = QPushButton("💰 Marcar como Pago")
        self.btn_nao_pago     = QPushButton("⛔ Marcar como NÃO Pago")
        for b in (self.btn_editar_valor, self.btn_pago, self.btn_nao_pago):
            b.setStyleSheet(estilo_botao)

        topbar.addWidget(self.btn_editar_valor)
        topbar.addWidget(self.btn_pago)
        topbar.addWidget(self.btn_nao_pago)

        root.addLayout(topbar)

        # =========================
        # Campo de busca
        # =========================
        filtro_layout = QHBoxLayout()
        self.campo_busca = QLineEdit()
        self.campo_busca.setPlaceholderText("🔍 Buscar por nome / origem / situação...")
        self.campo_busca.textChanged.connect(self.filtrar_linhas)
        filtro_layout.addWidget(self.campo_busca)
        root.addLayout(filtro_layout)

        # =========================
        # Separador
        # =========================
        linha_div = QFrame()
        linha_div.setFrameShape(QFrame.HLine)
        linha_div.setFrameShadow(QFrame.Sunken)
        root.addWidget(linha_div)

        # =========================
        # Tabela
        # =========================
        self.tabela = QTableWidget()
        self.tabela.setColumnCount(7)
        self.tabela.setHorizontalHeaderLabels(
            ["Nome", "Origem", "Periodicidade", "Vencimento", "Valor", "Situação", "Data Pag."]
        )
        self.tabela.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.tabela.setSelectionMode(QAbstractItemView.SingleSelection)
        self.tabela.setAlternatingRowColors(True)
        self.tabela.verticalHeader().setVisible(False)
        self.tabela.setEditTriggers(QAbstractItemView.NoEditTriggers)
        self.tabela.setSortingEnabled(True)

        header = self.tabela.horizontalHeader()
        header.setSectionResizeMode(QHeaderView.Stretch)
        header.setDefaultAlignment(Qt.AlignCenter)

        root.addWidget(self.tabela, 1)

        # =========================
        # Rodapé com legendas BONITO (mantido)
        # =========================
        self.footer_bar = QFrame()
        self.footer_bar.setObjectName("footerBar")
        self.footer_bar.setStyleSheet("""
            QFrame#footerBar {
                background: #fff;
                border-top: 1px solid #e9ecef;
                border-radius: 8px;
            }
            QLabel#value { font-weight: 700; }
            QLabel#title { color: #59636e; font-weight: 600; }
            QLabel#count { color: #7f8c8d; font-weight: 500; }
        """)

        footer_layout = QHBoxLayout(self.footer_bar)
        footer_layout.setContentsMargins(14, 10, 14, 10)
        footer_layout.setSpacing(12)

        # grupos: Total (cinza), Abertos (laranja), Pago (verde)
        self.grp_total  = self._make_legend("#34495e", "Total")
        self.grp_aberto = self._make_legend("#f39c12", "Em aberto / Próx. / Atrasado")
        self.grp_pago   = self._make_legend("#27ae60", "Pago")

        footer_layout.addWidget(self.grp_total["widget"])
        footer_layout.addStretch(1)
        footer_layout.addWidget(self.grp_aberto["widget"])
        footer_layout.addStretch(1)
        footer_layout.addWidget(self.grp_pago["widget"])

        root.addWidget(self.footer_bar)

        # Conexões dos botões
        self.btn_pago.clicked.connect(self.marcar_pago)
        self.btn_nao_pago.clicked.connect(self.marcar_nao_pago)
        self.btn_editar_valor.clicked.connect(self.editar_valor)

        # Carrega dados iniciais
        self.carregar_dados()

        
    def filtrar_linhas(self):
        texto = (self.campo_busca.text() or "").strip().lower()
        termos = [t for t in texto.split() if t]
        for row in range(self.tabela.rowCount()):
            linha_txt = []
            for col in range(self.tabela.columnCount()):
                item = self.tabela.item(row, col)
                linha_txt.append(item.text().lower() if item else "")
            blob = " ".join(linha_txt)
            visivel = all(t in blob for t in termos)
            self.tabela.setRowHidden(row, not visivel)

        
    def editar_valor(self):
        iid = self._get_selected_registro_id()
        if iid is None:
            QMessageBox.warning(self, "Atenção", "Selecione um imposto para editar o valor.")
            return

        # Valor atual mostrado na coluna 4 (formatado)
        linha = self.tabela.currentRow()
        valor_atual_txt = self.tabela.item(linha, 4).text() if linha >= 0 else "R$ 0,00"


        # Pede novo valor (texto)
        novo_txt, ok = QInputDialog.getText(
            self, "Editar Valor",
            "Novo valor (ex.: 1.234,56):",
            text=valor_atual_txt
        )
        if not ok:
            return

        # Valida/converte
        try:
            novo_valor = parse_money(novo_txt)
            if novo_valor < 0:
                raise ValueError("negativo")
        except Exception:
            QMessageBox.warning(self, "Valor inválido", "Digite um valor no formato 1.234,56 (positivo).")
            return

        # Persiste no registro da competência (NÃO mexe no valor padrão do cadastro)
        con = sqlite3.connect(CAMINHO_BANCO)
        cur = con.cursor()
        cur.execute("UPDATE impostos_competencia SET valor=? WHERE id=?", (novo_valor, iid))
        con.commit()
        con.close()

        # Recarrega e mantém seleção na mesma linha (pelo id guardado no UserRole)
        self.carregar_dados()
        for r in range(self.tabela.rowCount()):
            it = self.tabela.item(r, 0)
            if it and it.data(Qt.UserRole) == iid:
                self.tabela.selectRow(r)
                break


    # ----- UI helpers -----
    def _dot(self, color: str, size: int = 10) -> QLabel:
        dot = QLabel()
        dot.setFixedSize(size, size)
        dot.setStyleSheet(f"background:{color}; border-radius:{size//2}px;")
        return dot

    def _make_legend(self, color: str, title: str):
        """
        Cria um "card" compacto: [●]  Título   R$ Valor   (qtd)
        Retorna dict com widget e referências a labels de valor/contagem.
        """
        w = QFrame()
        w.setStyleSheet("QFrame { background: transparent; }")
        lay = QHBoxLayout(w)
        lay.setContentsMargins(8, 4, 8, 4)
        lay.setSpacing(8)

        dot = self._dot(color, 10)
        lbl_title = QLabel(title)
        lbl_title.setObjectName("title")
        lbl_title.setProperty("class", "title")
        lbl_title.setStyleSheet("")  # usa CSS do frame

        lbl_value = QLabel("R$ 0,00")
        lbl_value.setObjectName("value")
        lbl_value.setProperty("class", "value")

        lbl_count = QLabel("(0)")
        lbl_count.setObjectName("count")
        lbl_count.setProperty("class", "count")

        lay.addWidget(dot, 0, Qt.AlignVCenter)
        lay.addWidget(lbl_title, 0, Qt.AlignVCenter)
        lay.addSpacing(6)
        lay.addWidget(lbl_value, 0, Qt.AlignVCenter)
        lay.addSpacing(4)
        lay.addWidget(lbl_count, 0, Qt.AlignVCenter)

        return {"widget": w, "value": lbl_value, "count": lbl_count}

    # ----- Status -----
    def calcular_status(self, pago, vencimento_iso):
        hoje = date.today()
        try:
            venc = datetime.strptime(vencimento_iso, "%Y-%m-%d").date()
        except Exception:
            return "⚪ Em aberto"

        if int(pago) == 1:   # ⬅️ aqui
            return "🟢 Pago"
        elif venc < hoje:
            return "🔴 Atrasado"
        elif (venc - hoje).days <= 7:
            return "🟡 Próximo do venc."
        else:
            return "⚪ Em aberto"


    # ----- Core -----
    def carregar_dados(self, competencia: str = None, *_):
        
        # --- Garante coluna data_pagamento ---
        with sqlite3.connect(CAMINHO_BANCO) as _con:
            _cur = _con.cursor()
            _cur.execute("PRAGMA table_info(impostos_competencia)")
            cols = [r[1] for r in _cur.fetchall()]
            if "data_pagamento" not in cols:
                _cur.execute("ALTER TABLE impostos_competencia ADD COLUMN data_pagamento TEXT")
                _con.commit()

        
        """
        Recarrega a tabela com base na competência (MM/AAAA).
        Se 'competencia' não for passada, usa os combos de Mês/Ano.
        """
        # --- Competência atual (MM/AAAA) ---
        
        if not competencia:
            competencia = self.competencia_atual()

        self.tabela.setRowCount(0)

        # --- Banco ---
        with sqlite3.connect(CAMINHO_BANCO) as con:
            cur = con.cursor()

            mes, ano = competencia.split("/")
            y, m = int(ano), int(mes)
            comp_ym = f"{ano}-{mes}"

            # 1) Busca impostos ativos e na vigência + seus parâmetros
            cur.execute("""
                SELECT id, nome, origem, valor,
                    COALESCE(dia_vencimento, 20) as dia_venc,
                    COALESCE(periodicidade, 'Mensal') as periodicidade,
                    dia_semana, mes_vencimento,
                    COALESCE(vigencia_inicio,''), COALESCE(vigencia_fim,'')
                FROM cadastro_impostos
                WHERE ativo = 1
                AND (vigencia_inicio IS NULL OR vigencia_inicio <= ?)
                AND (vigencia_fim    IS NULL OR vigencia_fim    >= ?)
            """, (comp_ym, comp_ym))
            impostos = cur.fetchall()

            # 2) Gera ocorrências para a competência
            for (iid, _nome, _origem, valor_padrao, dia_venc, per, _dia_sem, mes_anual, _vi, _vf) in impostos:
                venc_list = []

                if per == "Mensal":
                    dia = min(int(dia_venc or 20), _ultimo_dia_mes(y, m))
                    venc_list = [f"{y:04d}-{m:02d}-{dia:02d}"]

                elif per == "Anual":
                    if mes_anual and int(mes_anual) == m:
                        dia = min(int(dia_venc or 20), _ultimo_dia_mes(y, m))
                        venc_list = [f"{y:04d}-{m:02d}-{dia:02d}"]
                    else:
                        venc_list = []

                for venc in venc_list:
                    cur.execute("""
                        INSERT OR IGNORE INTO impostos_competencia (imposto_id, competencia, valor, vencimento, pago)
                        VALUES (?, ?, ?, ?, 0)
                    """, (iid, competencia, float(valor_padrao), venc))


            con.commit()

            # 4) Agora carregue para exibição normalmente
            cur.execute("""
                SELECT i.id,
                    c.nome,
                    c.origem,
                    COALESCE(c.periodicidade, 'Mensal') AS periodicidade,
                    i.valor,
                    i.vencimento,
                    i.pago,
                    COALESCE(i.data_pagamento, '') AS data_pagamento
                FROM impostos_competencia i
                JOIN cadastro_impostos c ON i.imposto_id = c.id
                WHERE i.competencia=?
                ORDER BY c.nome, i.vencimento
            """, (competencia,))
            rows = cur.fetchall()



        # --- Totais/contadores ---
        total_geral = 0.0
        total_pago = 0.0
        total_aberto = 0.0
        count_pago = 0
        count_aberto = 0

        # --- Preenche tabela ---
        for row_idx, (iid, nome, origem, periodicidade, valor, vencimento, pago, data_pagamento) in enumerate(rows):
            self.tabela.insertRow(row_idx)

            # Nome (guarda id no UserRole)
            it_nome = QTableWidgetItem(nome)
            it_nome.setData(Qt.UserRole, iid)
            it_nome.setTextAlignment(Qt.AlignCenter)
            self.tabela.setItem(row_idx, 0, it_nome)

            # Origem
            it_org = QTableWidgetItem(origem)
            it_org.setTextAlignment(Qt.AlignCenter)
            self.tabela.setItem(row_idx, 1, it_org)

            # Periodicidade
            it_per = QTableWidgetItem(periodicidade or "Mensal")
            it_per.setTextAlignment(Qt.AlignCenter)
            self.tabela.setItem(row_idx, 2, it_per)

            # Vencimento (exibe BR, mantém ISO como chave de ordenação)
            it_venc = QTableWidgetItem(iso_to_br(vencimento))
            it_venc.setData(Qt.UserRole, vencimento or "")  # sort pela data real
            it_venc.setTextAlignment(Qt.AlignCenter)
            self.tabela.setItem(row_idx, 3, it_venc)

            # Valor
            it_valor = QTableWidgetItem(fmt_money(valor))
            it_valor.setTextAlignment(Qt.AlignCenter)
            self.tabela.setItem(row_idx, 4, it_valor)

            # Situação (usa seu calcular_status)
            situacao = self.calcular_status(pago, vencimento)
            it_sit = QTableWidgetItem(situacao)
            it_sit.setTextAlignment(Qt.AlignCenter)
            self.tabela.setItem(row_idx, 5, it_sit)

            # Data Pag. (pode vir em ISO de registros antigos)
            it_dt = QTableWidgetItem(iso_to_br(data_pagamento or ""))
            it_dt.setData(Qt.UserRole, data_pagamento or "")
            it_dt.setTextAlignment(Qt.AlignCenter)
            self.tabela.setItem(row_idx, 6, it_dt)

            # Totais
            total_geral += float(valor)
            if int(pago) == 1:
                total_pago += float(valor); count_pago += 1
            else:
                total_aberto += float(valor); count_aberto += 1


        # --- Atualiza rodapé ---
        self.grp_total["value"].setText(fmt_money(total_geral))
        self.grp_total["count"].setText(f"({len(rows)})")

        self.grp_aberto["value"].setText(fmt_money(total_aberto))
        self.grp_aberto["count"].setText(f"({count_aberto})")

        self.grp_pago["value"].setText(fmt_money(total_pago))
        self.grp_pago["count"].setText(f"({count_pago})")

        # Dicas (tooltip)
        self.grp_total["value"].setToolTip("Soma de todos os impostos da competência selecionada.")
        self.grp_aberto["value"].setToolTip("Inclui: Em aberto, Próximo do vencimento e Atrasado.")
        self.grp_pago["value"].setToolTip("Somente registros marcados como Pago.")

    # ----- Ações -----
    def _get_selected_registro_id(self):
        linha = self.tabela.currentRow()
        if linha < 0:
            return None
        item = self.tabela.item(linha, 0)
        if not item:
            return None
        return item.data(Qt.UserRole)

    def marcar_pago(self):
        iid = self._get_selected_registro_id()
        if iid is None:
            QMessageBox.warning(self, "Atenção", "Selecione um imposto para marcar como pago.")
            return
        con = sqlite3.connect(CAMINHO_BANCO)
        cur = con.cursor()
        cur.execute(
            "UPDATE impostos_competencia "
            "SET pago=1, data_pagamento=strftime('%d/%m/%Y','now','localtime') "
            "WHERE id=?",
            (iid,)
        )
        con.commit()
        con.close()
        self.carregar_dados()

    def marcar_nao_pago(self):
        iid = self._get_selected_registro_id()
        if iid is None:
            QMessageBox.warning(self, "Atenção", "Selecione um imposto para marcar como NÃO pago.")
            return
        con = sqlite3.connect(CAMINHO_BANCO)
        cur = con.cursor()
        cur.execute("UPDATE impostos_competencia SET pago=0, data_pagamento=NULL WHERE id=?", (iid,))
        con.commit()
        con.close()
        self.carregar_dados()

