from PyQt5.QtWidgets import (
    QApplication, QMainWindow, QTabWidget, QWidget, QVBoxLayout, QLabel,
    QHBoxLayout, QToolButton, QMenu, QAction, QShortcut, QTabBar
)
from PyQt5.QtCore import Qt, QCoreApplication
from PyQt5.QtGui import QKeySequence
import sys

from PyQt5.QtGui import QIcon
import os

# ===== seus imports =====
from gui.modulo1_tabela_registros import ModuloTabelaRegistros
from gui.modulo2_cadastro_medicos import ModuloCadastroMedicos
from gui.modulo3_cadastro_convenio import ModuloCadastroConvenio
from gui.modulo4_vendedores import ModuloVendedores
from gui.modulo5_faturamento_convenios import ModuloFaturamentoConvenios
from gui.modulo6_faturamento_medicos import ModuloFaturamentoMedicos
from gui.modulo12_configuracoes_medical_laudos import ModuloConfiguracoesEmpresa
from gui.modulo7_prestadores import ModuloPrestadores
from gui.modulo_medicosemabas import ModuloMedicos
from gui.modulo_convenios_em_abas import ModuloConvenios
from gui.modulo8_funcionarios import ModuloFuncionarios
from gui.modulo9_impostos import ModuloImpostos
from gui.modulo10_outros_custos import ModuloOutrosCustos
from gui.modulo11_faturamento import ModuloFaturamento
from gui.modulo13_analises import ModuloAnalises
from gui.modulo14_vendas import ModuloVendas, ensure_schema_vendas
from db.inicializa_banco import inicializar_banco
from gui.modulo9_cadastro_impostos import ensure_schema_impostos

def resource_path(relative_path: str) -> str:
    base = getattr(sys, "_MEIPASS", os.path.abspath("."))
    return os.path.join(base, relative_path)

APP_ICON = None
for p in ("icones/medical_laudos.ico", "icones/medical_laudos.png"):
    full = resource_path(p)
    if os.path.exists(full):
        APP_ICON = full
        break


# HiDPI antes do QApplication
QCoreApplication.setAttribute(Qt.AA_EnableHighDpiScaling, True)
QCoreApplication.setAttribute(Qt.AA_UseHighDpiPixmaps, True)

# ====== CONFIG VISUAL ======
HAMBURGER_BP   = 1000   # hambúrguer aparece abaixo desse width
TOP_FONT_PT    = 7      # fonte topo (pequena)
SUB_FONT_PT    = 7      # fonte submenus
TOP_TAB_H      = 36     # altura da aba do topo (px)
SUB_TAB_THICK  = 32     # espessura dos submenus: altura (horizontal) ou largura (vertical)

# >>> CONTROLES DE LARGURA DAS ABAS DO TOPO <<<
TOP_TABS_FILL  = False  # True=preencher 100% (iguais); False=respeitar min/max
TOP_TAB_MIN_W  = 190
TOP_TAB_MAX_W  = 220
TOP_TAB_GAP    = 5

# >>> LARGURA DOS BOTÕES DOS SUBMENUS <<<
SUB_TABS_FILL  = False  # True=preencher barra; False=min/max
SUB_TAB_MIN_W  = 190
SUB_TAB_MAX_W  = 220
SUB_TAB_GAP    = 5

# Ícones (emojis) pro MENU HAMBÚRGUER
ICONES = {
    "Tabela de Registros": "📊",
    "Médicos": "🩺",
    "Convênios": "🤝",
    "Vendedores": "🧾",
    "Prestadores de Serviço": "🧑‍🔧",
    "Funcionários": "👥",
    "Impostos": "💸",
    "Outros Custos": "💼",
    "Faturamento Geral": "📈",
    "Configurações da Empresa": "⚙️",
    "Análises & Projeções": "📈",
    "Vendas": "🛒",
}

# ---------- QSS base ----------
PRIMARY_QSS = f"""
QTabWidget::pane {{
  border: 1px solid #c7d7e5; border-radius: 10px; padding: 0; background: #f4f7fb;
}}
QTabBar::tab {{
  background: #e2e8f0; color: #0f172a;
  border: 1px solid #d0d7e1; border-bottom: none;
  border-top-left-radius: 10px; border-top-right-radius: 10px;
  padding: 6px 12px; height: {TOP_TAB_H}px;
  font-weight: 600; font-size: {TOP_FONT_PT}pt;
  margin-right: {TOP_TAB_GAP}px;
}}
QTabBar::tab:selected {{
  background: #0ea5e9; color: white;
  border: 1px solid #0284c7; border-bottom: 2px solid #f4f7fb;
}}
QTabBar::tab:hover {{ background: #dbe5ef; }}
"""

# Submenus: sem height fixo (pra não matar verticais). Gap em ambas as direções.
SECONDARY_QSS_BASE = f"""
QTabWidget::pane {{
  border: 1px solid #dfe6ef; border-radius: 8px; padding: 0; background: #fafbfd;
}}
QTabBar::tab {{
  background: #eef2f7; color: #1f2937;
  border: 1px solid #e2e8f0; border-bottom: none;
  border-top-left-radius: 8px; border-top-right-radius: 8px;
  padding: 4px 10px;
  font-weight: 600; font-size: {SUB_FONT_PT}pt;
  margin-right: {SUB_TAB_GAP}px;
  margin-bottom: {SUB_TAB_GAP}px;
}}
QTabBar::tab:selected {{
  background: #22c55e; color: white;
  border: 1px solid #16a34a; border-bottom: 2px solid #fafbfd;
}}
QTabBar::tab:hover {{ background: #e8edf4; }}
"""

HAMBURGER_BTN_QSS = """
QToolButton#Hamburger {
  background: #0ea5e9; color: white; font-weight: 700;
  padding: 8px 14px; border-radius: 12px; border: 1px solid #0284c7;
}
QToolButton#Hamburger:hover { background: #38bdf8; }
QToolButton#Hamburger:pressed { background: #0284c7; }
QToolButton#Hamburger::menu-indicator { image: none; }
"""

HAMBURGER_MENU_QSS = """
QMenu { background: #0b1220; color: #e5e7eb; border: 1px solid #1f2a44; padding: 6px 0; }
QMenu::item { padding: 10px 18px; font-size: 10pt; font-weight: 600; }
QMenu::item:selected { background: #2563eb; color: white; }
"""

# ---------- TabBar configurável (apenas para o TOPO) ----------
class ConfigurableTabBar(QTabBar):
    def __init__(self, fill=True, min_w=110, max_w=220, gap=6, scroll=True, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._fill   = fill
        self._min_w  = min_w
        self._max_w  = max_w
        self._gap    = gap
        self._scroll = scroll
        self.setElideMode(Qt.ElideRight)
        self.setUsesScrollButtons(self._scroll)
        self.setMovable(False)

    def setParams(self, *, fill=None, min_w=None, max_w=None, gap=None, scroll=None):
        if fill   is not None: self._fill   = fill
        if min_w  is not None: self._min_w  = min_w
        if max_w  is not None: self._max_w  = max_w
        if gap    is not None: self._gap    = gap
        if scroll is not None:
            self._scroll = scroll
            self.setUsesScrollButtons(self._scroll)
        self.updateGeometry(); self.update()

    def tabSizeHint(self, index):
        s = super().tabSizeHint(index)
        n = max(1, self.count())
        bar_w = max(1, self.width())
        total_gap = self._gap * (n - 1)
        if self._fill:
            w = int((bar_w - total_gap) / n)
        else:
            per = int(bar_w / n)
            w = max(self._min_w, min(self._max_w, per))
        s.setWidth(max(10, w))
        return s

    def resizeEvent(self, e):
        super().resizeEvent(e)
        self.updateGeometry()

class FinanceiroMainWindow(QMainWindow):
    def __init__(self):
        super().__init__()
        if APP_ICON:
            self.setWindowIcon(QIcon(APP_ICON))
        self.setWindowTitle("Sistema Financeiro - Medical Laudos")
        self.resize(1200, 760)

        central = QWidget(self); self.setCentralWidget(central)
        root = QVBoxLayout(central); root.setContentsMargins(10,10,10,10); root.setSpacing(8)

        # ---------- Header (hambúrguer) ----------
        self.header = QWidget(self); hl = QHBoxLayout(self.header); hl.setContentsMargins(0,0,0,0)
        self.btn_menu = QToolButton(self.header); self.btn_menu.setObjectName("Hamburger")
        self.btn_menu.setText("☰  Menu"); self.btn_menu.setToolButtonStyle(Qt.ToolButtonTextOnly)
        self.btn_menu.setPopupMode(QToolButton.InstantPopup)
        self.btn_menu.setStyleSheet(HAMBURGER_BTN_QSS)
        self.menu_tabs = QMenu(self.btn_menu); self.menu_tabs.setStyleSheet(HAMBURGER_MENU_QSS)
        self.btn_menu.setMenu(self.menu_tabs)
        hl.addWidget(self.btn_menu); hl.addStretch(1)
        self.header.setVisible(False)
        root.addWidget(self.header)

        # ---------- Tabs topo ----------
        self.tabs = QTabWidget(self); self.tabs.setDocumentMode(True)
        root.addWidget(self.tabs, 1)

        self.top_bar = ConfigurableTabBar(
            fill=TOP_TABS_FILL,
            min_w=TOP_TAB_MIN_W,
            max_w=TOP_TAB_MAX_W,
            gap=TOP_TAB_GAP,
            scroll=not TOP_TABS_FILL
        )
        self.tabs.setTabBar(self.top_bar)
        self.tabs.setStyleSheet(PRIMARY_QSS)
        self.top_bar.setMinimumHeight(TOP_TAB_H)
        self.top_bar.setMaximumHeight(TOP_TAB_H)

        # ---------- Módulos ----------
        mods = {
            "Tabela de Registros": ModuloTabelaRegistros(),
            "Médicos": ModuloMedicos(self),
            "Convênios": ModuloConvenios(),
            "Vendedores": ModuloVendedores(self),
            "Prestadores de Serviço": ModuloPrestadores(),
            "Funcionários": ModuloFuncionarios(),
            "Impostos": ModuloImpostos(self),
            "Outros Custos": ModuloOutrosCustos(self),
            "Faturamento Geral": ModuloFaturamento(self),
            "Análises & Projeções": ModuloAnalises(self),  # ⬅️ NOVO
            "Vendas": ModuloVendas(self),
            "Configurações da Empresa": ModuloConfiguracoesEmpresa(self),
        }
        self._titles = []
        for nome, w in mods.items():
            if w is None:
                dummy = QWidget(); ly = QVBoxLayout(dummy); ly.addWidget(QLabel("Área de " + nome))
                self.tabs.addTab(dummy, nome)
            else:
                self.tabs.addTab(w, nome)
            self._titles.append(nome)
        for i, t in enumerate(self._titles):
            self.tabs.setTabToolTip(i, t)

        # menu hambúrguer (COM ÍCONES)
        self._rebuild_menu()
        self.tabs.currentChanged.connect(lambda i: self._sync_menu_checked(i))

        # atalho p/ forçar hambúrguer (teste)
        self._compact_forced = False
        QShortcut(QKeySequence("Ctrl+M"), self, activated=self._toggle_compact)

        # primeira aplicação
        self._apply_responsive()

    # ---------- Menu ----------
    def _rebuild_menu(self):
        self.menu_tabs.clear()
        for i, t in enumerate(self._titles):
            label = f"{ICONES.get(t,'•')}  {t}"
            act = QAction(label, self.menu_tabs); act.setCheckable(True)
            act.triggered.connect(lambda _, ix=i: self.tabs.setCurrentIndex(ix))
            self.menu_tabs.addAction(act)
        self._sync_menu_checked(self.tabs.currentIndex())

    def _sync_menu_checked(self, idx):
        for i, act in enumerate(self.menu_tabs.actions()):
            act.setChecked(i == idx)

    # ---------- Helpers de submenu ----------
    def _submenu_qss(self, fill: bool) -> str:
        """
        Gera QSS de submenus. Se fill=False, aplica min/max de largura do botão.
        Usamos min-width/max-width (funciona também para barras verticais).
        """
        limits = ""
        if not fill:
            limits = f"min-width: {SUB_TAB_MIN_W}px; max-width: {SUB_TAB_MAX_W}px;"
        return SECONDARY_QSS_BASE.replace("QTabBar::tab {", f"QTabBar::tab {{ {limits}")

    def _tune_submenu(self, tw: QTabWidget):
        tb = tw.tabBar()
        tb.setVisible(True)
        tb.setElideMode(Qt.ElideRight)
        tb.setUsesScrollButtons(True)       # setinhas quando não couber
        tb.setExpanding(SUB_TABS_FILL)      # fill ou não
        tw.setStyleSheet(self._submenu_qss(SUB_TABS_FILL))

        # espessura consistente: altura p/ horizontais, largura p/ verticais
        pos = tw.tabPosition()
        if pos in (QTabWidget.North, QTabWidget.South):
            tb.setMinimumHeight(SUB_TAB_THICK)
            tb.setMaximumHeight(SUB_TAB_THICK)
            tb.setMinimumWidth(0); tb.setMaximumWidth(16777215)
        else:
            tb.setMinimumHeight(0); tb.setMaximumHeight(16777215)
            tb.setMinimumWidth(SUB_TAB_THICK); tb.setMaximumWidth(SUB_TAB_THICK)

        # tooltips completos
        for i in range(tw.count()):
            tw.setTabToolTip(i, tw.tabText(i).replace("\n", " "))

    def _force_show_all_submenus(self):
        """Garantia extra: mantém toda QTabWidget interna com barra visível e espessura correta."""
        for tw in self.findChildren(QTabWidget):
            if tw is self.tabs:
                continue
            self._tune_submenu(tw)

    # ---------- Responsividade ----------
    def _toggle_compact(self):
        self._compact_forced = not self._compact_forced
        self._apply_responsive()

    def _apply_responsive(self):
        w = max(1, self.width())
        compact = self._compact_forced or (w < HAMBURGER_BP)

        if compact:
            self.header.setVisible(True)
            self.top_bar.setMinimumHeight(0)
            self.top_bar.setMaximumHeight(0)
            self.top_bar.setVisible(True)
            self.menu_tabs.setMinimumWidth(int(min(w * 0.85, 1200)))
        else:
            self.header.setVisible(False)
            self.top_bar.setMinimumHeight(TOP_TAB_H)
            self.top_bar.setMaximumHeight(TOP_TAB_H)
            if TOP_TABS_FILL:
                self.top_bar.setParams(fill=True, min_w=TOP_TAB_MIN_W, max_w=TOP_TAB_MAX_W,
                                       gap=TOP_TAB_GAP, scroll=False)
                self.top_bar.setExpanding(True)
            else:
                self.top_bar.setParams(fill=False, min_w=TOP_TAB_MIN_W, max_w=TOP_TAB_MAX_W,
                                       gap=TOP_TAB_GAP, scroll=True)
                self.top_bar.setExpanding(False)
            self.top_bar.updateGeometry()
            self.tabs.updateGeometry()
            self.tabs.setStyleSheet(PRIMARY_QSS)

        # aplicar em TODOS os submenus (sem substituir tabBar)
        for tw in self.findChildren(QTabWidget):
            if tw is self.tabs:
                continue
            self._tune_submenu(tw)

        # reforço final (evita "sumiço" após voltar do hambúrguer)
        self._force_show_all_submenus()

    def resizeEvent(self, e):
        super().resizeEvent(e)
        self._apply_responsive()
        self.top_bar.repaint()
        self.tabs.repaint()

if __name__ == "__main__":
    inicializar_banco()
    ensure_schema_impostos()
    ensure_schema_vendas()

    # Taskbar usa o seu ícone no Windows
    if sys.platform.startswith("win"):
        try:
            import ctypes
            ctypes.windll.shell32.SetCurrentProcessExplicitAppUserModelID("MedicalLaudos.Financeiro")
        except Exception:
            pass

    app = QApplication(sys.argv)
    if APP_ICON:
        app.setWindowIcon(QIcon(APP_ICON))  # ícone global do app

    win = FinanceiroMainWindow()
    scr = app.primaryScreen().size()
    win.setMinimumSize(820, 600)
    win.resize(min(1600, scr.width()-100), min(980, scr.height()-100))
    win.show()
    sys.exit(app.exec_())

