# nfse_db.py
from datetime import datetime
from lxml import etree
import sqlite3
import os

CAMINHO_BANCO = os.path.join("db", "sistema_financeiro.db")

DDL = r"""
CREATE TABLE IF NOT EXISTS nfse_lotes (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  protocolo TEXT UNIQUE,
  numero_lote TEXT, id_lote_xml TEXT,
  cnpj_prestador TEXT, im_prestador TEXT,
  competencia TEXT, convenio_id INTEGER, convenio_nome TEXT,
  qtd_rps INTEGER, data_envio TEXT,
  ambiente TEXT, url_ws TEXT,
  xml_lote_path TEXT, soap_req_path TEXT, soap_resp_path TEXT, outputxml_path TEXT,
  situacao INTEGER, data_ultima_consulta TEXT, mensagem_retorno TEXT
);
CREATE INDEX IF NOT EXISTS idx_nfse_lotes_protocolo ON nfse_lotes(protocolo);

CREATE TABLE IF NOT EXISTS nfse_rps (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  protocolo TEXT,
  numero_rps INTEGER, serie_rps TEXT, tipo_rps TEXT, id_infrps TEXT, data_emissao_rps TEXT,
  item_lista_servico TEXT, codigo_trib_municipio TEXT, discriminacao TEXT, codigo_municipio_prestacao TEXT,
  cnpj_tomador TEXT, im_tomador TEXT, razao_tomador TEXT,
  endereco TEXT, numero TEXT, complemento TEXT, bairro TEXT,
  codigo_municipio_ibge TEXT, uf TEXT, cep TEXT,
  valor_servicos TEXT, base_calculo TEXT, valor_liquido TEXT, iss_retido TEXT, aliquota TEXT, valor_iss TEXT,
  qtd_registros_consolidados INTEGER, valor_total_convenio TEXT, fator_base_utilizado TEXT
);
CREATE INDEX IF NOT EXISTS idx_nfse_rps_protocolo ON nfse_rps(protocolo);

CREATE TABLE IF NOT EXISTS nfse_notas (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  protocolo TEXT, numero_lote TEXT,
  numero_nfse TEXT, codigo_verificacao TEXT, data_emissao_nfse TEXT,
  competencia TEXT, cnpj_prestador TEXT, im_prestador TEXT,
  codigo_municipio TEXT, uf TEXT,
  valor_servicos TEXT, valor_iss TEXT, aliquota TEXT, iss_retido TEXT,
  xml_comp_nfse_path TEXT,
  UNIQUE (numero_nfse, cnpj_prestador)
);
CREATE INDEX IF NOT EXISTS idx_nfse_notas_protocolo ON nfse_notas(protocolo);

CREATE TABLE IF NOT EXISTS nfse_eventos (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  protocolo TEXT, numero_nfse TEXT,
  tipo TEXT, data_evento TEXT, situacao INTEGER, mensagem TEXT, payload_xml_path TEXT
);
"""

def init_nfse_schema(conn):
    conn.executescript(DDL)    
    conn.commit()

    # ---- Tabela de sequência RPS (já existia) ----
    conn.execute("""
    CREATE TABLE IF NOT EXISTS nfse_sequencia_rps (
        cnpj_prestador TEXT NOT NULL,
        serie          TEXT NOT NULL,
        proximo        INTEGER NOT NULL DEFAULT 1,
        PRIMARY KEY (cnpj_prestador, serie)
    )
    """)

    # ---- (NOVO) Garante a tabela de status, usada pelo GUI ----
    conn.execute("""
    CREATE TABLE IF NOT EXISTS notas_emitidas_status (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        convenio TEXT NOT NULL,
        competencia TEXT NOT NULL,
        status TEXT NOT NULL,
        data_emissao TEXT,
        caminho_xml TEXT,
        mensagem_erro TEXT,
        numero_nfse TEXT,
        UNIQUE (convenio, competencia)
    )
    """)
    conn.commit()

    # ---- (NOVO) Migração: adiciona colunas em nfse_notas, se faltarem ----
    def _has_col(table, col):
        cur = conn.execute(f"PRAGMA table_info({table})")
        return any(r[1] == col for r in cur.fetchall())

    if not _has_col("nfse_notas", "cancelada"):
        conn.execute("ALTER TABLE nfse_notas ADD COLUMN cancelada INTEGER DEFAULT 0")
    if not _has_col("nfse_notas", "data_cancelamento"):
        conn.execute("ALTER TABLE nfse_notas ADD COLUMN data_cancelamento TEXT")
    if not _has_col("nfse_notas", "cancelamento_pendente"):
        conn.execute("ALTER TABLE nfse_notas ADD COLUMN cancelamento_pendente INTEGER DEFAULT 0")
    conn.commit()

    # ---- (opcional) Índice para acelerar buscas por número da NF ----
    conn.execute("CREATE INDEX IF NOT EXISTS idx_nfse_notas_numero ON nfse_notas(numero_nfse)")
    conn.commit()



def upsert_lote(conn, d):
    conn.execute("""
      INSERT INTO nfse_lotes (
        protocolo, numero_lote, id_lote_xml, cnpj_prestador, im_prestador, competencia,
        convenio_id, convenio_nome, qtd_rps, data_envio, ambiente, url_ws,
        xml_lote_path, soap_req_path, soap_resp_path, outputxml_path,
        situacao, data_ultima_consulta, mensagem_retorno
      ) VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)
      ON CONFLICT(protocolo) DO UPDATE SET
        numero_lote=excluded.numero_lote, id_lote_xml=excluded.id_lote_xml,
        cnpj_prestador=excluded.cnpj_prestador, im_prestador=excluded.im_prestador,
        competencia=excluded.competencia, convenio_id=excluded.convenio_id, convenio_nome=excluded.convenio_nome,
        qtd_rps=excluded.qtd_rps, data_envio=excluded.data_envio, ambiente=excluded.ambiente, url_ws=excluded.url_ws,
        xml_lote_path=excluded.xml_lote_path, soap_req_path=excluded.soap_req_path,
        soap_resp_path=excluded.soap_resp_path, outputxml_path=excluded.outputxml_path,
        situacao=COALESCE(excluded.situacao, nfse_lotes.situacao),
        data_ultima_consulta=COALESCE(excluded.data_ultima_consulta, nfse_lotes.data_ultima_consulta),
        mensagem_retorno=COALESCE(excluded.mensagem_retorno, nfse_lotes.mensagem_retorno)
    """, (
      d.get("protocolo"), d.get("numero_lote"), d.get("id_lote_xml"),
      d.get("cnpj_prestador"), d.get("im_prestador"), d.get("competencia"),
      d.get("convenio_id"), d.get("convenio_nome"), d.get("qtd_rps"),
      d.get("data_envio"), d.get("ambiente"), d.get("url_ws"),
      d.get("xml_lote_path"), d.get("soap_req_path"), d.get("soap_resp_path"), d.get("outputxml_path"),
      d.get("situacao"), d.get("data_ultima_consulta"), d.get("mensagem_retorno"),
    ))
    conn.commit()

def insert_rps_snapshot(conn, protocolo, rps):
    conn.execute("""
      INSERT INTO nfse_rps (
        protocolo, numero_rps, serie_rps, tipo_rps, id_infrps, data_emissao_rps,
        item_lista_servico, codigo_trib_municipio, discriminacao, codigo_municipio_prestacao,
        cnpj_tomador, im_tomador, razao_tomador, endereco, numero, complemento, bairro,
        codigo_municipio_ibge, uf, cep,
        valor_servicos, base_calculo, valor_liquido, iss_retido, aliquota, valor_iss,
        qtd_registros_consolidados, valor_total_convenio, fator_base_utilizado
      ) VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)
    """, (
      protocolo, rps["numero_rps"], rps["serie_rps"], rps["tipo_rps"], rps["id_infrps"], rps["data_emissao_rps"],
      rps["item_lista_servico"], rps["codigo_trib_municipio"], rps["discriminacao"], rps["codigo_municipio_prestacao"],
      rps["cnpj_tomador"], rps["im_tomador"], rps["razao_tomador"], rps["endereco"], rps["numero"], rps["complemento"], rps["bairro"],
      rps["codigo_municipio_ibge"], rps["uf"], rps["cep"],
      rps["valor_servicos"], rps["base_calculo"], rps["valor_liquido"], rps["iss_retido"], rps["aliquota"], rps["valor_iss"],
      rps["qtd_registros_consolidados"], rps["valor_total_convenio"], rps["fator_base_utilizado"]
    ))
    conn.commit()

def log_evento(conn, protocolo, tipo, situacao=None, mensagem=None, payload_path=None, numero_nfse=None):
    conn.execute("""
      INSERT INTO nfse_eventos (protocolo, numero_nfse, tipo, data_evento, situacao, mensagem, payload_xml_path)
      VALUES (?, ?, ?, ?, ?, ?, ?)
    """, (protocolo, numero_nfse, tipo, datetime.now().strftime("%Y-%m-%d %H:%M:%S"),
          situacao, mensagem, payload_path))
    conn.commit()
    
def alocar_numero_rps(conn, cnpj_prestador: str, serie: str = "SN") -> int:
    """
    Retorna o próximo número sequencial do RPS para (cnpj_prestador, serie)
    e já incrementa o contador de forma atômica.
    - Se for a primeira vez, inicia pela maior numeração já emitida + 1 (se existir),
      senão começa em 1.
    """
    cur = conn.cursor()
    # trava curta para garantir atomicidade
    cur.execute("BEGIN IMMEDIATE")
    # cria a linha se não existir
    cur.execute("""
        INSERT OR IGNORE INTO nfse_sequencia_rps (cnpj_prestador, serie, proximo)
        VALUES (?, ?, 1)
    """, (cnpj_prestador, serie))

    # bootstrap: se nunca inicializou, tenta começar de MAX(numero_rps)+1
    cur.execute("""
        UPDATE nfse_sequencia_rps
           SET proximo = COALESCE((
               SELECT COALESCE(MAX(CAST(r.numero_rps AS INTEGER)), 0) + 1
                 FROM nfse_rps r
                 JOIN nfse_lotes l ON l.protocolo = r.protocolo
                WHERE l.cnpj_prestador = ? AND r.serie_rps = ?
           ), proximo)
         WHERE cnpj_prestador = ? AND serie = ? AND proximo = 1
    """, (cnpj_prestador, serie, cnpj_prestador, serie))

    # lê o atual
    cur.execute("""
        SELECT proximo FROM nfse_sequencia_rps
         WHERE cnpj_prestador = ? AND serie = ?
    """, (cnpj_prestador, serie))
    proximo = int(cur.fetchone()[0])

    # incrementa
    cur.execute("""
        UPDATE nfse_sequencia_rps
           SET proximo = proximo + 1
         WHERE cnpj_prestador = ? AND serie = ?
    """, (cnpj_prestador, serie))

    conn.commit()
    return proximo


def atualizar_situacao_lote(conn, protocolo, situacao, mensagem=None, outputxml_path=None):
    conn.execute("""
      UPDATE nfse_lotes
         SET situacao=?, data_ultima_consulta=?, mensagem_retorno=COALESCE(?, mensagem_retorno),
             outputxml_path=COALESCE(?, outputxml_path)
       WHERE protocolo=?
    """, (situacao, datetime.now().strftime("%Y-%m-%d %H:%M:%S"), mensagem, outputxml_path, protocolo))
    conn.commit()

def salvar_notas_da_consulta_lote(conn, protocolo, numero_lote, cnpj_prest, im_prest, competencia, base_files, xml_inner):
    ns = {"n": "http://www.abrasf.org.br/nfse.xsd"}
    root = etree.fromstring(xml_inner.encode("utf-8"))
    comp_list = root.findall(".//n:CompNfse", namespaces=ns)
    saved = 0
    for i, comp in enumerate(comp_list, start=1):
        n_inf = comp.find(".//n:InfNfse", namespaces=ns)
        if n_inf is None: continue
        numero_nfse = (n_inf.findtext("n:Numero", namespaces=ns) or "").strip()
        codigo_verif = (n_inf.findtext("n:CodigoVerificacao", namespaces=ns) or "").strip()
        data_em_nfse = (n_inf.findtext("n:DataEmissao", namespaces=ns) or "").strip()
        vs = n_inf.find(".//n:ValoresNfse", namespaces=ns)
        valor_servicos = (vs.findtext("n:ValorServicos", namespaces=ns) or "").strip() if vs is not None else ""
        valor_iss      = (vs.findtext("n:ValorIss", namespaces=ns) or "").strip() if vs is not None else ""
        aliquota       = (n_inf.findtext(".//n:Aliquota", namespaces=ns) or "").strip()
        iss_retido     = (n_inf.findtext(".//n:IssRetido", namespaces=ns) or "").strip()
        cod_mun = (n_inf.findtext(".//n:CodigoMunicipio", namespaces=ns) or "").strip()
        uf      = (n_inf.findtext(".//n:Uf", namespaces=ns) or "").strip()

        comp_xml_path = f"{base_files}_nfse_{i:02d}.xml"
        with open(comp_xml_path, "wb") as f:
            f.write(etree.tostring(comp, encoding="utf-8", xml_declaration=True))

        conn.execute("""
          INSERT OR IGNORE INTO nfse_notas (
            protocolo, numero_lote, numero_nfse, codigo_verificacao, data_emissao_nfse,
            competencia, cnpj_prestador, im_prestador, codigo_municipio, uf,
            valor_servicos, valor_iss, aliquota, iss_retido, xml_comp_nfse_path
          ) VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)
        """, (protocolo, numero_lote, numero_nfse, codigo_verif, data_em_nfse,
              competencia, cnpj_prest, im_prest, cod_mun, uf,
              valor_servicos, valor_iss, aliquota, iss_retido, comp_xml_path))
        conn.commit()
        log_evento(conn, protocolo, "consulta_lote",
                   mensagem=f"NFS-e {numero_nfse} registrada",
                   payload_path=comp_xml_path, numero_nfse=numero_nfse)
        saved += 1
    return saved
  
from datetime import datetime
import sqlite3
import os

CAMINHO_BANCO = os.path.join("db", "sistema_financeiro.db")

def upsert_status_nf(convenio, *, competencia_servico, status,
                     numero_nfse=None, competencia_nfse=None,
                     caminho_xml=None, mensagem_erro=None):
    """
    Atualiza/insere uma linha em notas_emitidas_status para (convenio, competência do serviço).
    Funciona com bases antigas (coluna 'competencia') e novas ('competencia_servico' e 'competencia_nfse').
    """
    ts = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    num = (str(numero_nfse).strip() if numero_nfse else None)
    msg = (mensagem_erro or None)

    with sqlite3.connect(CAMINHO_BANCO) as conn:
        cur = conn.cursor()
        cur.execute("PRAGMA table_info(notas_emitidas_status)")
        cols = {r[1] for r in cur.fetchall()}
        tem_comp_serv = "competencia_servico" in cols
        tem_comp_lega = "competencia" in cols
        tem_comp_nfse = "competencia_nfse" in cols

        comp_col = "competencia_servico" if tem_comp_serv else "competencia"

        # UPDATE
        set_part = ["status = ?", "data_emissao = ?", "numero_nfse = ?"]
        params = [status, ts, num]
        if caminho_xml is not None:
            set_part.append("caminho_xml = ?"); params.append(caminho_xml)
        if msg is not None:
            set_part.append("mensagem_erro = ?"); params.append(msg)
        if tem_comp_nfse:
            set_part.append("competencia_nfse = COALESCE(?, competencia_nfse)")
            params.append(competencia_nfse)

        sql_upd = f"UPDATE notas_emitidas_status SET {', '.join(set_part)} WHERE convenio = ? AND {comp_col} = ?"
        cur.execute(sql_upd, params + [convenio, competencia_servico])

        if cur.rowcount == 0:
            # INSERT
            cols_ins = ["convenio", comp_col, "status", "data_emissao", "numero_nfse"]
            vals_ins = [convenio, competencia_servico, status, ts, num]
            if tem_comp_lega and comp_col != "competencia":
                cols_ins.append("competencia"); vals_ins.append(competencia_servico)
            if caminho_xml is not None:
                cols_ins.append("caminho_xml"); vals_ins.append(caminho_xml)
            if msg is not None:
                cols_ins.append("mensagem_erro"); vals_ins.append(msg)
            if tem_comp_nfse:
                cols_ins.append("competencia_nfse"); vals_ins.append(competencia_nfse)
            ph = ",".join("?" for _ in cols_ins)
            cur.execute(f"INSERT INTO notas_emitidas_status ({', '.join(cols_ins)}) VALUES ({ph})", vals_ins)

        conn.commit()


