# -*- coding: utf-8 -*-
import os
import re
import html
import uuid
import sqlite3
import logging
from datetime import datetime
from decimal import Decimal, ROUND_HALF_UP

import requests
from requests_pkcs12 import Pkcs12Adapter
from lxml import etree
from signxml import XMLSigner, methods
from cryptography.hazmat.backends import default_backend
from cryptography.hazmat.primitives.serialization import pkcs12, Encoding
from OpenSSL.crypto import load_certificate, FILETYPE_PEM

# ======================= CONFIG =======================
CAMINHO_BANCO = os.path.join("db", "sistema_financeiro.db")

HOMOLOG_URL = "https://bhisshomologaws.pbh.gov.br/bhiss-ws/nfse"
PROD_URL    = "https://bhissws.pbh.gov.br/bhiss-ws/nfse"

NS_DADOS = "http://www.abrasf.org.br/nfse.xsd"
NS_DS    = "http://www.w3.org/2000/09/xmldsig#"
etree.register_namespace("ds", NS_DS)

BASE_DIR = os.path.dirname(os.path.abspath(__file__))
XSD_PATH = os.path.join(BASE_DIR, "schemas", "pbh", "nfse.xsd")  # opcional

FATOR_BASE = Decimal("1.00")   # mesmo da emissão consolidada
ALIQUOTA   = None              # ex.: Decimal("0.02")

logging.basicConfig(level=logging.INFO, format="%(asctime)s - %(levelname)s - %(message)s")
log_subst = logging.getLogger("substituicao_nfse")

# ======================= DICIONÁRIOS ===================
CANCEL_MOTIVOS = {
    "1": "Erro na emissão",
    "2": "Serviço não prestado",
    "3": "Duplicidade de nota",
    "4": "Erro de preenchimento",
    "5": "Outros",
}

# ======================= HELPERS =======================
_SIG_LOCALS = {
    "Signature", "SignedInfo", "CanonicalizationMethod", "SignatureMethod",
    "Reference", "Transforms", "Transform", "DigestMethod", "DigestValue",
    "SignatureValue", "KeyInfo", "X509Data", "X509Certificate"
}
_DEF_ENV  = "http://www.w3.org/2000/09/xmldsig#enveloped-signature"
_DEF_C14N = "http://www.w3.org/2001/10/xml-exc-c14n#"

def only_digits(s): 
    return re.sub(r"\D", "", str(s or ""))

def montar_cabecalho_nfse(versao="1.00") -> str:
    cab = etree.Element(f"{{{NS_DADOS}}}cabecalho", versao=versao, nsmap={None: NS_DADOS})
    etree.SubElement(cab, f"{{{NS_DADOS}}}versaoDados").text = versao
    return etree.tostring(cab, encoding="utf-8", method="xml").decode("utf-8")

def forcar_namespace_xmlsig(root: etree._Element):
    for el in root.iter():
        try:
            qn = etree.QName(el)
            local, uri = qn.localname, qn.namespace
        except Exception:
            continue
        if local in _SIG_LOCALS and uri != NS_DS:
            el.tag = f"{{{NS_DS}}}{local}"
    return root

def _fix_signature(sig_el: etree._Element) -> None:
    si = sig_el.find(f".//{{{NS_DS}}}SignedInfo")
    if si is None:
        return
    cm = si.find(f".//{{{NS_DS}}}CanonicalizationMethod")
    if cm is not None:
        cm.set("Algorithm", _DEF_C14N)
    ref = si.find(f".//{{{NS_DS}}}Reference")
    if ref is None:
        return
    tr_parent = ref.find(f".//{{{NS_DS}}}Transforms")
    if tr_parent is None:
        return
    for t in list(tr_parent):
        tr_parent.remove(t)
    etree.SubElement(tr_parent, f"{{{NS_DS}}}Transform").set("Algorithm", _DEF_ENV)
    etree.SubElement(tr_parent, f"{{{NS_DS}}}Transform").set("Algorithm", _DEF_C14N)

def extrair_mensagens_pbh(inner_xml: str):
    try:
        root = etree.fromstring(inner_xml.encode("utf-8"))
    except Exception:
        return []
    ns = {"n": NS_DADOS}
    msgs = []
    for m in root.findall(".//n:ListaMensagemRetorno/n:MensagemRetorno", namespaces=ns):
        cod = (m.findtext("n:Codigo", default="", namespaces=ns) or "").strip()
        msg = re.sub(r"\s+", " ", (m.findtext("n:Mensagem", default="", namespaces=ns) or "")).strip()
        cor = re.sub(r"\s+", " ", (m.findtext("n:Correcao", default="", namespaces=ns) or "")).strip()
        msgs.append({"codigo": cod, "mensagem": msg, "correcao": cor})
    return msgs

def raise_if_pbh_error(inner_xml: str):
    msgs = extrair_mensagens_pbh(inner_xml)
    if msgs:
        m = msgs[0]
        raise Exception(
            f"PBH [{m['codigo']}]: {m['mensagem']}"
            + (f" | Correção: {m['correcao']}" if m['correcao'] else "")
        )

def validar_estrutura_xml(xml_path: str, xsd_path: str, log_path: str) -> bool:
    try:
        schema = etree.XMLSchema(etree.parse(xsd_path))
        schema.assertValid(etree.parse(xml_path))
        with open(log_path, "w", encoding="utf-8") as f:
            f.write("✅ Validação OK: XML é válido conforme o XSD\n")
        return True
    except Exception as e:
        with open(log_path, "w", encoding="utf-8") as f:
            f.write(f"❌ XML inválido conforme XSD.\n• {str(e)}\n")
        return False

def _registrar_status(convenio_nome, *, competencia_servico, status, numero_nfse_antiga=None, numero_nfse_nova=None, mensagem="", caminho_xml=""):
    ts = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    try:
        with sqlite3.connect(CAMINHO_BANCO) as conn:
            cur = conn.cursor()
            cur.execute("PRAGMA table_info(notas_emitidas_status)")
            cols = {r[1] for r in cur.fetchall()}
            # upsert simples por (convenio, competencia_servico)
            if "competencia_servico" in cols:
                cur.execute("""
                    UPDATE notas_emitidas_status
                       SET status = ?, data_emissao = ?, mensagem_erro = ?, caminho_xml = ?,
                           numero_nfse = COALESCE(?, numero_nfse)
                     WHERE convenio = ? AND competencia_servico = ?
                """, (status, ts, mensagem[:2000], caminho_xml or None,
                      str(numero_nfse_nova or numero_nfse_antiga or "") or None,
                      convenio_nome, competencia_servico))
                if cur.rowcount == 0:
                    cur.execute("""
                        INSERT INTO notas_emitidas_status
                            (convenio, competencia_servico, status, data_emissao, mensagem_erro, caminho_xml, numero_nfse)
                        VALUES (?, ?, ?, ?, ?, ?, ?)
                    """, (convenio_nome, competencia_servico, status, ts, mensagem[:2000], caminho_xml or None,
                          str(numero_nfse_nova or numero_nfse_antiga or "") or None))
            else:
                # compat antigo
                cur.execute("""
                    UPDATE notas_emitidas_status
                       SET status = ?, data_emissao = ?, mensagem_erro = ?, caminho_xml = ?,
                           numero_nfse = COALESCE(?, numero_nfse)
                     WHERE convenio = ? AND competencia = ?
                """, (status, ts, mensagem[:2000], caminho_xml or None,
                      str(numero_nfse_nova or numero_nfse_antiga or "") or None,
                      convenio_nome, competencia_servico))
                if cur.rowcount == 0:
                    cur.execute("""
                        INSERT INTO notas_emitidas_status
                            (convenio, competencia, status, data_emissao, mensagem_erro, caminho_xml, numero_nfse)
                        VALUES (?, ?, ?, ?, ?, ?, ?)
                    """, (convenio_nome, competencia_servico, status, ts, mensagem[:2000], caminho_xml or None,
                          str(numero_nfse_nova or numero_nfse_antiga or "") or None))
            conn.commit()
    except Exception as e:
        log_subst.warning(f"Falha ao registrar status: {e}")

# ================== XML: SUBSTITUIÇÃO =================
def _montar_rps_novo(parent, *, base_dec: Decimal, discr: str,
                     cnpj_prest: str, im_prest: str, cod_mun_prest: str):
    """
    Monta um RPS ABRASF 1.00 'simples' (consolidado) dentro de parent.
    Retorna o elemento InfRps (para assinatura) e o seu Id.
    """
    NS = NS_DADOS
    rps = etree.SubElement(parent, f"{{{NS}}}Rps")
    inf_id = f"R{uuid.uuid4().hex[:8]}"
    inf = etree.SubElement(rps, f"{{{NS}}}InfRps", Id=inf_id)

    ident = etree.SubElement(inf, f"{{{NS}}}IdentificacaoRps")
    # Para PBH, a série livre "SN" e numerador sequencial devem vir do seu banco.
    # Aqui, como é substituição atômica (não lote), usamos "SN" e timestamp como fallback.
    numero_rps = int(datetime.now().strftime("%H%M%S%f")[:9])
    etree.SubElement(ident, f"{{{NS}}}Numero").text = str(numero_rps)
    etree.SubElement(ident, f"{{{NS}}}Serie").text  = "SN"
    etree.SubElement(ident, f"{{{NS}}}Tipo").text   = "1"

    etree.SubElement(inf, f"{{{NS}}}DataEmissao").text = datetime.now().strftime("%Y-%m-%dT%H:%M:%S")
    etree.SubElement(inf, f"{{{NS}}}NaturezaOperacao").text = "1"
    etree.SubElement(inf, f"{{{NS}}}RegimeEspecialTributacao").text = "6"
    etree.SubElement(inf, f"{{{NS}}}OptanteSimplesNacional").text = "1"
    etree.SubElement(inf, f"{{{NS}}}IncentivadorCultural").text = "2"
    etree.SubElement(inf, f"{{{NS}}}Status").text = "1"

    servico = etree.SubElement(inf, f"{{{NS}}}Servico")
    valores = etree.SubElement(servico, f"{{{NS}}}Valores")
    etree.SubElement(valores, f"{{{NS}}}ValorServicos").text    = format(base_dec, ".2f")
    etree.SubElement(valores, f"{{{NS}}}IssRetido").text        = "2"
    etree.SubElement(valores, f"{{{NS}}}BaseCalculo").text      = format(base_dec, ".2f")
    etree.SubElement(valores, f"{{{NS}}}ValorLiquidoNfse").text = format(base_dec, ".2f")

    if ALIQUOTA is not None:
        valor_iss = (base_dec * ALIQUOTA).quantize(Decimal("0.01"), ROUND_HALF_UP)
        etree.SubElement(valores, f"{{{NS}}}Aliquota").text = format(ALIQUOTA, ".4f")
        etree.SubElement(valores, f"{{{NS}}}ValorIss").text = format(valor_iss, ".2f")

    etree.SubElement(servico, f"{{{NS}}}ItemListaServico").text = "17.02"
    etree.SubElement(servico, f"{{{NS}}}CodigoTributacaoMunicipio").text = "170200688"
    etree.SubElement(servico, f"{{{NS}}}Discriminacao").text = discr
    etree.SubElement(servico, f"{{{NS}}}CodigoMunicipio").text = only_digits(cod_mun_prest) or "3106200"

    prest = etree.SubElement(inf, f"{{{NS}}}Prestador")
    etree.SubElement(prest, f"{{{NS}}}Cnpj").text = only_digits(cnpj_prest)
    if im_prest:
        etree.SubElement(prest, f"{{{NS}}}InscricaoMunicipal").text = only_digits(im_prest)

    return inf, inf_id, rps

def gerar_xml_substituicao(*, numero_nfse_antiga: str, codigo_cancelamento: str,
                           cnpj_prest: str, im_prest: str, cod_mun_prest: str,
                           base_dec: Decimal, discriminacao: str,
                           caminho_saida: str) -> str:
    """
    Estrutura (ABRASF 1.00 – PBH):
    <SubstituirNfseEnvio>
      <SubstituicaoNfse>
        <Pedido>
          <InfPedidoCancelamento Id="...">...</InfPedidoCancelamento>
          <!-- ds:Signature (irmã) -->
        </Pedido>
        <Rps> ... <InfRps Id="..."> ... </InfRps> <!-- ds:Signature (irmã de InfRps) -->
      </SubstituicaoNfse>
    </SubstituirNfseEnvio>
    """
    NS = NS_DADOS
    os.makedirs(os.path.dirname(caminho_saida) or ".", exist_ok=True)

    root = etree.Element(f"{{{NS}}}SubstituirNfseEnvio", nsmap={None: NS})
    sub  = etree.SubElement(root, f"{{{NS}}}SubstituicaoNfse")
    pedido = etree.SubElement(sub, f"{{{NS}}}Pedido")

    infc_id = f"C{uuid.uuid4().hex[:8]}"
    infc = etree.SubElement(pedido, f"{{{NS}}}InfPedidoCancelamento", Id=infc_id)
    ident = etree.SubElement(infc, f"{{{NS}}}IdentificacaoNfse")
    etree.SubElement(ident, f"{{{NS}}}Numero").text = str(int(numero_nfse_antiga))
    etree.SubElement(ident, f"{{{NS}}}Cnpj").text = only_digits(cnpj_prest)
    etree.SubElement(ident, f"{{{NS}}}InscricaoMunicipal").text = only_digits(im_prest)
    etree.SubElement(ident, f"{{{NS}}}CodigoMunicipio").text = only_digits(cod_mun_prest) or "3106200"
    etree.SubElement(infc, f"{{{NS}}}CodigoCancelamento").text = only_digits(codigo_cancelamento)

    # RPS novo (consolidado)
    inf_rps, inf_rps_id, rps = _montar_rps_novo(
        sub, base_dec=base_dec, discr=discriminacao,
        cnpj_prest=cnpj_prest, im_prest=im_prest, cod_mun_prest=cod_mun_prest
    )

    etree.ElementTree(root).write(caminho_saida, encoding="utf-8", xml_declaration=True, pretty_print=False)
    return caminho_saida, infc_id, inf_rps_id

# ================== ASSINATURA ========================
def _carregar_chave(cert_path: str, senha: str):
    with open(cert_path, "rb") as f:
        pfx = f.read()
    private_key, certificate, _ = pkcs12.load_key_and_certificates(
        pfx, senha.encode(), backend=default_backend()
    )
    if not private_key or not certificate:
        raise Exception("Falha ao carregar chave/cert do PFX.")
    pem = certificate.public_bytes(Encoding.PEM)
    openssl_cert = load_certificate(FILETYPE_PEM, pem)
    return private_key, openssl_cert

def assinar_substituicao(xml_path: str, cert_path: str, senha_cert: str,
                         infc_id: str, infrps_id: str) -> str:
    with open(xml_path, "rb") as f:
        root = etree.parse(f).getroot()

    ns = {"n": NS_DADOS, "ds": NS_DS}
    pedido = root.find(".//n:SubstituicaoNfse/n:Pedido", namespaces=ns)
    infc   = root.find(".//n:SubstituicaoNfse/n:Pedido/n:InfPedidoCancelamento", namespaces=ns)
    inf_rps = root.find(".//n:SubstituicaoNfse/n:Rps/n:InfRps", namespaces=ns)
    if pedido is None or infc is None or inf_rps is None:
        raise Exception("Estrutura inválida de SubstituirNfseEnvio.")

    pk, cert = _carregar_chave(cert_path, senha_cert)

    signer = XMLSigner(
        method=methods.enveloped,
        signature_algorithm="rsa-sha1",
        digest_algorithm="sha1",
        c14n_algorithm=_DEF_C14N,
    )

    # --- Assina InfPedidoCancelamento (irma DS em Pedido)
    infc_copy = etree.fromstring(etree.tostring(infc))
    signed_c = signer.sign(infc_copy, key=pk, cert=[cert], reference_uri=f"#{infc_id}")
    sig_c = signed_c if (etree.QName(signed_c).localname == "Signature" and etree.QName(signed_c).namespace == NS_DS) \
        else signed_c.find(f".//{{{NS_DS}}}Signature")
    if sig_c is None:
        raise Exception("Falha na assinatura do InfPedidoCancelamento.")
    pedido.insert(pedido.index(infc) + 1, sig_c)

    # --- Assina InfRps (irma DS em Rps)
    rps = inf_rps.getparent()
    infrps_copy = etree.fromstring(etree.tostring(inf_rps))
    signed_r = signer.sign(infrps_copy, key=pk, cert=[cert], reference_uri=f"#{infrps_id}")
    sig_r = signed_r if (etree.QName(signed_r).localname == "Signature" and etree.QName(signed_r).namespace == NS_DS) \
        else signed_r.find(f".//{{{NS_DS}}}Signature")
    if sig_r is None:
        raise Exception("Falha na assinatura do InfRps.")
    rps.insert(rps.index(inf_rps) + 1, sig_r)

    # Normaliza DS + ordena transforms
    forcar_namespace_xmlsig(root)
    for s in root.xpath("//ds:Signature", namespaces=ns):
        _fix_signature(s)

    etree.ElementTree(root).write(xml_path, encoding="utf-8", xml_declaration=True, pretty_print=False)
    return xml_path

# ================== ENVIO SOAP ========================
def enviar_substituicao(xml_assinado_path: str, cert_path: str, senha_cert: str, url: str = HOMOLOG_URL) -> str:
    with open(xml_assinado_path, "r", encoding="utf-8") as f:
        raw = f.read()
    dados_xml = re.sub(r"<\?xml.*?\?>", "", raw).lstrip()
    cabecalho_xml = montar_cabecalho_nfse("1.00")

    envelope = f"""<?xml version="1.0" encoding="UTF-8"?>
<soapenv:Envelope xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/"
                  xmlns:ws="http://ws.bhiss.pbh.gov.br">
  <soapenv:Header/>
  <soapenv:Body>
    <ws:SubstituirNfseRequest>
      <nfseCabecMsg><![CDATA[{cabecalho_xml}]]></nfseCabecMsg>
      <nfseDadosMsg><![CDATA[{dados_xml}]]></nfseDadosMsg>
    </ws:SubstituirNfseRequest>
  </soapenv:Body>
</soapenv:Envelope>"""

    base = os.path.splitext(xml_assinado_path)[0]
    req_path  = base + "_soap_request.xml"
    resp_path = base + "_soap_response.xml"
    out_path  = base + "_outputXML.xml"

    with open(req_path, "w", encoding="utf-8") as f:
        f.write(envelope)

    s = requests.Session()
    s.mount("https://", Pkcs12Adapter(pkcs12_filename=cert_path, pkcs12_password=senha_cert))
    r = s.post(
        url,
        data=envelope.encode("utf-8"),
        headers={
            "Content-Type": "text/xml; charset=UTF-8",
            "SOAPAction": "http://ws.bhiss.pbh.gov.br/SubstituirNfse",
        },
        verify=True,
        timeout=(15, 90),
    )

    with open(resp_path, "wb") as f:
        f.write(r.content)
    if r.status_code != 200:
        raise Exception(f"Erro HTTP {r.status_code}: {r.text[:800]}")

    soap = etree.fromstring(r.content)
    fault = soap.find(".//{http://schemas.xmlsoap.org/soap/envelope/}Fault")
    if fault is not None:
        fs = (fault.findtext("faultstring") or "").strip()
        fd = (fault.findtext("detail") or "").strip()
        raise Exception(f"SOAP Fault: {fs or 'Falha desconhecida'} | {fd}")

    out = soap.find(".//{*}outputXML")
    if out is None or not (out.text or "").strip():
        raise Exception("Sem <outputXML> na resposta do SubstituirNfse.")
    inner = html.unescape(out.text or "")

    with open(out_path, "w", encoding="utf-8") as f:
        f.write(inner)

    raise_if_pbh_error(inner)
    return inner

def _confirmacao_substituicao_ok(inner: str) -> tuple[bool, str | None]:
    """
    Retorna (ok, numero_nfse_nova or None)
    A PBH costuma retornar a NFS-e emitida no bloco interno.
    Como os tags variam entre municípios, tentamos algumas possibilidades.
    """
    if "<RetSubstituicao" in inner and "<Confirmacao" in inner and "<ListaMensagemRetorno" not in inner:
        # tenta achar o número da NFS-e nova
        m = re.search(r"<NumeroNfse>(\d+)</NumeroNfse>", inner)
        if not m:
            m = re.search(r"<Numero>(\d+)</Numero>", inner)
        return True, (m.group(1) if m else None)
    return False, None

# ================== ORQUESTRAÇÃO ======================
def substituir_nfse_por_convenio(
    *,
    numero_nfse_antiga: str,
    id_convenio: int,
    codigo_cancelamento: str = "2",
    competencia_servico: str | None = None,
    url_envio: str = HOMOLOG_URL
) -> tuple[bool, str]:
    """
    Faz a SUBSTITUIÇÃO da NFS-e:
      - Cancela a antiga usando o código 1..5
      - Emite dentro da mesma chamada um novo RPS consolidado
      - Atualiza 'notas_emitidas_status'
    """
    if not str(numero_nfse_antiga).strip().isdigit():
        return False, "Número de NFS-e inválido."
    if codigo_cancelamento not in CANCEL_MOTIVOS:
        return False, "Código de cancelamento inválido (1..5)."

    try:
        # 1) Prestador + convênio + competência
        with sqlite3.connect(CAMINHO_BANCO) as conn:
            cur = conn.cursor()

            cur.execute("""
                SELECT certificado_path, senha_certificado, cnpj, inscricao_municipal, codigo_municipio_ibge
                FROM medical_laudos
                LIMIT 1
            """)
            row = cur.fetchone()
            if not row:
                return False, "Configuração do prestador não encontrada."
            cert_path, senha_cert, cnpj_prest, im_prest, cod_mun_prest = row
            cod_mun_prest = only_digits(cod_mun_prest or "3106200")

            cur.execute("SELECT nome FROM convenios WHERE id = ?", (id_convenio,))
            r = cur.fetchone()
            if not r:
                return False, "Convênio não encontrado."
            convenio_nome = r[0]

            # competência de serviço (fallback p/ a atual guarda)
            if not competencia_servico:
                q = cur.execute("""
                    SELECT competencia
                      FROM nfse_lotes
                     WHERE convenio_id = ?
                  ORDER BY id DESC LIMIT 1
                """, (id_convenio,))
                last = q.fetchone()
                competencia_servico = (last[0] if last else None) or datetime.now().strftime("%m/%Y")

            # valor consolidado: soma da competência (mesmo critério da emissão)
            q = cur.execute("""
                SELECT COALESCE(SUM(CAST(REPLACE(REPLACE([Valor Convenio],'.',''),',','.') AS REAL)),0)
                FROM registros_financeiros
                WHERE Convenio = ? AND [Competência] = ?
            """, (convenio_nome, competencia_servico))
            total_convenio = float(q.fetchone()[0] or 0.0)

        base_dec = (Decimal(str(total_convenio)) * FATOR_BASE).quantize(Decimal("0.01"), ROUND_HALF_UP)
        discr = f"Substituição da NFS-e nº {int(numero_nfse_antiga)} – {CANCEL_MOTIVOS[codigo_cancelamento]} – competência {competencia_servico}"

        # 2) Monta XML
        base_dir = os.path.join("notas_emitidas", "substituicoes")
        os.makedirs(base_dir, exist_ok=True)
        xml_path = os.path.join(base_dir, f"substituicao_nfse_{int(numero_nfse_antiga)}.xml")

        xml_path, infc_id, infrps_id = gerar_xml_substituicao(
            numero_nfse_antiga=str(int(numero_nfse_antiga)),
            codigo_cancelamento=codigo_cancelamento,
            cnpj_prest=cnpj_prest,
            im_prest=im_prest,
            cod_mun_prest=cod_mun_prest,
            base_dec=base_dec,
            discriminacao=discr,
            caminho_saida=xml_path
        )

        if os.path.exists(XSD_PATH):
            validar_estrutura_xml(xml_path, XSD_PATH, xml_path.replace(".xml", "_pre_validacao.log"))

        # 3) Assina (InfPedidoCancelamento e InfRps)
        assinar_substituicao(xml_path, cert_path, senha_cert, infc_id, infrps_id)

        # 4) Pré-status
        _registrar_status(
            convenio_nome,
            competencia_servico=competencia_servico,
            status="Substituição solicitada",
            numero_nfse_antiga=numero_nfse_antiga,
            caminho_xml=xml_path
        )

        # 5) Envia e interpreta
        inner = enviar_substituicao(xml_path, cert_path, senha_cert, url=url_envio)
        ok, nf_nova = _confirmacao_substituicao_ok(inner)

        if ok:
            _registrar_status(
                convenio_nome,
                competencia_servico=competencia_servico,
                status="Substituída",
                numero_nfse_antiga=numero_nfse_antiga,
                numero_nfse_nova=nf_nova,
                mensagem=inner,
                caminho_xml=xml_path
            )
            return True, f"NFS-e {int(numero_nfse_antiga)} substituída com sucesso." + (f" Nova: {nf_nova}." if nf_nova else "")
        else:
            _registrar_status(
                convenio_nome,
                competencia_servico=competencia_servico,
                status="Substituição em análise",
                numero_nfse_antiga=numero_nfse_antiga,
                mensagem=inner,
                caminho_xml=xml_path
            )
            return True, "Substituição enviada. PBH retornou confirmação parcial (em análise)."

    except Exception as e:
        log_subst.exception("Falha na substituição")
        return False, f"Falha ao substituir NFS-e: {e}"

# ================== CLI ===============================
if __name__ == "__main__":
    ok, msg = substituir_nfse_por_convenio(
        numero_nfse_antiga="123",
        id_convenio=4,
        codigo_cancelamento="2",
        competencia_servico="06/2025",
        url_envio=HOMOLOG_URL
    )
    print(ok, msg)
